;-------------------------------------------------------------------------------
; File Name: snrProj.asm                                                       |
; File Description: This program will receive a three bit opcode and perform   |
;                   the instruction.  Opcodes are:                             |
;                   	000 Keyboard Out           Output                      |
;                       001 Serial In              Output                      |
;                       010 Serial Out             Input                       |
;                       011 Video Out              Input                       |
;                       100 Latch LED array        Input                       |
;                       111 Hello acknowledgement  Output                      |
;                   Data bus is an input by default.  Output to the data bus   |
;                   allowed when I/O address is 1.  When opcode action has been|
;                   completed, ACK should be set to 1. CPU will clear I/O      |
;                   addresses and then data bus may set to input again. ACK can|
;                   be cleared then.                                           |
;                   Opcode 111, Hello Acknoledgement should return A5 hex.     |
;                   Data received from the CPU will be latched on the LED array|
;-------------------------------------------------------------------------------

	list	P=PIC16F877	; PIC16C77 Microprocessor
	include "C:\arbadell\chipset\P16F877.INC"; include information about this chip

	org	0x00		; Start the program at adress 00 HEX
	goto	start		;

	org	0x04		; Interrupt service routine
;--------------------------- ISR -----------------------------------------------
; This subroutine gets called when a keyboard interrup or when a serial
; interrup occur.  The subroutine reads checks to see which of two interrupts
; has occured and calles the respective function.
;-------------------------------------------------------------------------------
isr	movwf	w_temp		; save the contents of w register
	movf	STATUS,w	; save the STATUS Register
	movwf	STATUS_temp	;
	clrf	STATUS		; Clear STATUS byte for ISR

	btfsc	INTCON,0	; check if interrupt is from keyboard in
	call	keyint
	
	btfsc	PIR1,5		; if not, check for serial interrupt
	call	srlint		

	movf	STATUS_temp,w	; replace value of STATUS register
	movwf	STATUS		;
	movf	w_temp,w	; replace value of w register

	retfie			; if neither, return


;--------------------------- keyint --------------------------------------------
; This subroutine gets called when a keyboard interrup occurs.  The
; subroutine reads keybard input and stores result in buffer, advancing the
; keyboard buffer pointers accordingly.
;-------------------------------------------------------------------------------
keyint	btfss	INTCON,0	; check if interrupt is from keyboard in
	return			; if not, return

	movwf	w_temp		; save the contents of w register
	movf	STATUS,w	; save the STATUS Register
	movwf	STATUS_temp	;
	clrf	STATUS		; Clear STATUS byte for ISR

keycont
	bsf	STATUS,5
	btfsc	PCON,1
	goto	nopor	

	bcf	STATUS,5
	btfsc	PORTB,7		; If clock is set, return
	goto	doneKb		; 
	goto	getbit

nopor
	bcf	STATUS,5	; If clock is clear, return
	btfss	PORTB,7
	goto	doneKb
	btfss	onetime,0
	goto	getbit2
	bcf	onetime,0
	goto	doneKb

getbit2
	bsf	PORTA,1	
getbit	
	btfsc	PORTE,0		; if data bit is clear, skip
	goto	bit_set		; data bit is set, so goto set it on temp
	btfsc	byteStat,0	; check for start bit
	goto	bit_clr		; start bit found
	bsf	byteStat,0	; mark start bit as found
	goto	doneKb

bit_set	bsf	tempBit,0	; data bit is set, so set it on temp
	goto	get_0		; goto setbyte 
bit_clr bcf	tempBit,0	; data bit is clear, so clear it on temp

get_0	btfsc	bitNum,0	; check for bit 0
	goto	get_1		; since bit 0 found, check bit 1
	bsf	bitNum,0	; mark bit 0 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD0 = 0
	goto	set_0		; bit = 1, so goto set bit 0
	bcf	tempD,0		; bit = 0, so set tempD0 = 0
	goto	doneKb		; done getting bit	
set_0	bsf	tempD,0		; bit 0 = 1
	goto	doneKb		; done getting bit

get_1	btfsc	bitNum,1	; check for bit 1
	goto	get_2		; since bit 1 found, check bit 2
	bsf	bitNum,1	; mark bit 1 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD1 = 0
	goto	set_1		; bit = 1, so goto set bit 1
	bcf	tempD,1		; bit = 0, so set tempD0 = 0
	goto	doneKb		; done getting bit	
set_1	bsf	tempD,1		; bit 1 = 1
	goto	doneKb		; done getting bit

get_2	btfsc	bitNum,2	; check for bit 2
	goto	get_3		; since bit 2 found, check bit 3
	bsf	bitNum,2	; mark bit 2 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD2 = 0
	goto	set_2		; bit = 1, so goto set bit 2
	bcf	tempD,2		; bit = 0, so set tempD2 = 0
	goto	doneKb		; done getting bit	
set_2	bsf	tempD,2		; bit 2 = 1
	goto	doneKb		; done getting bit

get_3	btfsc	bitNum,3	; check for bit 3
	goto	get_4		; since bit 3 found, check bit 4
	bsf	bitNum,3	; mark bit 3 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD3 = 0
	goto	set_3		; bit = 1, so goto set bit 3
	bcf	tempD,3		; bit = 0, so set tempD3 = 0
	goto	doneKb		; done getting bit	
set_3	bsf	tempD,3		; bit 3 = 1
	goto	doneKb		; done getting bit

get_4	btfsc	bitNum,4	; check for bit 4
	goto	get_5		; since bit 4 found, check bit 5
	bsf	bitNum,4	; mark bit 4 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD4 = 0
	goto	set_4		; bit = 1, so goto set bit 0
	bcf	tempD,4		; bit = 0, so set tempD4 = 0
	goto	doneKb		; done getting bit	
set_4	bsf	tempD,4		; bit 4 = 1
	goto	doneKb		; done getting bit

get_5	btfsc	bitNum,5	; check for bit 5
	goto	get_6		; since bit 5 found, check bit 6
	bsf	bitNum,5	; mark bit 5 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD5 = 0
	goto	set_5		; bit = 1, so goto set bit 0
	bcf	tempD,5		; bit = 0, so set tempD5 = 0
	goto	doneKb		; done getting bit	
set_5	bsf	tempD,5		; bit 5 = 1
	goto	doneKb		; done getting bit

get_6	btfsc	bitNum,6	; check for bit 6
	goto	get_7		; since bit 6 found, check bit 7
	bsf	bitNum,6	; mark bit 6 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD6 = 0
	goto	set_6		; bit = 1, so goto set bit 0
	bcf	tempD,6		; bit = 0, so set tempD6 = 0
	goto	doneKb		; done getting bit 6	
set_6	bsf	tempD,6		; bit 6 = 1
	goto	doneKb		; done getting bit 6

get_7	btfsc	bitNum,7	; check for bit 7
	goto	get_p		; since bit 7 found, check parity bit
	bsf	bitNum,7	; mark bit 7 as found
	btfsc	tempBit,0	; if bit = 0, bit tempD7 = 0
	goto	set_7		; bit = 1, so goto set bit 0
	bcf	tempD,7		; bit = 0, so set tempD7 = 0
	goto	doneKb		; done getting bit 7
set_7	bsf	tempD,7		; bit 7 = 1
	goto	doneKb		; done getting bit 7

get_p	btfsc	byteStat,1	; check for parity 
	goto	get_stp		; since parity found, get stop bit
	bsf	byteStat,1	; mark parity bit as found
	goto 	doneKb		; done getting parity bit

get_stp	bsf	byteStat,2	; mark stop bit as found

	btfsc	byteStat,3	; Check for break code found
	goto	cont0		; if break code found, continue
	movlw	0xF0		; check for break code scanned
	xorwf	tempD,w		; 
	btfsc	STATUS,2	; if break code found, set byteStat,3
	goto	ignoreN		; ignore next
	movf	tempsc,w	; check for repeated scancode
	xorwf	tempD,w		; 
	btfsc	STATUS,2	; if repeated code found, 
	goto	cont		; ignore it
	movlw	0x12		; check for a left shift
	xorwf	tempD,w
	btfsc	STATUS,Z
	goto	setShift	; if there is a left shift then set a bit
	movlw	0x59		; check for right shift
	xorwf	tempD,w
	btfsc	STATUS,Z
	goto	setShift
	movlw	0xaa
	xorwf	tempD,w
	btfsc	STATUS,Z
	goto	AAfound
	goto	store		; if not, then put in buffer

AAfound
	bsf	STATUS,5
	bsf	PCON,1
	bcf	STATUS,5
	bsf	onetime,0
	goto	cont

setShift
	bsf	sMode,0		; set bit0 of sMode to indicate a shift
	goto	cont
clearShift
	bcf	sMode,0		; clear the shift bit
	goto	cont

ignoreN	clrf	byteStat	; clear all of byteStat
	bsf	byteStat,3	; set bit to ignore next byte
	goto	contI		; continue without clearing byteStat,3

store	movf	tempD,w		; store scancode
	movwf	tempsc		; 
	movwf	sCode		; 	
	movlw	0x80		; check for end of buffer
	xorwf	buffin,w	; compare with address of buffin
	btfsc	STATUS,2	;
	goto	fullbuf		; if end of buffer, check for full buffer
	goto	incre		; if not end, increment buffin address

fullbuf	movlw	0x70		; check for buffer full
	xorwf	buffout,w	; compare to buffout with begginning of buffer
	btfsc	STATUS,2	;
	goto	cont		; if buffer full, ignore byte
	movlw	0x70		; address for buffin is one less than 
	movwf	buffin		; beginning of buffer

	
incre	movf	buffin,w	; move address of buffin to w register
	movwf	FSR		; move buffin address to FSR
	movf	sCode,w		; copy tempD to working register
	movwf	INDF		; soter scancode at buffin address
	bsf	PORTB,5		; set interrupt for CPU --> input on buffer
	incf	buffin,f		; increment the pointer by one byte
	goto	cont

cont0	clrf	tempsc		; clear the temp key holder
	movlw	0x12		; check for a left shift
	xorwf	tempD,w
	btfsc	STATUS,Z
	goto	clearShift	; if there is a left shift then set a bit
	movlw	0x59		; check for right shift
	xorwf	tempD,w
	btfsc	STATUS,Z
	goto	clearShift

cont	clrf	byteStat	; clear the byte status
contI	clrf	tempBit		; clear the temp bit
	clrf	tempD		; clear the temp scancode
	clrf	bitNum		; clear bitNum

doneKb	movf	STATUS_temp,w	; replace value of STATUS register
	movwf	STATUS		;
	movf	w_temp,w	; replace value of w register

	bcf	INTCON,0	; clear keyboard interrupt
	return
;------------------------------------------------------------------------------


;--------------------------- srlint -------------------------------------------
; This subroutine gets called when a serial interrup occurs.  The
; subroutine receives the input and stores result in buffer, advancing the
; serial buffer pointers accordingly.
;------------------------------------------------------------------------------
srlint	btfss	INTCON,6	; check if interrupt is from keyboard in
	return			; if not, return

	movwf	w_temp		; save the contents of w register
	movf	STATUS,w	; save the STATUS Register
	movwf	STATUS_temp	;
	clrf	STATUS		; Clear STATUS byte for ISR

	movf	RCREG,w		; read the receive register
	movwf	haha		; move the byte and store it in haha
	bsf	PORTB,6		; set interrup to CPU

doneSrl	movf	STATUS_temp,w	; replace value of STATUS register
	movwf	STATUS		;
	movf	w_temp,w	; replace value of w register

	return
;------------------------------------------------------------------------------



start	nop			; 

;------------------------------------------------------------------------------
; The following sections is storing the scancode or ascii for the given       |
; characters in the four banks of memory.  Bank 0 contains the scancode       |
; for the characters, bank 2 contains their ascii values.  Bank 1 contain     |
; the scancodes of the Shifted characters, and bank 3 contains their ascii    |
; equivalent values.                                                          |
;------------------------------------------------------------------------------

;--------------------------------- Bank 0 -------------------------------------
lshift	equ	0x20		; Memory location for
	movlw	0x12		; Scancode for "Left Shift"
	movwf	lshift		;
capsl	equ	0x21		; Memory location for
	movlw	0x58		; Scancode for "Caps Lock"
	movwf	capsl		;
rshift	equ	0x22		; Memory location for 
	movlw	0x59		; Scancode for "Right Shift"
	movwf	rshift		;
tab	equ	0x23		; Memory location for
	movlw	0x0D		; Scancode for "Tab"
	movwf	tab		;
comilla	equ	0x24		; Memory location for
	movlw	0x0E		; Scancode for " ` "
	movwf	comilla		;
qq	equ	0x25		; Memory location for 
	movlw	0x15		; Scancode for "q"
	movwf	qq		;
one	equ	0x26		; Memory location for 
	movlw	0x16		; Scancode for "1"
	movwf	one		;
zz	equ	0x27		; Memory location for 
	movlw	0x1A		; Scancode for "z" 
	movwf	zz		;
ss	equ	0x28		; Memory location for 
	movlw	0x1B		; Scancode for "s"
	movwf	ss		;
aa	equ	0x29		; Memory location for 
	movlw	0x1C		; Scancode for "a"
	movwf	aa		;
ww	equ	0x2A		; Memory location for 
	movlw	0x1D		; Scancode for "w"
	movwf	ww		;
two	equ	0x2B		; Memory location for 
	movlw	0x1E		; Scancode for "2"
	movwf	two		;
cc	equ	0x2C		; Memory location for 
	movlw	0x21		; Scancode for "c" 
	movwf	cc		;
xx	equ	0x2D		; Memory location for 
	movlw	0x22		; Scancode for "x"
	movwf	xx		;
dd	equ	0x2E		; Memory location for 
	movlw	0x23		; Scancode for "d"
	movwf	dd		;
ee	equ	0x2F		; Memory location for 
	movlw	0x24		; Scancode for "e"
	movwf	ee		;
four	equ	0x30		; Memory location for
	movlw	0x25		; Scancode for "4"
	movwf	four		;
three	equ	0x31		; Memory location for 
	movlw	0x26		; Scancode for "3"
	movwf	three		;
spacio	equ	0x32		; Memory location for 
	movlw	0x29		; Scancode for space bar
	movwf	spacio		;
vv	equ	0x33		; Memory location for 
	movlw	0x2A		; Scancode for "v"
	movwf	vv		;
ff	equ	0x34		; Memory location for
	movlw	0x2B		; Scancode for "f"
	movwf	ff		;
tt	equ	0x35		; Memory location for 
	movlw	0x2C		; Scancode for "t"
	movwf	tt		;
rr	equ	0x36		; Memory location for
	movlw	0x2D		; Scancode for "r"
	movwf	rr		;
five	equ	0x37		; Memory location for
	movlw	0x2E		; Scancode for "5"
	movwf	five		;
nn	equ	0x38		; Memory location for
	movlw	0x31		; Scancode for "n"
	movwf	nn		;
bb	equ	0x39		; Memory location for
	movlw	0x32		; Scancode for "b"
	movwf	bb		;
hh	equ	0x3A		; Memory location for
	movlw	0x33		; Scancode for "h"
	movwf	hh		;
gg	equ	0x3B		; Memory location for
	movlw	0x34		; Scancode for "g"
	movwf	gg		; 
yy	equ	0x3C		; Memory location for
	movlw	0x35		; Scancode for "y"
	movwf	yy		;
six	equ	0x3D		; Memory location for
	movlw	0x36		; Scancode for "6"
	movwf	six		;
mm	equ	0x3E		; Memory location for
	movlw	0x3A		; Scancode for "m"
	movwf	mm		;
jj	equ	0x3F		; Memory location for
	movlw	0x3B		; Scancode for "j"
	movwf	jj		;
uu	equ	0x40		; Memory location for
	movlw	0x3C		; Scancode for "u"
	movwf	uu		;
seven	equ	0x41		; Memory location for
	movlw	0x3D		; Scancode for "7"
	movwf	seven		;
eight	equ	0x42		; Memory location for
	movlw	0x3E		; Scancode for "8"
	movwf	eight		;
coma	equ	0x43		; Memory location for
	movlw	0x41		; Scancode for ","
	movwf	coma		;
kk	equ	0x44		; Memory location for
	movlw	0x42		; Scancode for "k"
	movwf	kk		;
ii	equ	0x45		; Memory location for
	movlw	0x43		; Scancode for "i"
	movwf	ii		;
oo	equ	0x46		; Memory location for
	movlw	0x44		; Scancode for "o"
	movwf	oo		;
zero	equ	0x47		; Memory location for
	movlw	0x45		; Scancode for "0"
	movwf	zero		;
nine	equ	0x48		; Memory location for
	movlw	0x46		; Scancode for "9"
	movwf	nine		;
punto	equ	0x49		; Memory location for
	movlw	0x49		; Scancode for "."
	movwf	punto		;
bslash	equ	0x4A		; Memory location for
	movlw	0x4A		; Scancode for "/"
	movwf	bslash		;
ll	equ	0x4B		; Memory location for
	movlw	0x4B		; Scancode for "l"
	movwf	ll		;
colon	equ	0x4C		; Memory location for
	movlw	0x4C		; Scancode for ";"
	movwf	colon		;
pp	equ	0x4D		; Memory location for
	movlw	0x4D		; Scancode for "p"
	movwf	pp		;
dash	equ	0x4E		; Memory location for
	movlw	0x4E		; Scancode for "-"
	movwf	dash		;
apostro	equ	0x4F		; Memory location for
	movlw	0x52		; Scancode for " ' "
	movwf	apostro		;
obrakt	equ	0x50		; Memory location for
	movlw	0x54		; Scancode for "["
	movwf	obrakt		;
eqals	equ	0x51		; Memory location for
	movlw	0x55		; Scancode for "="
	movwf	eqals		;
enter	equ	0x52		; Memory location for
	movlw	0x5A		; Scancode for "Enter"
	movwf	enter		;
cbrakt	equ	0x53		; Memory location for
	movlw	0x5B		; Scancode for "]"
	movwf	cbrakt		;
fslash	equ	0x54		; Memory location for
	movlw	0x5D		; Scancode for "\"
	movwf	fslash		;
backspc	equ	0x55		; Memory location for
	movlw	0x66		; Scancode for "Backspace"
	movwf	backspc		;
escape	equ	0x56		; Memory location for
	movlw	0x76		; Scancode for "Esc"
	movwf	escape		;
;---------------------------------- -------------------------------------------

	bsf	STATUS,6	; Change to Bank 2 for Ascii

;--------------------------------- Bank 2 -------------------------------------
atab	equ	0x123		; Memory location for
	movlw	0x09		; Ascii for "Tab"
	movwf	atab		;
acomia	equ	0x124		; Memory location for
	movlw	0x60		; Ascii for " ` "
	movwf	acomia		;
aqq	equ	0x125		; Memory location for
	movlw	0x71		; Ascii for "q"
	movwf	aqq		;
aone	equ	0x126		; Memory location for
	movlw	0x31		; Ascii for "1"
	movwf	aone		;
azz	equ	0x127		; Memory location for
	movlw	0x7A		; Ascii for "z"
	movwf	azz		;
ass	equ	0x128		; Memory location for
	movlw	0x73		; Ascii for "s"
	movwf	ass		;
aaa	equ	0x129		; Memory location for
	movlw	0x61		; Ascii for "a"
	movwf	aaa		;
aww	equ	0x12A		; Memory location for
	movlw	0x77		; Ascii for "w"
	movwf	aww		;
atwo	equ	0x12B		; Memory location for
	movlw	0x32		; Ascii for "2"
	movwf	atwo		;
acc	equ	0x12C		; Memory location for
	movlw	0x63		; Ascii for "c"
	movwf	acc		;
axx	equ	0x12D		; Memory location for
	movlw	0x78		; Ascii for "x"
	movwf	axx		;
add	equ	0x12E		; Memory location for
	movlw	0x64		; Ascii for "d"
	movwf	add		;
aee	equ	0x12F		; Memory location for
	movlw	0x65		; Ascii for "e"
	movwf	aee		;
afour	equ	0x130		; Memory location for
	movlw	0x34		; Ascii for "4"
	movwf	afour		;
athree	equ	0x131		; Memory location for
	movlw	0x33		; Ascii for "3"
	movwf	athree		;
aspacio	equ	0x132		; Memory location for
	movlw	0x20		; Ascii for space bar
	movwf	aspacio		;
avv	equ	0x133		; Memory location for
	movlw	0x76		; Ascii for "v"
	movwf	avv		;
aff	equ	0x134		; Memory location for
	movlw	0x66		; Ascii for "f"
	movwf	aff		;
att	equ	0x135		; Memory location for
	movlw	0x74		; Ascii for "t"
	movwf	att		;
arr	equ	0x136		; Memory location for
	movlw	0x72		; Ascii for "r"
	movwf	arr		;
afive	equ	0x137		; Memory location for
	movlw	0x35		; Ascii for "5"
	movwf	afive		;
ann	equ	0x138		; Memory location for
	movlw	0x6E		; Ascii for "n"
	movwf	ann		;
abb	equ	0x139		; Memory location for
	movlw	0x62		; Ascii for "b"
	movwf	abb		;
ahh	equ	0x13A		; Memory location for
	movlw	0x68		; Ascii for "h"
	movwf	ahh		;
agg	equ	0x13B		; Memory location for
	movlw	0x67		; Ascii for "g"
	movwf	agg		;
ayy	equ	0x13C		; Memory location for
	movlw	0x79		; Ascii for "y"
	movwf	ayy		;
asix	equ	0x13D		; Memory location for
	movlw	0x36		; Ascii for "6"
	movwf	asix		;
amm	equ	0x13E		; Memory location for
	movlw	0x6D		; Ascii for "m"
	movwf	amm		;
ajj	equ	0x13F		; Memory location for
	movlw	0x6A		; Ascii for "j"
	movwf	ajj		;	
auu	equ	0x140		; Memory location for
	movlw	0x75		; Ascii for "u"
	movwf	auu		;
aseven	equ	0x141		; Memory location for
	movlw	0x37		; Ascii for "7"
	movwf	aseven		;
aeight	equ	0x142		; Memory location for
	movlw	0x38		; Ascii for "8"
	movwf	aeight		;
acoma	equ	0x143		; Memory location for
	movlw	0x2C		; Ascii for ","
	movwf	acoma		;
akk	equ	0x144		; Memory location for
	movlw	0x6B		; Ascii for "k"
	movwf	akk		;
aii	equ	0x145		; Memory location for
	movlw	0x69		; Ascii for "i"
	movwf	aii		;
aoo	equ	0x146		; Memory location for
	movlw	0x6F		; Ascii for "o"
	movwf	aoo		;
azero	equ	0x147		; Memory location for
	movlw	0x30		; Ascii for "0"
	movwf	azero		;
anine	equ	0x148		; Memory location for
	movlw	0x39		; Ascii for "9"
	movwf	anine		;
apunto	equ	0x149		; Memory location for
	movlw	0x2E		; Ascii for "."
	movwf	apunto		;
abslash	equ	0x14A		; Memory location for
	movlw	0x2F		; Ascii for "/"	
	movwf	abslash		;
all	equ	0x14B		; Memory location for
	movlw	0x6C		; Ascii for "l"
	movwf	all		; 
acolon	equ	0x14C		; Memory location for
	movlw	0x3B		; Ascii for ";"
	movwf	acolon		;
app	equ	0x14D		; Memory location for
	movlw	0x70		; Ascii for "q"
	movwf	app		;		
adash	equ	0x14E		; Memory location for
	movlw	0x2D		; Ascii for "-"
	movwf	adash		;
aapost	equ	0x14F		; Memory location for
	movlw	0x27		; Ascii for " ' "
	movwf	aapost		;
aobrakt	equ	0x150		; Memory location for
	movlw	0x5B		; Ascii for "["
	movwf	aobrakt		;
aeqals	equ	0x151		; Memory location for
	movlw	0x3D		; Ascii for "="
	movwf	aeqals		;
aenter	equ	0x152		; Memory location for
	movlw	0x0D		; Ascii for "Enter"
	movwf	aenter		;	
acbrakt	equ	0x153		; Memory location for
	movlw	0x5D		; Ascii for "]"
	movwf	acbrakt		;
afslash	equ	0x154		; Memory location for
	movlw	0x5C		; Ascii for "\"
	movwf	afslash		;
abacks	equ	0x155		; Memory location for
	movlw	0x08		; Ascii for "Backspace"
	movwf	abacks		;
aescape	equ	0x156		; Memory location for
	movlw	0x1B		; Ascii for "Esc"
	movwf	aescape		;
;------------------------------------------------------------------------------

	bcf	STATUS,6	; Clear bit STAUS.6 to return to Bank 0
	bsf	STATUS,5	; Set bit STATUS.5 to change to Bank 1

;--------------------------------- Bank 1 -------------------------------------
slshift	equ	0xA0		; Memory location for
	movlw	0x12		; Scancode for "Left Shift"
	movwf	slshift		;
scapsl	equ	0xA1		; Memory location for
	movlw	0x58		; Scancode for "Caps Lock"
	movwf	scapsl		;
srshift	equ	0xA2		; Memory location for 
	movlw	0x59		; Scancode for "Right Shift"
	movwf	srshift		;
stab	equ	0xA3		; Memory location for
	movlw	0x0D		; Scancode for "Tab"
	movwf	stab		;
tilda	equ	0xA4		; Memory location for
	movlw	0x0E		; Scancode for " ~ "
	movwf	tilda		;
sqq	equ	0xA5		; Memory location for 
	movlw	0x15		; Scancode for "Q"
	movwf	sqq		;
excla	equ	0xA6		; Memory location for 
	movlw	0x16		; Scancode for "!"
	movwf	excla		;
szz	equ	0xA7		; Memory location for 
	movlw	0x1A		; Scancode for "Z" 
	movwf	szz		;
sss	equ	0xA8		; Memory location for 
	movlw	0x1B		; Scancode for "S"
	movwf	sss		;
saa	equ	0xA9		; Memory location for 
	movlw	0x1C		; Scancode for "A"
	movwf	saa		;
sww	equ	0xAA		; Memory location for 
	movlw	0x1D		; Scancode for "W"
	movwf	sww		;
at	equ	0xAB		; Memory location for 
	movlw	0x1E		; Scancode for "@"
	movwf	at		;
scc	equ	0xAC		; Memory location for 
	movlw	0x21		; Scancode for "C" 
	movwf	scc		;
sxx	equ	0xAD		; Memory location for 
	movlw	0x22		; Scancode for "X"
	movwf	sxx		;
sdd	equ	0xAE		; Memory location for 
	movlw	0x23		; Scancode for "D"
	movwf	sdd		;
see	equ	0xAF		; Memory location for 
	movlw	0x24		; Scancode for "E"
	movwf	see		;
dollar	equ	0xB0		; Memory location for 
	movlw	0x25		; Scancode for "$"
	movwf	dollar		;
hash	equ	0xB1		; Memory location for 
	movlw	0x26		; Scancode for "#"
	movwf	hash		;
sspacio	equ	0xB2		; Memory location for 
	movlw	0x29		; Scancode for spacebar
	movwf	sspacio		;
svv	equ	0xB3		; Memory location for 
	movlw	0x2A		; Scancode for "V"
	movwf	svv		;
sff	equ	0xB4		; Memory location for 
	movlw	0x2B		; Scancode for "F"
	movwf	sff		;
stt	equ	0xB5		; Memory location for 
	movlw	0x2C		; Scancode for "T"
	movwf	stt		;
srr	equ	0xB6		; Memory location for
	movlw	0x2D		; Scancode for "R"
	movwf	srr		;
percent	equ	0xB7		; Memory location for
	movlw	0x2E		; Scancode for "%"
	movwf	percent		;
snn	equ	0xB8		; Memory location for
	movlw	0x31		; Scancode for "N"
	movwf	snn		;
sbb	equ	0xB9		; Memory location for
	movlw	0x32		; Scancode for "B"
	movwf	sbb		;
shh	equ	0xBA		; Memory location for
	movlw	0x33		; Scancode for "H"
	movwf	shh		;
sgg	equ	0xBB		; Memory location for
	movlw	0x34		; Scancode for "G"
	movwf	sgg		; 
syy	equ	0xBC		; Memory location for
	movlw	0x35		; Scancode for "Y"
	movwf	syy		;
carrot	equ	0xBD		; Memory location for
	movlw	0x36		; Scancode for " ^ "
	movwf	carrot		;
smm	equ	0xBE		; Memory location for
	movlw	0x3A		; Scancode for "M"
	movwf	smm		;
sjj	equ	0xBF		; Memory location for
	movlw	0x3B		; Scancode for "J"
	movwf	sjj		;
suu	equ	0xC0		; Memory location for
	movlw	0x3C		; Scancode for "U"
	movwf	suu		;
anper	equ	0xC1		; Memory location for
	movlw	0x3D		; Scancode for "&"
	movwf	anper		;
star	equ	0xC2		; Memory location for
	movlw	0x3E		; Scancode for "*"
	movwf	star		;
slesst	equ	0xC3		; Memory location for
	movlw	0x41		; Scancode for "<"
	movwf	slesst		;
skk	equ	0xC4		; Memory location for
	movlw	0x42		; Scancode for "K"
	movwf	skk		;
sii	equ	0xC5		; Memory location for
	movlw	0x43		; Scancode for "I"
	movwf	sii		;
soo	equ	0xC6		; Memory location for
	movlw	0x44		; Scancode for "O"
	movwf	soo		;
cParen	equ	0xC7		; Memory location for
	movlw	0x45		; Scancode for ")"
	movwf	cParen		;
oParen	equ	0xC8		; Memory location for
	movlw	0x46		; Scancode for "("
	movwf	oParen		;
smoret	equ	0xC9		; Memory location for
	movlw	0x49		; Scancode for ">"
	movwf	smoret		;
quest	equ	0xCA		; Memory location for
	movlw	0x4A		; Scancode for "?"
	movwf	quest		;
sll	equ	0xCB		; Memory location for
	movlw	0x4B		; Scancode for "L"
	movwf	sll		;
scolon	equ	0xCC		; Memory location for
	movlw	0x4C		; Scancode for ":"
	movwf	scolon		;
spp	equ	0xCD		; Memory location for
	movlw	0x4D		; Scancode for "P"
	movwf	spp		;
under	equ	0xCE		; Memory location for
	movlw	0x4E		; Scancode for "_"
	movwf	under		;
quote	equ	0xCF		; Memory location for
	movlw	0x52		; Scancode for "
	movwf	quote		;
ocurly	equ	0xD0		; Memory location for
	movlw	0x54		; Scancode for "{"
	movwf	ocurly		;
plus	equ	0xD1		; Memory location for
	movlw	0x55		; Scancode for "+"
	movwf	plus		;
senter	equ	0xD2		; Memory location for
	movlw	0x5A		; Scancode for "Enter"
	movwf	senter		;
ccurly	equ	0xD3		; Memory location for
	movlw	0x5B		; Scancode for "}"
	movwf	ccurly		;
bar	equ	0xD4		; Memory location for
	movlw	0x5D		; Scancode for "|"
	movwf	bar		;
sbacks	equ	0xD5		; Memory location for
	movlw	0x66		; Scancode for "Backspace"
	movwf	sbacks		;
sescape	equ	0xD6		; Memory location for
	movlw	0x76		; Scancode for "Esc"
	movwf	sescape		;
;------------------------------------------------------------------------------

	bsf	STATUS,6	; Set bit STATUS.6 to change to Bank 3
				; Bit STATUS.5 is already set

;--------------------------------- Bank 3 -------------------------------------
astab	equ	0x1A3		; Memory location for
	movlw	0x09		; Ascii for "Tab"
	movwf	astab		;
atilda	equ	0x1A4		; Memory location for 
	movlw	0x7E		; Ascii for " ~ "
	movwf	atilda		;
asqq	equ	0x1A5		; Memory location for 
	movlw	0x51		; Ascii for "Q"
	movwf	asqq		;
aexcla	equ	0x1A6		; Memory location for 
	movlw	0x21		; Ascii for "!"
	movwf	aexcla		;
aszz	equ	0x1A7		; Memory location for 
	movlw	0x5A		; Ascii for "Z" 
	movwf	aszz		;
asss	equ	0x1A8		; Memory location for 
	movlw	0x53		; Ascii for "S"
	movwf	asss		;
asaa	equ	0x1A9		; Memory location for 
	movlw	0x41		; Ascii for "A"
	movwf	asaa		;
asww	equ	0x1AA		; Memory location for 
	movlw	0x57		; Ascii for "W"
	movwf	asww		;
aat	equ	0x1AB		; Memory location for 
	movlw	0x40		; Ascii for "@"
	movwf	aat		;
ascc	equ	0x1AC		; Memory location for 
	movlw	0x43		; Ascii for "C" 
	movwf	ascc		;
asxx	equ	0x1AD		; Memory location for 
	movlw	0x58		; Ascii for "X"
	movwf	asxx		;
asdd	equ	0x1AE		; Memory location for 
	movlw	0x44		; Ascii for "D"
	movwf	asdd		;
asee	equ	0x1AF		; Memory location for 
	movlw	0x45		; Ascii for "E"
	movwf	asee		;
adollar	equ	0x1B0		; Memory location for 
	movlw	0x24		; Ascii for "$"
	movwf	adollar		;
ahash	equ	0x1B1		; Memory location for 
	movlw	0x23		; Ascii for "#"
	movwf	ahash		;
asspac	equ	0x1B2		; Memory location for
	movlw	0x20		; Ascii for space
	movwf	asspac		;
asvv	equ	0x1B3		; Memory location for
	movlw	0x56		; Ascii for "V"
	movwf	asvv		;
asff	equ	0x1B4		; Memory location for
	movlw	0x46		; Ascii for "F"
	movwf	asff		;
astt	equ	0x1B5		; Memory location for 
	movlw	0x54		; Ascii for "T"
	movwf	astt		;
asrr	equ	0x1B6		; Memory location for
	movlw	0x52		; Ascii for "R"
	movwf	asrr		;
apercen	equ	0x1B7		; Memory location for
	movlw	0x25		; Ascii for "%"
	movwf	apercen		;
asnn	equ	0x1B8		; Memory location for
	movlw	0x4E		; Ascii for "N"
	movwf	asnn		;
asbb	equ	0x1B9		; Memory location for
	movlw	0x42		; Ascii for "B"
	movwf	asbb		;
ashh	equ	0x1BA		; Memory location for
	movlw	0x48		; Ascii for "H"
	movwf	ashh		;
asgg	equ	0x1BB		; Memory location for
	movlw	0x47		; Ascii for "G"
	movwf	asgg		; 
asyy	equ	0x1BC		; Memory location for
	movlw	0x59		; Ascii for "Y"
	movwf	asyy		;
acarrot	equ	0x1BD		; Memory location for
	movlw	0x5E		; Ascii for " ^ "
	movwf	acarrot		;
asmm	equ	0x1BE		; Memory location for
	movlw	0x4D		; Ascii for "M"
	movwf	asmm		;
asjj	equ	0x1BF		; Memory location for
	movlw	0x4A		; Ascii for "J"
	movwf	asjj		;
asuu	equ	0x1C0		; Memory location for
	movlw	0x55		; Ascii for "U"
	movwf	asuu		;
aanper	equ	0x1C1		; Memory location for
	movlw	0x26		; Ascii for "&"
	movwf	aanper		;
astar	equ	0x1C2		; Memory location for
	movlw	0x2A		; Ascii for "*"
	movwf	astar		;
alesst	equ	0x1C3		; Memory location for
	movlw	0x3C		; Ascii for "<"
	movwf	alesst		;
askk	equ	0x1C4		; Memory location for
	movlw	0x4B		; Ascii for "K"
	movwf	askk		;
asii	equ	0x1C5		; Memory location for
	movlw	0x49		; Ascii for "I"
	movwf	asii		;
asoo	equ	0x1C6		; Memory location for
	movlw	0x4F		; Ascii for "O"
	movwf	asoo		;
acParen	equ	0x1C7		; Memory location for
	movlw	0x29		; Ascii for ")"
	movwf	acParen		;
aoParen	equ	0x1C8		; Memory location for
	movlw	0x28		; Ascii for "("
	movwf	aoParen		;
amoret	equ	0x1C9		; Memory location for
	movlw	0x3E		; Ascii for ">"
	movwf	amoret		;
aquest	equ	0x1CA		; Memory location for
	movlw	0x3F		; Ascii for "?"
	movwf	aquest		;
asll	equ	0x1CB		; Memory location for
	movlw	0x4C		; Ascii for "L"
	movwf	asll		;
ascolon	equ	0x1CC		; Memory location for
	movlw	0x3A		; Ascii for ":"
	movwf	ascolon		;
aspp	equ	0x1CD		; Memory location for
	movlw	0x50		; Ascii for "P"
	movwf	aspp		;
aunder	equ	0x1CE		; Memory location for
	movlw	0x5F		; Ascii for "_"
	movwf	aunder		;
aquote	equ	0x1CF		; Memory location for
	movlw	0x22		; Ascii for "
	movwf	aquote		;
aocurly	equ	0x1D0		; Memory location for
	movlw	0x7B		; Ascii for "{"
	movwf	aocurly		;
aplus	equ	0x1D1		; Memory location for
	movlw	0x2B		; Ascii for "+"
	movwf	aplus		;
asenter	equ	0x1D2		; Memory location for
	movlw	0x0D		; Ascii for "Enter"
	movwf	asenter		;
accurly	equ	0x1D3		; Memory location for
	movlw	0x7D		; Ascii for "}"
	movwf	accurly		;
abar	equ	0x1D4		; Memory location for
	movlw	0x7C		; Ascii for "|"
	movwf	abar		;
asbacks	equ	0x1D5		; Memory location for
	movlw	0x08		; Ascii for "Backspace"
	movwf	asbacks		;
asescap	equ	0x1D6		; Memory location for
	movlw	0x1B		; Ascii for "Esc"
	movwf	asescap		; 
;------------------------------------------------------------------------------

	bcf	STATUS,5	; Clear bit STATUS.5 and STATUS.6 to 
	bcf	STATUS,6	; To return to Bank 0

;------------------------------------------------------------------------------
tempBit	equ	0x57		; temporary byte
	clrf	tempBit		; clearing temporary byte
	movlw	0XFF		; move 1111 1111 to w
	movwf	tempBit		; set tempBit to 1111 1111
tempD	equ	0x58		; temporary values for scancode
	clrf	tempD		; clear tempD
asciiH	equ	0x59		; location for Ascii value
	clrf	asciiH		;
sMode	equ	0x5A		; flag for shift mode
	clrf	sMode		;
breakc	equ	0x5B		; memory location for 
	movlw	0xF0		; break code
	movwf	breakc		;
opcode	equ	0x5C		; memory location for opcode
	movlw	0xFF		; acknoledgement opcode
	movwf	opcode		;
opc	equ	0x5D		; memory location for working opcode
	clrf	opc		; 
hi	equ	0x5F		; memory location for hello acknowledge
	movlw	0xA5		; hello return byte
	movwf	hi		
buffin	equ	0x60		; holds memory location for buffer in
	movlw	0x70		; begginning of keyboard buffer
	movwf	buffin		; 
buffout	equ	0x61		; holds memory location for buffer out
	movlw	0x70		; begginning of keyboard buffer
	movwf	buffout		;
sCode	equ	0x62		; location for scancode
	clrf	sCode		; clear sCode
value	equ	0x63		; holds value to put on output port
	clrf	value		;
w_temp	equ	0x64		; memory location for temp register w
STATUS_temp equ	0x65		; memory location for temp STATUS register
bitNum	equ	0x66		; memory loacation for bitNum
	clrf	bitNum		;
byteStat equ	0x67		; memory location for byte status
	clrf	byteStat	;
tempsc	equ	0x68		; temporary code
	clrf	tempsc
srlSta	equ	0x69		; memory location for serial status
	clrf	srlSta		
haha	equ	0x6A		; memeory location for serial byte
	clrf	haha
onetime	equ	0x6B
	clrf	onetime

;------------------------------- begin ---------------------------------------
; This is the start of the program. 
;-----------------------------------------------------------------------------
begin	movlw	0x00		; clear w
	clrf	PORTA
	
	bsf	STATUS,5	; change to bank 1

	movlw	0x06		; binary 0000 011X
	movwf	ADCON1		; configure all pins as digital inputs

	movlw	0x01		; binary 0000 0001
	movwf	TRISA		; A0 as input, A1 - A5 as outputs

	movlw	0x87		; binary 1000 0111
	movwf	TRISB		; Setting B7 as input KB Clock
				; B4 Latch enable out, B0-B2 Opcode in

	movlw	0xC2		; binary 1100 0010
	movwf	TRISC		; Setting Port  C7 and C6 as serial inputs
				; Setting C1 to VACK(input) and C0 VDATA(output)
	
	movlw	0xFF		; binary 1111 1111
	movwf	TRISD		; Setting Port D as data port. Input by default
	
	movlw	0x03		; binary 0000 0011
	movwf	TRISE		; Setting E0 as input Data from Keyboard
				; E1 input I/O line, E2 output ACK

	bcf	OPTION_REG,7

	bcf	STATUS,5	; change back to bank 0
	
	clrf	PORTA		; Clear port A
	clrf	PORTB		; Clear port B
	clrf	PORTC		; Clear port C
	clrf 	PORTD		; Clear port D
	clrf	PORTE		; clear port E
	
; ~~ Setting up Serial Communication ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~
	bsf	STATUS,5	; change to bank 1 for serial port config
	movlw	0x81		; 81 hex = 129 decimal
	movwf	SPBRG		; Fosc = 9.6k baud at 20Mhz
	bsf	TXSTA,2		; set BRGH for high speed sampling
	bcf	TXSTA,4		; clearing SYNC for asynchronous serial
	bsf	TXSTA,5		; enable transmission, TXEN=1
	bcf	TXSTA,6		; 8-bit transmissions
	bcf	PIE1,4		; Diable transfer interrupt
	bsf	PIE1,5		; Enable interrupt for Serial Receive
	bcf	STATUS,5	; change to bank 0	
	bsf	RCSTA,7		; SPEN bit set for serial port enable
	bsf	RCSTA,4		; Enable asynchronous reception CREN =1
 	bsf	PIR1,4		; set TXIF=1 for TXREG empty
; ~~  ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~

; ~~ Enable interrupts for Serial Communication and Keyboard In ~~ ~~ ~~ ~
	bcf	INTCON,0	; Clear keyboard interrup
	bsf	INTCON,7	; Global Interrupt enable
	bsf	INTCON,3	; RBIE interrupt enable 
	bsf	INTCON,6	; PEIE interrupt enable
; ~~  ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~ ~~

; !!! TO DO ~~ clear out buffer

ioset	clrf	opc		; clear working opcode
	btfss	PORTE,1		; Test for I/O Address line to be set
	goto	ioset		; If not ready, test again
	bcf	STATUS,5	; Make sure bank 0 is current
	bcf	STATUS,6	; Make sure bank 0 is current
	movf	PORTB,w		; read opcode
	movwf	opcode		; store opcode

	btfsc	opcode,2	; test for opcode 2 set
	bsf	opc,2		; if not set, clear opc2
	btfsc	opcode,1	; test for opcode 1 set
	bsf	opc,1		; if not set, clear opc1
	btfsc	opcode,0	; test for opcode 0 set
	bsf	opc,0		; if not set, clear opc0

	movlw	0x00		; test for keyboard in opcode
	xorwf	opc,w		;
	btfsc	STATUS,2	;
	goto 	keyout		; opcode for keyboard in found

	movlw	0x01		; test for serial in opcode
	xorwf	opc,w		;
	btfsc	STATUS,2	;
	goto	srlin		; opcode for serial in found

	movlw	0x02		; test for serial out opcode
	xorwf	opc,w		;
	btfsc	STATUS,2	;
	goto 	srlout		; opcode for serial out found

	movlw	0x03		; test for Video out opcode
	xorwf	opc,w		;
	btfsc	STATUS,2	;
	goto	video		; opcode for video out found

	movlw	0x04		; test for Latch LED Array opcode
	xorwf	opc,w		;
	btfsc	STATUS,2	;
	goto 	latch		; opcode for Latch LED Array found

	movlw	0x07		; test for Hello out opcode
	xorwf	opc,w		;
	btfsc	STATUS,2	;
	goto	hello		; opcode for Hello out found

	goto	badopc		; opcode not found, send error

keyout	
	btfss	PORTB,5		; is there buffer input?
	goto	keyout		; wait for input to be available on buffer

	movf	buffout,w	; move buffout pointer address to w
	movwf	FSR		; move pointer to FSR
	movf	INDF,w		; copy value to w
	movwf	value		; hold value
	incf	buffout,f	; increment buffer out pointer

	movlw	0x80		; check for end of buffer
	xorwf	buffout,w	; if buffout is end of buffer
	btfsc	STATUS,2	;
	goto	bufrol		; if end of buffer, got to bufrol
	goto	contin		; either way, increment buffer out pointer

bufrol	movf	buffin,w	; move buffer in pointer to w
	xorwf	buffout,w	; check for no more input 
	btfsc	STATUS,2	; 
	goto	reinit		; start buffer over
	movlw	0x70
	movwf	buffout		; move beginning of buffer to buffout
	goto	contin		; don't reinitiate the buffer pointers

reinit	bcf	PORTB,5		; clear keyboard interrupt to CPU
	movlw	0x70
	movwf	buffout		; move beginning of buffer to buffout
	movwf	buffin		; move beginning of buffer to buffin

contin	call 	ascii		; convert scan code to ascii
	btfsc	sMode,1		; check for shift mode
	goto 	keyout		; ignore shift scancode
	btfss	sMode,2		; 
	goto	check		; check for anymore input in buffer
	
	movlw	0x01		
	subwf	buffout,w	; move bufferout minus 1 to w	
	xorwf	buffin,w	; check for no more input 
	btfsc	STATUS,2	; 
	bcf	PORTB,5		; clear keyboard interrupt to CPU
	goto	outp		; give output

check	movf	buffin,w	; move buffer in pointer to w
	xorwf	buffout,w	; check for no more input 
	btfsc	STATUS,2	; 
	bcf	PORTB,5		; clear keyboard interrupt to CPU

outp	bsf	STATUS,5	; change to bank 1
	movlw	0x00		; binary 0000 0000
	movwf	TRISD		; setting data port as output
	bcf	STATUS,5	; change to bank 0

	movf	asciiH,w	

	bcf	STATUS,5	; Make sure current bank is 0
	bcf	STATUS,6	; Make sure current bank is 0

	movwf	PORTD		; move ascii hex value to output port

	call	complete	; tell CPU that opcode action has been completed	
	goto 	ioset		; continue with I/O

srlin	
	btfss	PORTB,6		; is there serial input?
	goto	srlin		; wait for input to be available on buffer

	bsf	STATUS,5	; change to bank 1
	movlw	0x00		; binary 0000 0000
	movwf	TRISD		; setting data port as output
	bcf	STATUS,5	; change to bank 0

	movf	haha,w		; move ascii for byte to w
	bcf	PORTB,6		; clear serial interrup to CPU

	bcf	STATUS,5	; Make sure current bank is 0
	bcf	STATUS,6	; Make sure current bank is 0
	movwf	PORTD

	call	complete	; tell CPU that opcode action has been completed	
	goto 	ioset		; continue with I/O

srlout
	
	nop			;
	btfss	PIR1,4		; test for tx ready, TXIF=1
	goto	srlout		; if not ready, wait until ready
	movf	PORTD,w		; move ascii for 1 to transmit register
	movwf	TXREG		; send all bits
	nop			;	

	call	complete	; tell CPU opcode action has been completed	
	goto 	ioset		; continue with I/O

video	
	bsf	PORTC,0		; tell video PIC data is for it
vPIC	btfss	PORTC,1		; Test for PIC to respond
	goto	vPIC		; If not ready, test again 
	bcf	PORTC,0		; clear the VDATA signal		
	call	complete	; tell CPU opcode action has been completed	
	goto 	ioset		; continue with I/O

latch	
	
	bsf	PORTB,3		; enable latch
	nop			; wait for latch set up and hold time
	nop			;
	nop			;
				; input data port D holds data to be latched	
	nop			; wait for latch set up and hold time
	nop			;
	nop			;
	bcf	PORTB,3		; disable latch
	
	call	complete	; tell CPU opcode action has been completed	
	goto 	ioset		; continue with I/O

hello	
	bcf	PORTA,5
	bsf	STATUS,5	; change to bank 1
	movlw	0x00		; binary 0000 0000
	movwf	TRISD		; setting data port as output
	bcf	STATUS,5	; change to bank 0

	movlw	0xA5		; Set acknowledgement of hello
	bcf	STATUS,5	; Make sure current bank is 0
	bcf	STATUS,6	; Make sure current bank is 0
	movwf	PORTD

	call	complete	; tell CPU opcode action has been completed
	goto 	ioset		; continue with I/O

badopc
	
	goto	ioset		; continue with I/O

;-------------------------------------------------------------------------------
;---------------------------- complete -----------------------------------------
; This subroutine sets the ACK bit and waits for I/O to be cleared.  At this 
; point the ACK gets cleared too acknoledging that the action has been executed.
; Data port is set to input again.
;-------------------------------------------------------------------------------
complete
	bsf	PORTE,2		; If ready, set ACK
ioclr	btfsc	PORTE,1		; wait for ACK to be received and I/O cleared
	goto	ioclr		; if I/O set, wait for it clear
	bsf	STATUS,5	; change to bank 1
	movlw	0xFF		; binary 1111 1111
	movwf 	TRISD		; setting data port as input
	bcf	STATUS,5
	bcf	PORTE,2		; Clear Ack 
	return
;-------------------------------------------------------------------------------
;--------------------------- ascii ---------------------------------------------
; This subroutine takes the scancode stored at sCode and finds the 
; corresponding ascii equivalent. 
;-------------------------------------------------------------------------------
ascii	bcf	sMode,1		; Clear Port A for debugging
	
	movlw	0x23		; Check for all other characters
	movwf	FSR
again	movf	value,w		;
	xorwf	INDF,w		;
	btfsc	STATUS,2	; test z bit
	goto 	match		;
	incf	FSR,f		; increment FSR
	goto	again		; try again

match
	btfss	sMode,0		; check for shift mode
	goto	normal		; if cleared, got to normal
	goto	shift		; if set, goto shift

normal	nop			; found scan code
	bsf	STATUS,6	; Bank 2
	bsf	STATUS,7	; set IRP
	movf	FSR,w		; move address of ascii hex to FSR
	movf	INDF,w		; copy ascii hex to w
	bcf	STATUS,6	; Change back to bank 0
	bcf	STATUS,7	; clear IRP
	movwf 	asciiH		; store ascii hex to asciiH
	goto	doneAH

shift	
	bsf	STATUS,6	; Change to Bank 3
	bsf	STATUS,5	;
	bsf	STATUS,7	; set IRP
	movf	FSR,w		; move address of ascii hex to FSR
	movlw	0x80
	addwf	FSR,f
	movf	INDF,w		; copy ascii hex to w
	bcf	STATUS,6	; Go back to Bank 0
	bcf	STATUS,5	;
	bcf	STATUS,7	; clear IRP
	movwf 	asciiH		; store ascii hex to asciiH
	goto	doneAH

doneAH	return
;---------------------------------------------------------------------------
	end





