/**************************************************************
* Herein lies the main loop for the robot. This file will 
* contain the code for various functions to pass data back
* and forth between the robots and earthsat. The main loop
* simply waits for the XBModule to get data and then takes
* action.
*
* Written by: Shane Avery
****************************************************************/
#include<p33FJ128GP206.h>
#include "MarsSat.h"
#include "IOFuncs.h"
#include "HumanInt.h"

// -------------------------------------------------------
// Here are the declarations for the global extern
// variables in the Robot.h header file.
// -------------------------------------------------------
// Here are the const definitions for the LEDs.
const unsigned int LED0 = 0x100;	// dsPIC Initialized OK.
const unsigned int LED1 = 0x200;	// Robot status complete.
const unsigned int LED2 = 0x400;	// We have the mutex.
const unsigned int LED3 = 0x800;
const unsigned int LED4 = 0x1000;
const unsigned int LED5 = 0x2000;	
const unsigned int LED6 = 0x4000;	// General Purpose Error (red).
const unsigned int LEDAll = 0x7F00;

// Global buffers for XBTX and XBRX.
unsigned char XBTX[17];
unsigned char XBRX[17];
unsigned char XBTXSize;
unsigned char XBRXSize;
unsigned char XBTXIndex;
unsigned char XBRXIndex;

// Global buffers for I2C EEPROM.
unsigned char I2CMSB,I2CLSB;
unsigned char I2CBuffer[64];
unsigned char I2CRW = 0;
unsigned char I2CDone = 1;
unsigned char I2CSize;

// Set the StatusData pointer variable equal to
// the address of the XB Receiver buffer.
struct StatusData *StatData = (struct StatusData *)XBRX;
struct ReceivedData RecData;
struct TransmitData TransData;

// Here are the structs to hold data about all robots and balls.
struct RobotStruct Robot[4];
struct BallStruct Ball[8];

// Global buffer that define the messages to the main loop.
char Button1Pressed = 0;
char Button2Pressed = 0;
char XBMessRec = 0;
char Mutex = 0;
char WaitForMutex = 0;
char ReleaseMtx = 0;
char GotStatus = 0;
char FirstTimeMutex = 0;
char FirstTimeMutexDone = 0;
int EEPROMAddress = 0;
char RecEEPROM = 0; 
char WriteEEPROMRX = 0;

// ---------------------------------------------------------
// End of extern global variable definitions
// ---------------------------------------------------------

// This function will determine the status of the robots. This will require
// a bit of intellegence as some robots maybe dead or unable to communicate
// with us. If we can't comm with a robot we will need to ask another robot
// if he can (we need to be sure we ask a robot that has good local comm).
// If can then get the damage via relay. Else consider him dead. Once we 
// have determined the damage to the robots we will store those pieces of 
// information in local variables, inform all robots of each others' damage
// (by relay if needed).
void GetRobotStatus()
{
	// To start off we will assume robots are all ok.
	Robot[Seeker1Arg].Status = OK;Robot[Seeker2Arg].Status = OK;
	Robot[Reader1Arg].Status = OK;Robot[Reader2Arg].Status = OK;

	// ---------------------------------------------------------
	// First lets go once around and see who can talk to us.
	// ---------------------------------------------------------
	// Ask Seeker1 if he is there and get his damage info.
	TransData.DestAddLSB = Seeker1;			// Sending to Seeker1.
	TransData.Data[0] = AreYouThereRobots;	// Send an AreYouThere.
	TransData.Data[1] = Seeker1Arg;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
	
 
	while(XBWriteAPI() == -1);				// Write the API.
	
	// This is not needed for to final configuration but instead
	// we need a little delay for the Robot Emulator tool to keep up.
	Wait100ms();
	
	// If we got data from the robot then store the status. Else
	// we should temporarily mark the robot as un-reponsive.
	if(XBMessRec)
	{
		XBMessRec = 0;		// Clear the flag;
		Robot[Seeker1Arg].Status = RecData.Data[2];	// Store the status.

		// Since we are not in the main loop to service the RX EEPROM write
		// we need to do it here. Thus we will write the RX data, increment 
		// the EEPROM address, and clear the flag.
		while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
		EEPROMAddress += sizeof(RecData.Data);	
		WriteEEPROMRX = 0;
	}
	else
	{
		Robot[Seeker1Arg].Status = 0xff;	// Mark robot as unresponsive.
	}

	// Ask Seeker2 if he is there and get his damage info.
	TransData.DestAddLSB = Seeker2;			// Sending to Seeker2.
	TransData.Data[0] = AreYouThereRobots;	// Send an AreYouThere.
	TransData.Data[1] = Seeker2Arg;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
 
	while(XBWriteAPI() == -1);				// Write the API.
	
	// This is not needed for to final configuration but instead
	// we need a little delay for the Robot Emulator tool to keep up.
	Wait100ms();
	
	// If we got data from the robot then store the status. Else
	// we should temporarily mark the robot as un-reponsive.
	if(XBMessRec)
	{
		XBMessRec = 0;		// Clear the flag;
		Robot[Seeker2Arg].Status = RecData.Data[2];	// Store the status.

		// Since we are not in the main loop to service the RX EEPROM write
		// we need to do it here. Thus we will write the RX data, increment 
		// the EEPROM address, and clear the flag.
		while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
		EEPROMAddress += sizeof(RecData.Data);	
		WriteEEPROMRX = 0;
	}
	else
	{
		Robot[Seeker2Arg].Status = 0xff;	// Mark robot as unresponsive.
	}
	
	// Ask Reader1 if he is there and get his damage info.
	TransData.DestAddLSB = Reader1;			// Sending to Reader1.
	TransData.Data[0] = AreYouThereRobots;	// Send an AreYouThere.
	TransData.Data[1] = Reader1Arg;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
 
	while(XBWriteAPI() == -1);				// Write the API.

	// This is not needed for to final configuration but instead
	// we need a little delay for the Robot Emulator tool to keep up.
	Wait100ms();
	
	// If we got data from the robot then store the status. Else
	// we should temporarily mark the robot as un-reponsive.
	if(XBMessRec)
	{
		XBMessRec = 0;		// Clear the flag;
		Robot[Reader1Arg].Status = RecData.Data[2];	// Store the status.

		// Since we are not in the main loop to service the RX EEPROM write
		// we need to do it here. Thus we will write the RX data, increment 
		// the EEPROM address, and clear the flag.
		while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
		EEPROMAddress += sizeof(RecData.Data);	
		WriteEEPROMRX = 0;
	}
	else
	{
		Robot[Reader1Arg].Status = 0xff;	// Mark robot as unresponsive.
	}

	// Ask Reader2 if he is there and get his damage info.
	TransData.DestAddLSB = Reader2;			// Sending to Reader2.
	TransData.Data[0] = AreYouThereRobots;	// Send an AreYouThere.
	TransData.Data[1] = Reader2Arg;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);

	while(XBWriteAPI() == -1);				// Write the API.

	// This is not needed for to final configuration but instead
	// we need a little delay for the Robot Emulator tool to keep up.	
	Wait100ms();
	
	// If we got data from the robot then store the status. Else
	// we should temporarily mark the robot as un-reponsive.
	if(XBMessRec)
	{
		XBMessRec = 0;		// Clear the flag;
		Robot[Reader2Arg].Status = RecData.Data[2];	// Store the status.

		// Since we are not in the main loop to service the RX EEPROM write
		// we need to do it here. Thus we will write the RX data, increment 
		// the EEPROM address, and clear the flag.
		while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
		EEPROMAddress += sizeof(RecData.Data);	
		WriteEEPROMRX = 0;
	}
	else
	{
		Robot[Reader2Arg].Status = 0xff;	// Mark robot as unresponsive.
	}

	// -------------------------------------------------------------------------
	// Now go back to the non-responsive robots and determine if they have lost
	// Satellite COMM or if they are dead. We do this by asking robots that have
	// good COMM if they can speak to the unresponsive robot. If the un-responsive
	// robot does not respond to the robot with good COMM then we assume the
	// un-responsive robot is dead. The robot with good COMM will report the
	// status of the un-responsive robot as dead if there is no response.
	// -------------------------------------------------------------------------
	// If Seeker1 is unresponsive.
	if(Robot[Seeker1Arg].Status == 0xff)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Make sure Seeker2 can both talk to us and talk to other robots.
		if(Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoMove)
		{
			// Ask Seeker2 if he can talk to Seeker1
			TransData.DestAddLSB = Seeker2;	// Destination add = Seeker2
			TransData.Data[0] = Talk;		// Command = talk.
			TransData.Data[1] = Seeker2Arg;	// Arguements.
			TransData.Data[2] = Seeker1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.	

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;

			// Store the status
			Robot[Seeker1Arg].Status = RecData.Data[3];
			
		}
		// Make sure Reader1 can both talk to us and talk to other robots.
		else if(Robot[Reader1Arg].Status == OK || Robot[Reader1Arg].Status == NoMove)
		{
			// Ask Reader1 if he can talk to Seeker1
			TransData.DestAddLSB = Reader1;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Reader1Arg;
			TransData.Data[2] = Seeker1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);

			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;	

			// Store the status
			Robot[Seeker1Arg].Status = RecData.Data[3];
		}
		// Make sure Reader2 can both talk to us and talk to other robots.
		else if(Robot[Reader2Arg].Status == OK || Robot[Reader2Arg].Status == NoMove)
		{
			// Ask Reader2 if he can talk to Seeker1
			TransData.DestAddLSB = Reader2;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Reader2Arg;
			TransData.Data[2] = Seeker1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);

			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.	

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);		
			WriteEEPROMRX = 0;

			// Store the status
			Robot[Seeker1Arg].Status = RecData.Data[3];
		}
	}

	// If Seeker2 is unresponsive.
	if(Robot[Seeker2Arg].Status == 0xff)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Make sure Reader1 can both talk to us and talk to other robots.
		if(Robot[Reader1Arg].Status == OK || Robot[Reader1Arg].Status == NoMove)
		{
			// Ask Reader1 if he can talk to Seeker2
			TransData.DestAddLSB = Reader1;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Reader1Arg;
			TransData.Data[2] = Seeker2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;		

			// Store the status
			Robot[Seeker2Arg].Status = RecData.Data[3];
		}
		// Make sure Reader2 can both talk to us and talk to other robots.
		else if(Robot[Reader2Arg].Status == OK || Robot[Reader2Arg].Status == NoMove)
		{
			// Ask Reader2 if he can talk to Seeker2
			TransData.DestAddLSB = Reader2;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Reader2Arg;
			TransData.Data[2] = Seeker2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);		
			WriteEEPROMRX = 0;	

			// Store the status
			Robot[Seeker2Arg].Status = RecData.Data[3];
		}
		// Make sure Seeker1 can both talk to us and talk to other robots.
		else if(Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoMove)
		{
			// Ask Seeker1 if he can talk to Seeker2
			TransData.DestAddLSB = Seeker1;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Seeker1Arg;
			TransData.Data[2] = Seeker2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);		
			WriteEEPROMRX = 0;	

			// Store the status
			Robot[Seeker2Arg].Status = RecData.Data[3];
			
		}
	}

	// If Reader1 is unresponsive.
	if(Robot[Reader1Arg].Status == 0xff)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Make sure Reader2 can both talk to us and talk to other robots.
		if(Robot[Reader2Arg].Status == OK || Robot[Reader2Arg].Status == NoMove)
		{
			// Ask Reader2 if he can talk to Reader1
			TransData.DestAddLSB = Reader2;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Reader2Arg;
			TransData.Data[2] = Reader1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.	

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);		
			WriteEEPROMRX = 0;

			// Store the status
			Robot[Reader1Arg].Status = RecData.Data[3];
		}
		// Make sure Seeker1 can both talk to us and talk to other robots.
		else if(Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoMove)
		{
			// Ask Seeker1 if he can talk to Reader1
			TransData.DestAddLSB = Seeker1;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Seeker1Arg;
			TransData.Data[2] = Reader1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;		

			// Store the status
			Robot[Reader1Arg].Status = RecData.Data[3];
			
		}
		// Make sure Seeker2 can both talk to us and talk to other robots.
		else if(Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoMove)
		{
			// Ask Seeker2 if he can talk to Reader1
			TransData.DestAddLSB = Seeker2;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Seeker2Arg;
			TransData.Data[2] = Reader1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.	

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;	

			// Store the status
			Robot[Reader1Arg].Status = RecData.Data[3];
		}
	}

	// If Reader2 is unresponsive.
	if(Robot[Reader2Arg].Status == 0xff)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Make sure Seeker1 can both talk to us and talk to other robots.
		if(Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoMove)
		{
			// Ask Seeker1 if he can talk to Reader2
			TransData.DestAddLSB = Seeker1;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Seeker1Arg;
			TransData.Data[2] = Reader2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.	

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;	

			// Store the status
			Robot[Reader2Arg].Status = RecData.Data[3];
			
		}
		// Make sure Seeker2 can both talk to us and talk to other robots.
		else if(Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoMove)
		{
			// Ask Seeker2 if he can talk to Reader2
			TransData.DestAddLSB = Seeker2;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Seeker2Arg;
			TransData.Data[2] = Reader2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.	

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;	

			// Store the status
			Robot[Reader2Arg].Status = RecData.Data[3];
		}
		// Make sure Reader1 can both talk to us and talk to other robots.
		else if(Robot[Reader1Arg].Status == OK || Robot[Reader1Arg].Status == NoMove)
		{
			// Ask Reader1 if he can talk to Reader2
			TransData.DestAddLSB = Reader1;
			TransData.Data[0] = Talk;
			TransData.Data[1] = Reader1Arg;
			TransData.Data[2] = Reader2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
			while(XBMessRec == 0);			// Wait for response.
			XBMessRec = 0;					// Clear the flag.

			// Since we are not in the main loop to service the RX EEPROM write
			// we need to do it here. Thus we will write the RX data, increment 
			// the EEPROM address, and clear the flag.
			while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
			EEPROMAddress += sizeof(RecData.Data);	
			WriteEEPROMRX = 0;		

			// Store the status
			Robot[Reader2Arg].Status = RecData.Data[3];
		}
	}

	// --------------------------------------------------------------------
	// Tell all robots the status of all other robots. Relay via other
	// robots if needed.
	// --------------------------------------------------------------------
	// First send data to all robots that have satellite COMMs.
	if(Robot[Seeker1Arg].Status == OK || 
	   Robot[Seeker1Arg].Status == NoMove || 
	   Robot[Seeker1Arg].Status == NoLocalCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Send the status of the other robots one command at a time.
		TransData.DestAddLSB = Seeker1;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Seeker1Arg;
		TransData.Data[2] = Seeker2Arg;
		TransData.Data[3] = Robot[Seeker2Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Seeker1;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Seeker1Arg;
		TransData.Data[2] = Reader1Arg;
		TransData.Data[3] = Robot[Reader1Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Seeker1;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Seeker1Arg;
		TransData.Data[2] = Reader2Arg;
		TransData.Data[3] = Robot[Reader2Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.
	}
	if(Robot[Seeker2Arg].Status == OK || 
	   Robot[Seeker2Arg].Status == NoMove || 
	   Robot[Seeker2Arg].Status == NoLocalCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Send the status of the other robots one command at a time.
		TransData.DestAddLSB = Seeker2;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Seeker2Arg;
		TransData.Data[2] = Seeker1Arg;
		TransData.Data[3] = Robot[Seeker1Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Seeker2;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Seeker2Arg;
		TransData.Data[2] = Reader1Arg;
		TransData.Data[3] = Robot[Reader1Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Seeker2;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Seeker2Arg;
		TransData.Data[2] = Reader2Arg;
		TransData.Data[3] = Robot[Reader2Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.
	}
	if(Robot[Reader1Arg].Status == OK || 
	   Robot[Reader1Arg].Status == NoMove ||
	   Robot[Reader1Arg].Status == NoLocalCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();
	
		// Send the status of the other robots one command at a time.
		TransData.DestAddLSB = Reader1;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Reader1Arg;
		TransData.Data[2] = Seeker1Arg;
		TransData.Data[3] = Robot[Seeker1Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Reader1;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Reader1Arg;
		TransData.Data[2] = Seeker2Arg;
		TransData.Data[3] = Robot[Seeker2Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Reader1;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Reader1Arg;
		TransData.Data[2] = Reader2Arg;
		TransData.Data[3] = Robot[Reader2Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.
	}
	if(Robot[Reader2Arg].Status == OK || 
	   Robot[Reader2Arg].Status == NoMove || 
	   Robot[Reader2Arg].Status == NoLocalCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Send the status of the other robots one command at a time.
		TransData.DestAddLSB = Reader2;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Reader2Arg;
		TransData.Data[2] = Seeker1Arg;
		TransData.Data[3] = Robot[Seeker1Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Reader2;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Reader2Arg;
		TransData.Data[2] = Seeker2Arg;
		TransData.Data[3] = Robot[Seeker2Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.

		Wait100ms();
		TransData.DestAddLSB = Reader2;
		TransData.Data[0] = MRDamage;
		TransData.Data[1] = Reader2Arg;
		TransData.Data[2] = Reader1Arg;
		TransData.Data[3] = Robot[Reader1Arg].Status;

		// Write the Transmitted data to the EEPROM.
		while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
		EEPROMAddress += sizeof(TransData.Data);
 
		while(XBWriteAPI() == -1);		// Write the API.
	}

	// Now we go back and relay damage to robots that have no satellite COMM.
	// We go round robin until we find the first robot with both sat COMM and
	// local COMM and send a message to that robot to relay the damage.
	if(Robot[Seeker1Arg].Status == NoSatCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Determine which robot should relay the data and command
		// the robot to do so.
		if(Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Seeker2;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Seeker2Arg;
			TransData.Data[2] = Seeker1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Reader1Arg].Status == OK || Robot[Reader1Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Reader1;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Reader1Arg;
			TransData.Data[2] = Seeker1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Reader2Arg].Status == OK || Robot[Reader2Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Reader2;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Reader2Arg;
			TransData.Data[2] = Seeker1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
	}

	if(Robot[Seeker2Arg].Status == NoSatCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Determine which robot should relay the data and command
		// the robot to do so.
		if(Robot[Reader1Arg].Status == OK || Robot[Reader1Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Reader1;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Reader1Arg;
			TransData.Data[2] = Seeker2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Reader2Arg].Status == OK || Robot[Reader2Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Reader2;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Reader2Arg;
			TransData.Data[2] = Seeker2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Seeker1;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Seeker1Arg;
			TransData.Data[2] = Seeker2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
	}

	if(Robot[Reader1Arg].Status == NoSatCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Determine which robot should relay the data and command
		// the robot to do so.
		if(Robot[Reader2Arg].Status == OK || Robot[Reader2Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Reader2;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Reader2Arg;
			TransData.Data[2] = Reader1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Seeker1;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Seeker1Arg;
			TransData.Data[2] = Reader1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Seeker2;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Seeker2Arg;
			TransData.Data[2] = Reader1Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
	}

	if(Robot[Reader2Arg].Status == NoSatCOM)
	{
		// This is not needed for to final configuration but instead
		// we need a little delay for the Robot Emulator tool to keep up.
		Wait100ms();

		// Determine which robot should relay the data and command
		// the robot to do so.
		if(Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Seeker1;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Seeker1Arg;
			TransData.Data[2] = Reader2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Seeker2;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Seeker2Arg;
			TransData.Data[2] = Reader2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}
		else if(Robot[Reader1Arg].Status == OK || Robot[Reader1Arg].Status == NoMove)
		{
			TransData.DestAddLSB = Reader1;
			TransData.Data[0] = RelayDamage;
			TransData.Data[1] = Reader1Arg;
			TransData.Data[2] = Reader2Arg;

			// Write the Transmitted data to the EEPROM.
			while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
			EEPROMAddress += sizeof(TransData.Data);
 
			while(XBWriteAPI() == -1);		// Write the API.
		}	
	}

	// -----------------------------------------------------------
	// Tell EarthSat the status of all robots.
	// -----------------------------------------------------------
	TransData.DestAddLSB = EarthSat;
	TransData.Data[0] = MDamage;
	TransData.Data[1] = Seeker1Arg;
	TransData.Data[2] = Robot[Seeker1Arg].Status;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
 
	while(XBWriteAPI() == -1);			// Write the API.

	TransData.DestAddLSB = EarthSat;
	TransData.Data[0] = MDamage;
	TransData.Data[1] = Seeker2Arg;
	TransData.Data[2] = Robot[Seeker2Arg].Status;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
 
	while(XBWriteAPI() == -1);			// Write the API.
	
	TransData.DestAddLSB = EarthSat;
	TransData.Data[0] = MDamage;
	TransData.Data[1] = Reader1Arg;
	TransData.Data[2] = Robot[Reader1Arg].Status;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
 
	while(XBWriteAPI() == -1);			// Write the API.
	
	TransData.DestAddLSB = EarthSat;
	TransData.Data[0] = MDamage;
	TransData.Data[1] = Reader2Arg;
	TransData.Data[2] = Robot[Reader2Arg].Status;

	// Write the Transmitted data to the EEPROM.
	while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
	EEPROMAddress += sizeof(TransData.Data);
 
	while(XBWriteAPI() == -1);			// Write the API.

	// Turn on an LED to indicate that robot status is complete.
	LEDOn(LED1);
}

// This is the main loop for the system. Here we will first
// initialize the dsPIC and its IO and then fall into process
// loop which will process message flags set by interrupts.
int main()
{
	int counter1,counter2,counter3;
	unsigned char buffer[8];
	int size;
	int index;
	unsigned char array[64];
	unsigned char XArg;
	unsigned int EEPROMRobotAddress;

	unsigned char RelayUpdateLocSeeker1 = 0;
	unsigned char RelayUpdateLocSeeker2 = 0;
	unsigned char RelayUpdateLocReader1 = 0;
	unsigned char RelayUpdateLocReader2 = 0;
	unsigned char UpdateLocSeeker1 = 0;
	unsigned char UpdateLocSeeker2 = 0;
	unsigned char UpdateLocReader1 = 0;
	unsigned char UpdateLocReader2 = 0;
	unsigned char RelayUpdateLocStateS1 = 0,RelayUpdateLocStateS2 = 0;
	unsigned char RelayUpdateLocStateR1 = 0,RelayUpdateLocStateR2 = 0;
	unsigned char UpdateLocStateS1 = 0,UpdateLocStateS2 = 0;
	unsigned char UpdateLocStateR1 = 0,UpdateLocStateR2 = 0;

	unsigned char UpdateBallThere = 0;
	unsigned char UpdateBTState = 0;
	unsigned char UpdateBTIndex = 0;

	unsigned char UpdateBallData = 0;
	unsigned char UpdateBDState = 0;
	unsigned char UpdateBDIndex = 0;

	unsigned char GoFlag = 0;
	unsigned char GoState = 0;
	unsigned char GoIndex = 0;

	unsigned char StopFlag = 0;
	unsigned char StopRobot = 0;
	unsigned char StopState = 0;
	unsigned char StopIndex = 0;

	unsigned char ResumeFlag = 0;
	unsigned char ResumeRobot = 0;
	unsigned char ResumeState = 0;
	unsigned char ResumeIndex = 0;

	char string[64];

	Init();

	// Main loop will process messages flagged from
	// the ISRs. We could just do interrupt polling
	// and it would be the same thing but this allows
	// us to do work in the ISR if we so wish to.
	while(1)
	{
		// If the flag for the first we receive a mutex has been set
		// then check to see if we have erased the EEPROM. If we haven't
		// then erase the EEPROM and release the mutex.
		if(FirstTimeMutex && (FirstTimeMutexDone == 0))
		{
			// Build an array of zeros for clearing the EERPOM.
			for(index = 0;index < 64;index++)	// Clear the array.
				array[index] = 0;

			// Clear the EEPROM.
			for(index = 0;index < 0x100;index++)
				while(WriteEEPROM(index << 6, array, sizeof(array)));

			FirstTimeMutexDone = 1;	// Set the flag so we don't do this again.
			EEPROMAddress = 0;		// Reset the address in case we got data
									// before we got the first mutex.
			ReleaseMtx = 1;			// Release the mutex.
		}

		// If we are both waiting for the Mutex and we have grabbed the mutex.
		if(WaitForMutex && Mutex)
		{
			// ---------------------------------------------------------------
			// When EarthSat wants to tell a Robot to stop then it sends a
			// message to us which leads us here. We will determine if that
			// robot can talk to us and if it can we send the message. If it
			// can't then we relay the message via another robot.
			// ---------------------------------------------------------------
			if(ResumeFlag)
			{
				// State machine will build up a TX struct, write it to the EEPROM,
				// and then write to the XBee module.
				switch(ResumeState)
				{
					// First check the robot to determine if it can talk to us. 
					// If it can then build up the TX struct and try to write it 
					// to the EEPROM. If we fail at then then break out to the 
					// main loop which will find its way back here to try again 
					// later. If we succeed then try to write to the XBee module.
					case 0:	// Determine if we can talk to the robot.
							// If not then determine who to send the relay message to and
							// build up a TX struct to send the relay message.
							if(Robot[ResumeRobot].Status == NoSatCOM || Robot[ResumeRobot].Status == Dead)
							{
								ResumeIndex = ResumeRobot;	
	
								// This while loop will loop until we find a robot that is able
								// to talk to both us and other robots.
								while(Robot[ResumeIndex].Status != OK && Robot[ResumeIndex].Status != NoMove)
								{
									ResumeIndex++;		// Increment the index.
									if(ResumeIndex == 4)	// If we reach 4 then we need to roll over.
										ResumeIndex = 0;
								}

								// Fill in the TransData struct.
								TransData.DestAddLSB = ResumeIndex+3;	// Robot are we sending to.			
								TransData.Data[0] = RelayResume;		// The Stop message.
								TransData.Data[1] = ResumeIndex;		// Robot we are sending to.
								TransData.Data[2] = ResumeRobot;		// Robot to Relay Stop to.
							}

							// Else we can talk to the robot directly so build up the TX struct
							// to do that.
							else
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = ResumeRobot+3;	// Robot are we sending to.			
								TransData.Data[0] = Resume;			// The Stop message.
								TransData.Data[1] = ResumeRobot;		// Robot we are sending to.
							}

							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								ResumeState = 1;								// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
								{
									ResumeState = 0;	// If we succeed then reset the state machine.
									ResumeFlag = 0;	// Clear the flag that got us here.
								}		
							}

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine and increment the index to send to the next robot.
					case 1:	if(XBWriteAPI() == 0)
							{
								ResumeState = 0;	// Reset the state machine.
								ResumeFlag = 0;	// Clear the flag that got us here.
							}											

							break;	
				}
			}

			// ---------------------------------------------------------------
			// When EarthSat wants to tell a Robot to stop then it sends a
			// message to us which leads us here. We will determine if that
			// robot can talk to us and if it can we send the message. If it
			// can't then we relay the message via another robot.
			// ---------------------------------------------------------------
			else if(StopFlag)
			{
				// State machine will build up a TX struct, write it to the EEPROM,
				// and then write to the XBee module.
				switch(StopState)
				{
					// First check the robot to determine if it can talk to us. 
					// If it can then build up the TX struct and try to write it 
					// to the EEPROM. If we fail at then then break out to the 
					// main loop which will find its way back here to try again 
					// later. If we succeed then try to write to the XBee module.
					case 0:	// Determine if we can talk to the robot.
							// If not then determine who to send the relay message to and
							// build up a TX struct to send the relay message.
							if(Robot[StopRobot].Status == NoSatCOM || Robot[StopRobot].Status == Dead)
							{
								StopIndex = StopRobot;	
	
								// This while loop will loop until we find a robot that is able
								// to talk to both us and other robots.
								while(Robot[StopIndex].Status != OK && Robot[StopIndex].Status != NoMove)
								{
									StopIndex++;		// Increment the index.
									if(StopIndex == 4)	// If we reach 4 then we need to roll over.
										StopIndex = 0;
								}

								// Fill in the TransData struct.
								TransData.DestAddLSB = StopIndex+3;	// Robot are we sending to.			
								TransData.Data[0] = RelayStop;		// The Stop message.
								TransData.Data[1] = StopIndex;		// Robot we are sending to.
								TransData.Data[2] = StopRobot;		// Robot to Relay Stop to.
							}

							// Else we can talk to the robot directly so build up the TX struct
							// to do that.
							else
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = StopRobot+3;	// Robot are we sending to.			
								TransData.Data[0] = Stop;			// The Stop message.
								TransData.Data[1] = StopRobot;		// Robot we are sending to.
							}

							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								StopState = 1;								// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
								{
									StopState = 0;	// If we succeed then reset the state machine.
									StopFlag = 0;	// Clear the flag that got us here.
								}		
							}

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine and increment the index to send to the next robot.
					case 1:	if(XBWriteAPI() == 0)
							{
								StopState = 0;	// Reset the state machine.
								StopFlag = 0;	// Clear the flag that got us here.
							}											

							break;	
				}
			}

			// ----------------------------------------------------------------
			// Once we have determined everyones status and everyone knows the
			// the status of everyone else we set a flag to send the Go 
			// command to all the robots which leads us here.
			// ----------------------------------------------------------------
			else if(GoFlag)
			{
				// State machine which will send the Go message to any robot that
				// can talk to us.
				switch(GoState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the robot to determine
					// if it can talk to us. If it can then build up the TX struct 
					// and try to write it to the EEPROM. If we fail at then then 
					// break out to the main loop which will find its way back here 
					// to try again later. If we succeed then try to write to the 
					// XBee module.
					case 0:	if(GoIndex == 4)	// If we are done.
							{
								GoFlag = 0;		// Clear the flag that got us here.
								GoIndex = 0;	// Reset the index.
								break;
							}

							// Check to see that the currently indexed robot can talk to us.
							if(Robot[GoIndex].Status != Dead && Robot[GoIndex].Status != NoSatCOM)
							{
								// If it can fill in the TransData struct.
								TransData.DestAddLSB = GoIndex+3;	// Robot are we sending to.			
								TransData.Data[0] = Go;				// The Go message.
								TransData.Data[1] = GoIndex;		// Robot we are sending to.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									GoState = 1;								// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										GoState = 0;	// If we succeed then reset the state machine.
										GoIndex++;		// Increment index to send to next robot.
									}		
								}
								break;
							}
							
							// Else the robot can't talk to us so just increment the index to
							// try the next robot.
							else
							{
								GoIndex++;		// Increment index to send to next robot.
							}
							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine and increment the index to send to the next robot.
					case 1:	if(XBWriteAPI() == 0)
							{
								GoState = 0;	// Reset the state machine.
								GoIndex++;		// Increment index to send to next robot.
							}											

							break;
				}
			}

			// -----------------------------------------------------------------
			// Here we check the flag to see if we should send any updated
			// IR Data information about the balls
			// -----------------------------------------------------------------
			else if(UpdateBallData)
			{
				// State machine that checks each Ball's relay flag to see if the
				// IR Data should be sent to EarthSat and/or other robots.
				switch(UpdateBDState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the balls one at a time
					// to determine if we should relay its IR Data. If we should then
					// build up the TX struct and try to write it to the EEPROM. If we
					// fail at then then break out to the main loop which will find its
					// way back here to try again later. If we succeed then try to write
					// to the XBee module.
					case 0:	if(UpdateBDIndex == 8)		// Check that we are done.
							{
								UpdateBallData = 0;		// Clear the flag.
								UpdateBDIndex = 0;		// Clear the index.
								break;
							}

							// Check the relay flag of the next ball to determine if we should
							// relay the ball's IR Data.
							if(Ball[UpdateBDIndex].RelayData)
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = EarthSat;			// Send to EarthSat.
								TransData.Data[0] = MBallData;				// Ball IR DAta.		
								TransData.Data[1] = UpdateBDIndex + 0x41;	// Which ball.
								TransData.Data[2] = Ball[UpdateBDIndex].IRData;	// What data.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDState = 1;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBDState = 2;		// If successful then send to robots.
								}
								break;
							}

							// If we should not relay the status of this ball then increment the index to
							// check the relay flag of the next ball.
							else
								UpdateBDIndex++;	// Increment index to check next ball.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine, clear this ball's relay flag, and increment the index
					// to check the next ball.
					case 1:	if(XBWriteAPI() == 0)
								UpdateBDState = 2;						// Check to see if we send to robots.

							break;

					// Case 2 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball IR data.
					// 1. If the robot that sent the ball IR Data has NoLocalCOM then MarsSat must relay the 
					// ball IR Data to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball IR Data was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball IR Data data.
					case 2:	if( (Robot[Ball[UpdateBDIndex].Robot].Status == NoLocalCOM || Robot[Seeker1Arg].Status == NoLocalCOM) && 
								Ball[UpdateBDIndex].Robot != Seeker1Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker1Arg].Status == NoSatCOM || Robot[Seeker1Arg].Status == Dead)
								{
									UpdateBDState = 4;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Seeker1;				// Sending to Seeker1.
								TransData.Data[0] = IRData;					// Sending IR Data.		
								TransData.Data[1] = Seeker1Arg;				// Sending to Seeker1.
								TransData.Data[2] = UpdateBDIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBDIndex].IRData; // What data.
								TransData.Data[4] = Ball[UpdateBDIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDState = 3;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBDState = 4;	// If successful then check Seeker2.
								}
							}
							else
								UpdateBDState = 4;	// If we don't need to relay then check Seeker2.	

							break;

					
					// Case3 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 3:	if(XBWriteAPI() == 0)
								UpdateBDState = 4;	// If successful then check Seeker2.
							break;

					// Case 4 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball IR data.
					// 1. If the robot that sent the ball IR Data has NoLocalCOM then MarsSat must relay the 
					// ball IR Data to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball IR Data was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball IR data.
					case 4:	if( (Robot[Ball[UpdateBDIndex].Robot].Status == NoLocalCOM || Robot[Seeker2Arg].Status == NoLocalCOM) && 
								Ball[UpdateBDIndex].Robot != Seeker2Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker2Arg].Status == NoSatCOM || Robot[Seeker2Arg].Status == Dead)
								{
									UpdateBDState = 6;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Seeker2;				// Sending to Seeker2.
								TransData.Data[0] = IRData;					// Sending IR Data.		
								TransData.Data[1] = Seeker2Arg;				// Sending to Seeker2.
								TransData.Data[2] = UpdateBDIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBDIndex].IRData; // What data.
								TransData.Data[4] = Ball[UpdateBDIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDState = 5;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBDState = 6;	// If successful then check Reader1.
								}
							}
							else
								UpdateBDState = 6;	// If we don't need to relay then check Reader1.	

							break;

					
					// Case5 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 5:	if(XBWriteAPI() == 0)
								UpdateBDState = 6;	// If successful then check Reader1.
							break;

					// Case 6 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball IR data.
					// 1. If the robot that sent the ball IR Data has NoLocalCOM then MarsSat must relay the 
					// ball IR Data to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball IR Data was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball IR data.
					case 6:	if( (Robot[Ball[UpdateBDIndex].Robot].Status == NoLocalCOM || Robot[Reader1Arg].Status == NoLocalCOM) && 
								Ball[UpdateBDIndex].Robot != Reader1Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Reader1Arg].Status == NoSatCOM || Robot[Reader1Arg].Status == Dead)
								{
									UpdateBDState = 8;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Reader1;				// Sending to Reader1.
								TransData.Data[0] = IRData;					// Sending IR Data.		
								TransData.Data[1] = Reader1Arg;				// Sending to Reader1.
								TransData.Data[2] = UpdateBDIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBDIndex].IRData; // What data.
								TransData.Data[4] = Ball[UpdateBDIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDState = 7;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBDState = 8;	// If successful then check Reader1.
								}
							}
							else
								UpdateBDState = 8;	// If we don't need to relay then check Reader2.	

							break;

					
					// Case7 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 7:	if(XBWriteAPI() == 0)
								UpdateBDState = 8;	// If successful then check Reader2.
							break;

					// Case 8 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball IR data.
					// 1. If the robot that sent the ball IR Data has NoLocalCOM then MarsSat must relay the 
					// ball IR Data to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball IR Data was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball IR data.
					case 8:	if( (Robot[Ball[UpdateBDIndex].Robot].Status == NoLocalCOM || Robot[Reader2Arg].Status == NoLocalCOM) && 
								Ball[UpdateBDIndex].Robot != Reader2Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Reader2Arg].Status == NoSatCOM || Robot[Reader2Arg].Status == Dead)
								{
									UpdateBDState = 0;						// If we can't then check next ball.
									Ball[UpdateBDIndex].RelayData = 0;		// Clear this ball's relay flag.
									UpdateBDIndex++;						// Increment index to check next ball.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Reader2;				// Sending to Reader2.
								TransData.Data[0] = IRData;					// Sending IR Data.		
								TransData.Data[1] = Reader2Arg;				// Sending to Reader2.
								TransData.Data[2] = UpdateBDIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBDIndex].IRData; // What data.
								TransData.Data[4] = Ball[UpdateBDIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDState = 9;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateBDState = 0;						// If we can't then check next ball.
										Ball[UpdateBDIndex].RelayData = 0;		// Clear this ball's relay flag.
										UpdateBDIndex++;						// Increment index to check next ball.
									}
								}
							}
							else
							{
								UpdateBDState = 0;						// If we can't then check next ball.
								Ball[UpdateBDIndex].RelayData = 0;		// Clear this ball's relay flag.
								UpdateBDIndex++;						// Increment index to check next ball.	
							}

							break;

					
					// Case9 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 9:	if(XBWriteAPI() == 0)
							{
								UpdateBDState = 0;						// If we can't then check next ball.
								Ball[UpdateBDIndex].RelayData = 0;		// Clear this ball's relay flag.
								UpdateBDIndex++;						// Increment index to check next ball.
							}
							break;

				}
			}

			// -----------------------------------------------------------------
			// Here we check the flag to see if we should send any updated
			// status information about the balls
			// -----------------------------------------------------------------
			else if(UpdateBallThere)
			{
				// State machine that checks each Ball's relay flag to see if the
				// status should be sent to EarthSat.
				switch(UpdateBTState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the balls one at a time
					// to determine if we should relay its status. If we should then
					// build up the TX struct and try to write it to the EEPROM. If we
					// fail at then then break out to the main loop which will find its
					// way back here to try again later. If we succeed then try to write
					// to the XBee module.
					case 0:	if(UpdateBTIndex == 8)		// Check that we are done.
							{
								UpdateBallThere = 0;	// Clear the flag.
								UpdateBTIndex = 0;		// Clear the index.
								break;
							}

							// Check the relay flag of the next ball to determine if we should
							// relay the ball's status.
							if(Ball[UpdateBTIndex].RelayStatus)
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = EarthSat;			// Send to EarthSat.
								if(Ball[UpdateBTIndex].Status == There)		// Determine if we send found
									TransData.Data[0] = MBallFound;			// or NOT found.
								else
									TransData.Data[0] = MBallNotFound;			
								TransData.Data[1] = UpdateBTIndex + 0x41;	// Which ball.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTState = 1;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBTState = 2;		// If successful then send to robots.
								}
								break;
							}

							// If we should not relay the status of this ball then increment the index to
							// check the relay flag of the next ball.
							else
								UpdateBTIndex++;	// Increment index to check next ball.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine, clear this ball's relay flag, and increment the index
					// to check the next ball.
					case 1:	if(XBWriteAPI() == 0)
								UpdateBTState = 2;						// Check to see if we send to robots.

							break;

					// Case 2 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball status data.
					// 1. If the robot that sent the ball status has NoLocalCOM then MarsSat must relay the 
					// ball status to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball status was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball status data.
					case 2:	if( (Robot[Ball[UpdateBTIndex].Robot].Status == NoLocalCOM || Robot[Seeker1Arg].Status == NoLocalCOM) && 
								Ball[UpdateBTIndex].Robot != Seeker1Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker1Arg].Status == NoSatCOM || Robot[Seeker1Arg].Status == Dead)
								{
									UpdateBTState = 4;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Seeker1;				// Sending to Seeker1.
								if(Ball[UpdateBTIndex].Status == There)		// Determine if we send a found
									TransData.Data[0] = BallFound;			// or NOT found message.
								else
									TransData.Data[0] = BallNOTFound;
								TransData.Data[1] = Seeker1Arg;				// Sending to Seeker1.
								TransData.Data[2] = UpdateBTIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBTIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTState = 3;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBTState = 4;	// If successful then check Reader1.
								}
							}
							else
								UpdateBTState = 4;	// If we don't need to relay then check Seeker2.	

							break;

					
					// Case3 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 3:	if(XBWriteAPI() == 0)
								UpdateBTState = 4;	// If successful then check Seeker2.
							break;

					// Case 4 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball status data.
					// 1. If the robot that sent the ball status has NoLocalCOM then MarsSat must relay the 
					// ball status to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball status was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball status data.
					case 4:	if( (Robot[Ball[UpdateBTIndex].Robot].Status == NoLocalCOM || Robot[Seeker2Arg].Status == NoLocalCOM) && 
								Ball[UpdateBTIndex].Robot != Seeker2Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker2Arg].Status == NoSatCOM || Robot[Seeker2Arg].Status == Dead)
								{
									UpdateBTState = 6;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Seeker2;				// Sending to Seeker2.
								if(Ball[UpdateBTIndex].Status == There)		// Determine if we send a found
									TransData.Data[0] = BallFound;			// or NOT found message.
								else
									TransData.Data[0] = BallNOTFound;
								TransData.Data[1] = Seeker2Arg;				// Sending to Seeker2.
								TransData.Data[2] = UpdateBTIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBTIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTState = 5;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBTState = 6;	// If successful then check Reader1.
								}
							}
							else
								UpdateBTState = 6;	// If we don't need to relay then check Reader1.	

							break;

					
					// Case5 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 5:	if(XBWriteAPI() == 0)
								UpdateBTState = 6;	// If successful then check Reader1.
							break;

					// Case 6 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball status data.
					// 1. If the robot that sent the ball status has NoLocalCOM then MarsSat must relay the 
					// ball status to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball status was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball status data.
					case 6:	if( (Robot[Ball[UpdateBTIndex].Robot].Status == NoLocalCOM || Robot[Reader1Arg].Status == NoLocalCOM) && 
								Ball[UpdateBTIndex].Robot != Reader1Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Reader1Arg].Status == NoSatCOM || Robot[Reader1Arg].Status == Dead)
								{
									UpdateBTState = 8;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Reader1;				// Sending to Reader1.
								if(Ball[UpdateBTIndex].Status == There)		// Determine if we send a found
									TransData.Data[0] = BallFound;			// or NOT found message.
								else
									TransData.Data[0] = BallNOTFound;
								TransData.Data[1] = Reader1Arg;				// Sending to Reader1.
								TransData.Data[2] = UpdateBTIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBTIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTState = 7;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateBTState = 8;	// If successful then check Reader1.
								}
							}
							else
								UpdateBTState = 8;	// If we don't need to relay then check Reader2.	

							break;

					
					// Case7 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 7:	if(XBWriteAPI() == 0)
								UpdateBTState = 8;	// If successful then check Reader2.
							break;

					// Case 8 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the ball status data.
					// 1. If the robot that sent the ball status has NoLocalCOM then MarsSat must relay the 
					// ball status to any robot who can talk to MarsSat. 
					// 2. If the robot that sent the ball status was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the ball status data.
					case 8:	if( (Robot[Ball[UpdateBTIndex].Robot].Status == NoLocalCOM || Robot[Reader2Arg].Status == NoLocalCOM) && 
								Ball[UpdateBTIndex].Robot != Reader2Arg)
							{
								// Check that we can send to the robot.
								if(Robot[Reader2Arg].Status == NoSatCOM || Robot[Reader2Arg].Status == Dead)
								{
									UpdateBTState = 0;						// If we can't then check next ball.
									Ball[UpdateBTIndex].RelayStatus = 0;	// Clear this ball's relay flag.
									UpdateBTIndex++;						// Increment index to check next ball.
									break;
								}

								// Build up TX struct to relay ball status info to robot that needs the info.
								TransData.DestAddLSB = Reader2;				// Sending to Reader2.
								if(Ball[UpdateBTIndex].Status == There)		// Determine if we send a found
									TransData.Data[0] = BallFound;			// or NOT found message.
								else
									TransData.Data[0] = BallNOTFound;
								TransData.Data[1] = Reader2Arg;				// Sending to Reader2.
								TransData.Data[2] = UpdateBTIndex + 0x41;	// Which ball.
								TransData.Data[3] = Ball[UpdateBTIndex].Robot;	// Which robot.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTState = 9;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateBTState = 0;						// If we can't then check next ball.
										Ball[UpdateBTIndex].RelayStatus = 0;	// Clear this ball's relay flag.
										UpdateBTIndex++;						// Increment index to check next ball.
									}
								}
							}
							else
							{
								UpdateBTState = 0;						// If we can't then check next ball.
								Ball[UpdateBTIndex].RelayStatus = 0;	// Clear this ball's relay flag.
								UpdateBTIndex++;						// Increment index to check next ball.	
							}

							break;

					
					// Case9 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 9:	if(XBWriteAPI() == 0)
							{
								UpdateBTState = 0;						// If we can't then check next ball.
								Ball[UpdateBTIndex].RelayStatus = 0;	// Clear this ball's relay flag.
								UpdateBTIndex++;						// Increment index to check next ball.
							}
							break;

				}
			}

			// -----------------------------------------------------------------
			// Check to see who has been flagged to update their location.
			// -----------------------------------------------------------------
			else if(UpdateLocSeeker1)
			{
				// Here we have a simple state machine that will tell Earth the new
				// location of the robot and then determine if we should relay the
				// location data to one of the robots. This would occur if the robots
				// were unable to talk to each other.
				switch(UpdateLocStateS1)
				{
					// Case0 will build up the TX struct and attempt to write the TX data to the EEPROM.
					// If we can't write to EEPROM, then we break out to the main loop without changing
					// any flags which means the main loop will find its way back here so we can try
					// again later.
					case 0:	TransData.DestAddLSB = EarthSat;		// Fill in the TransData struct
							TransData.Data[0] = MLocation;			// to send the location data to EarthSat.
							TransData.Data[1] = Seeker1Arg;			// This says Seeker1's Location data.
							TransData.Data[2] = Robot[Seeker1Arg].CurrentLocation;	// The actual location data.

							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								UpdateLocStateS1 = 1;						// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
									UpdateLocStateS1 = 2;	// If successful then check if we should send to robots.
							}
							break;

					// Case1 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statement to clear flags so that we don't get here anymore.
					case 1:	if(XBWriteAPI() == 0)
								UpdateLocStateS1 = 2;	// If successful then check Seeker2.

							break;

					// Case 2 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 2:	if(Robot[Seeker1Arg].Status == NoLocalCOM || Robot[Seeker2Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker2Arg].Status == NoSatCOM || Robot[Seeker2Arg].Status == Dead)
								{
									UpdateLocStateS1 = 4;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Seeker2;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay location message.
								TransData.Data[1] = Seeker2Arg;			// Sending to Seeker2.
								TransData.Data[2] = Seeker1Arg;			// Location of Seeker1.
								TransData.Data[3] = Robot[Seeker1Arg].CurrentLocation;	// Actual Location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateS1 = 3;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateS1 = 4;	// If successful then check Reader1.
								}
							}
							else
								UpdateLocStateS1 = 4;	// If we don't need to relay then check Reader1.	

							break;

					
					// Case3 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 3:	if(XBWriteAPI() == 0)
								UpdateLocStateS1 = 4;	// If successful then check Reader1.
							break;

					// Case 4 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 4:	if(Robot[Seeker1Arg].Status == NoLocalCOM || Robot[Reader1Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Reader1Arg].Status == NoSatCOM || Robot[Reader1Arg].Status == Dead)
								{
									UpdateLocStateS1 = 6;	// If we can't then move on to the next.
									break;
								}
							
								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Reader1;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.
								TransData.Data[1] = Reader1Arg;			// Sending to Reader1.
								TransData.Data[2] = Seeker1Arg;			// Location of Seeker1.
								TransData.Data[3] = Robot[Seeker1Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateS1 = 5;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateS1 = 6;	// If successful then check Reader2.
								}
							}
							else
								UpdateLocStateS1 = 6;	// If we don't need to relay then check Reader2.	

							break;

					// Case5 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 5:	if(XBWriteAPI() == 0)
								UpdateLocStateS1 = 6;	// If successful then check Reader2.
							break;

					// Case 6 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 6:	if(Robot[Seeker1Arg].Status == NoLocalCOM || Robot[Reader2Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Reader2Arg].Status == NoSatCOM || Robot[Reader2Arg].Status == Dead)
								{
									UpdateLocStateS1 = 0;	// If we can't then move on to the next.
									UpdateLocSeeker1 = 0;	// Clear flag so we don't go here again.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Reader2;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.
								TransData.Data[1] = Reader2Arg;			// Sending to Reader2.
								TransData.Data[2] = Seeker1Arg;			// Location of Seeker1.
								TransData.Data[3] = Robot[Seeker1Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateS1 = 7;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateLocStateS1 = 0;	// If successful then done.
										UpdateLocSeeker1 = 0;	// Clear flag.
									}
								}
							}
							else
							{
								UpdateLocStateS1 = 0;	// If we don't need to relay then done.
								UpdateLocSeeker1 = 0;	// Clear flag.
							}	

							break;

					// Case7 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 7:	if(XBWriteAPI() == 0)
							{
								UpdateLocStateS1 = 0;	// If successful then done.
								UpdateLocSeeker1 = 0;	// Clear flag.
							}
							break;				 
				}
			}
			else if(UpdateLocSeeker2)
			{
				// Here we have a simple state machine that will tell Earth the new
				// location of the robot and then determine if we should relay the
				// location data to one of the robots. This would occur if the robots
				// were unable to talk to each other.
				switch(UpdateLocStateS2)
				{
					// Case0 will build up the TX struct and attempt to write the TX data to the EEPROM.
					// If we can't write to EEPROM, then we break out to the main loop without changing
					// any flags which means the main loop will find its way back here so we can try
					// again later.
					case 0:	TransData.DestAddLSB = EarthSat;		// Fill in the TransData struct
							TransData.Data[0] = MLocation;			// to send the location data to EarthSat.
							TransData.Data[1] = Seeker2Arg;			// This says Seeker2's Location data.
							TransData.Data[2] = Robot[Seeker2Arg].CurrentLocation;	// The actual location data.

							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								UpdateLocStateS2 = 1;						// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
									UpdateLocStateS2 = 2;	// If successful then check if we should send to robots.
							}
							break;

					// Case1 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 1:	if(XBWriteAPI() == 0)
								UpdateLocStateS2 = 2;	// If successful then reset the state machine.
							break;

					// Case 2 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 2:	if(Robot[Seeker2Arg].Status == NoLocalCOM || Robot[Seeker1Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker1Arg].Status == NoSatCOM || Robot[Seeker1Arg].Status == Dead)
								{
									UpdateLocStateS2 = 4;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Seeker1;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay location message.
								TransData.Data[1] = Seeker1Arg;			// Sending to Seeker1.
								TransData.Data[2] = Seeker2Arg;			// Location of Seeker2.
								TransData.Data[3] = Robot[Seeker2Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateS2 = 3;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateS2 = 4;	// If successful then check Reader1.
								}
							}
							else
								UpdateLocStateS2 = 4;	// If we don't need to relay then check Reader1.	

							break;


					// Case3 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 3:	if(XBWriteAPI() == 0)
								UpdateLocStateS2 = 4;	// If successful then check Reader1.
							break;

					// Case 4 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 4:	if(Robot[Seeker2Arg].Status == NoLocalCOM || Robot[Reader1Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Reader1Arg].Status == NoSatCOM || Robot[Reader1Arg].Status == Dead)
								{
									UpdateLocStateS2 = 6;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.							
								TransData.DestAddLSB = Reader1;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.
								TransData.Data[1] = Reader1Arg;			// Sending to Reader1.
								TransData.Data[2] = Seeker2Arg;			// Location of Seeker2.
								TransData.Data[3] = Robot[Seeker2Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateS2 = 5;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateS2 = 6;	// If successful then check Reader2.
								}
							}
							else
								UpdateLocStateS2 = 6;	// If we don't need to relay then check Reader2.	

							break;

					// Case5 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 5:	if(XBWriteAPI() == 0)
								UpdateLocStateS2 = 6;	// If successful then check Reader2.
							break;

					// Case 6 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 6:	if(Robot[Seeker2Arg].Status == NoLocalCOM || Robot[Reader2Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Reader2Arg].Status == NoSatCOM || Robot[Reader2Arg].Status == Dead)
								{
									UpdateLocStateS2 = 0;	// If we can't then move on to the next.
									UpdateLocSeeker2 = 0;	// Clear flag so that we don't go here again.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Reader2;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.
								TransData.Data[1] = Reader2Arg;			// Sending to Reader2.
								TransData.Data[2] = Seeker2Arg;			// Location of Seeker2.
								TransData.Data[3] = Robot[Seeker2Arg].CurrentLocation;	// Actual Location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateS2 = 7;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateLocStateS2 = 0;	// If successful then done.
										UpdateLocSeeker2 = 0;	// Clear flag.
									}
								}
							}
							else
							{
								UpdateLocStateS2 = 0;	// If we don't need to relay then done.
								UpdateLocSeeker2 = 0;	// Clear flag.
							}	

							break;

					// Case7 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 7:	if(XBWriteAPI() == 0)
							{
								UpdateLocStateS2 = 0;	// If successful then done.
								UpdateLocSeeker2 = 0;	// Clear flag.
							}
							break;			
				}
			}
			else if(UpdateLocReader1)
			{
				// Here we have a simple state machine that will tell Earth the new
				// location of the robot and then determine if we should relay the
				// location data to one of the robots. This would occur if the robots
				// were unable to talk to each other.
				switch(UpdateLocStateR1)
				{
					// Case0 will build up the TX struct and attempt to write the TX data to the EEPROM.
					// If we can't write to EEPROM, then we break out to the main loop without changing
					// any flags which means the main loop will find its way back here so we can try
					// again later.
					case 0:	TransData.DestAddLSB = EarthSat;		// Fill in the TransData struct
							TransData.Data[0] = MLocation;			// to send the location data to EarthSat.
							TransData.Data[1] = Reader1Arg;			// This says Reader1's Location data.
							TransData.Data[2] = Robot[Reader1Arg].CurrentLocation;	// The actual location data.

							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								UpdateLocStateR1 = 1;						// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
									UpdateLocStateR1 = 2;	// If successful then check if we should send to robots.
							}
							break;

					// Case1 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 1:	if(XBWriteAPI() == 0)
								UpdateLocStateR1 = 2;	// If successful then reset the state machine.
							break;

					// Case 2 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 2:	if(Robot[Reader1Arg].Status == NoLocalCOM || Robot[Seeker1Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker1Arg].Status == NoSatCOM || Robot[Seeker1Arg].Status == Dead)
								{
									UpdateLocStateR1 = 4;	// If we can't then move on to the next.
									break;
								}

								// Build up the TX struct to send the relayed message the robot.
								TransData.DestAddLSB = Seeker1;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.			
								TransData.Data[1] = Seeker1Arg;			// Sending to Seeker1.
								TransData.Data[2] = Reader1Arg;			// Location of Reader1.
								TransData.Data[3] = Robot[Reader1Arg].CurrentLocation;	// Actual Location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateR1 = 3;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateR1 = 4;	// If successful then check Seeker2.
								}
							}
							else
								UpdateLocStateR1 = 4;	// If we don't need to relay then check Seeker2.	

							break;
					
					// Case3 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 3:	if(XBWriteAPI() == 0)
								UpdateLocStateR1 = 4;	// If successful then check Reader1.
							break;

					// Case 4 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 4:	if(Robot[Reader1Arg].Status == NoLocalCOM || Robot[Seeker2Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker2Arg].Status == NoSatCOM || Robot[Seeker2Arg].Status == Dead)
								{
									UpdateLocStateR1 = 6;	// If we can't then move on to the next.
									break;
								}
							
								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Seeker2;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay location message.		
								TransData.Data[1] = Seeker2Arg;			// Sending to Seeker2.
								TransData.Data[2] = Reader1Arg;			// Location of Reader1.
								TransData.Data[3] = Robot[Reader1Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateR1 = 5;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateR1 = 6;	// If successful then check Reader2.
								}
							}
							else
								UpdateLocStateR1 = 6;	// If we don't need to relay then check Reader2.	

							break;

					// Case5 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 5:	if(XBWriteAPI() == 0)
								UpdateLocStateR1 = 6;	// If successful then check Reader2.
							break;

					// Case 6 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 6:	if(Robot[Reader1Arg].Status == NoLocalCOM || Robot[Reader2Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Reader2Arg].Status == NoSatCOM || Robot[Reader2Arg].Status == Dead)
								{
									UpdateLocStateR1 = 0;	// If we can't then move on to the next.
									UpdateLocReader1 = 0;	// Clear flag so we don't get here again.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Reader2;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.		
								TransData.Data[1] = Reader2Arg;			// Sending to Reader2.
								TransData.Data[2] = Reader1Arg;			// Location of Reader1.
								TransData.Data[3] = Robot[Reader1Arg].CurrentLocation;	// Acutal location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateR1 = 7;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateLocStateR1 = 0;	// If successful then done.
										UpdateLocReader1 = 0;	// Clear flag.
									}
								}
							}
							else
							{
								UpdateLocStateR1 = 0;	// If we don't need to relay then done.
								UpdateLocReader1 = 0;	// Clear flag.
							}	

							break;

					// Case7 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 7:	if(XBWriteAPI() == 0)
							{
								UpdateLocStateR1 = 0;	// If successful then done.
								UpdateLocReader1 = 0;	// Clear flag.
							}
							break;			
				}
			}
			else if(UpdateLocReader2)
			{
				// Here we have a simple state machine that will tell Earth the new
				// location of the robot and then determine if we should relay the
				// location data to one of the robots. This would occur if the robots
				// were unable to talk to each other.
				switch(UpdateLocStateR2)
				{
					// Case0 will build up the TX struct and attempt to write the TX data to the EEPROM.
					// If we can't write to EEPROM, then we break out to the main loop without changing
					// any flags which means the main loop will find its way back here so we can try
					// again later.
					case 0:	TransData.DestAddLSB = EarthSat;		// Fill in the TransData struct
							TransData.Data[0] = MLocation;			// to send the location data to EarthSat.
							TransData.Data[1] = Reader2Arg;			// This says Reader2's Location data.
							TransData.Data[2] = Robot[Reader2Arg].CurrentLocation;	// The actual location data.

							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								UpdateLocStateR2 = 1;						// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
									UpdateLocStateR2 = 2;	// If successful then check if we should send to robots.
							}
							break;

					// Case1 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 1:	if(XBWriteAPI() == 0)
								UpdateLocStateR2 = 2;	// If successful then reset the state machine.
							break;

					// Case 2 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 2:	if(Robot[Reader2Arg].Status == NoLocalCOM || Robot[Seeker1Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker1Arg].Status == NoSatCOM || Robot[Seeker1Arg].Status == Dead)
								{
									UpdateLocStateR2 = 4;	// If we can't then move on to the next.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Seeker1;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay location message.
								TransData.Data[1] = Seeker1Arg;			// Sending to Seeker1.
								TransData.Data[2] = Reader2Arg;			// Location of Reader2.
								TransData.Data[3] = Robot[Reader2Arg].CurrentLocation;	// Actual Location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateR2 = 3;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateR2 = 4;	// If successful then check Seeker2.
								}
							}
							else
								UpdateLocStateR2 = 4;	// If we don't need to relay then check Seeker2.	

							break;

					// Case3 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 3:	if(XBWriteAPI() == 0)
								UpdateLocStateR2 = 4;	// If successful then check Reader1.
							break;

					// Case 4 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 4:	if(Robot[Reader2Arg].Status == NoLocalCOM || Robot[Seeker2Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Seeker2Arg].Status == NoSatCOM || Robot[Seeker2Arg].Status == Dead)
								{
									UpdateLocStateR2 = 6;	// If we can't then move on to the next.
									break;
								}
							
								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Seeker2;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay location message.
								TransData.Data[1] = Seeker2Arg;			// Sending to Seeker2.
								TransData.Data[2] = Reader2Arg;			// Location of Reader2.
								TransData.Data[3] = Robot[Reader2Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateR2 = 5;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
										UpdateLocStateR2 = 6;	// If successful then check Reader1.
								}
							}
							else
								UpdateLocStateR2 = 6;	// If we don't need to relay then check Reader1.	

							break;

					// Case5 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 5:	if(XBWriteAPI() == 0)
								UpdateLocStateR2 = 6;	// If successful then check Reader2.
							break;

					// Case 6 is the same as all the even cases. There are two cases in which MarsSat needs to
					// relay the location data.
					// 1. If the robot with a location change has NoLocalCOM then MarsSat must relay the 
					// location to any robot who can talk to MarsSat. 
					// 2. If the robot with a location change was unable to talk to other robots because the 
					// other robot has NoLocalCOM then we need to relay the location change.
					case 6:	if(Robot[Reader2Arg].Status == NoLocalCOM || Robot[Reader1Arg].Status == NoLocalCOM)
							{
								// Check that we can send to the robot.
								if(Robot[Reader1Arg].Status == NoSatCOM || Robot[Reader1Arg].Status == Dead)
								{
									UpdateLocStateR2 = 0;	// If we can't then move on to the next.
									UpdateLocReader2 = 0;	// Clear flag so we don't get here again.
									break;
								}

								// Build up TX struct to relay location to robot that needs the info.
								TransData.DestAddLSB = Reader1;			// Fill in the TransData struct.
								TransData.Data[0] = MRLocation;			// Relay Location message.
								TransData.Data[1] = Reader1Arg;			// Sending to Reader1.
								TransData.Data[2] = Reader2Arg;			// Location of Reader2.
								TransData.Data[3] = Robot[Reader2Arg].CurrentLocation;	// Actual location data.

								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateLocStateR2 = 7;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateLocStateR2 = 0;	// If successful then done.
										UpdateLocReader2 = 0;	// Clear flag.
									}
								}
							}
							else
							{
								UpdateLocStateR2 = 0;	// If we don't need to relay then done.
								UpdateLocReader2 = 0;	// Clear flag.
							}	

							break;

					// Case7 is the same as all the odd cases (ie Case3, Case5, etc). It will constantly
					// try to send the TX struct to the XBee module. We can count on the fact that the
					// TX struct will not change because we have the mutex. Thus, there is no need to
					// build up a new one. If we are successful then we move to the next state. The last
					// case will have an additional statment to clear flags so that we don't get here anymore.
					case 7:	if(XBWriteAPI() == 0)
							{
								UpdateLocStateR2 = 0;	// If successful then done.
								UpdateLocReader2 = 0;	// Clear flag.
							}
							break;			
				}
			}

			// Else we have no more flags to service so we will release the mutex.
			else
			{
				WaitForMutex = 0;	// Clear the WaitForMutex flag so we don't get here anymore.
				ReleaseMtx = 1;		// Schedule the mutex for release.
			}
				
		}

		// If the mutex has been flagged for release then call a function to
		// release the mutex.
		if(ReleaseMtx)
		{
			if(ReleaseMutex() == 0)
				ReleaseMtx = 0;
		}
		
		// If we are flagged to do so store any data we get from the XBee module in 
		// the EERPOM, increment the EEPROM address, and clear the flag. 
		if(WriteEEPROMRX)
		{
			// Attempt to write the EEPROM data.
			if(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)) == 0)
			{	
					// If successful then increment the EEPROM address and clear flag.
					EEPROMAddress += sizeof(RecData.Data);
					WriteEEPROMRX = 0;
			}
		}

		// This message is to indicate that the
		// XB module has sent us data for us to
		// process.
		if(XBMessRec && WriteEEPROMRX == 0)
		{	
			XBMessRec = 0;		// Clear the flag.

			// Here we will look at what data we received and
			// determine what to do.
			switch(RecData.Data[0])
			{
				// -----------------------------------------------------------------------
				// Here are the responses and commands from EarthSat.
				// -----------------------------------------------------------------------

				// The first message we will get is from EarthSat asking
				// if we are here. We should reply with an ACK and get
				// the status of the robots which we will pass to all
				// other robots and to EarthSat.
				case AreYouThere:	TransData.DestAddLSB = EarthSat;	// Sending to EarthSat.
									TransData.Data[0] = ACK;			// Send an ACK.
									
									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);
 
									while(XBWriteAPI() == -1);			// Write the API.
									GetMutex();							// Get the Mutex.
									GetRobotStatus();					// Get RobotStatus.
									while(ReleaseMutex() == -1);		// Release the Mutex.

									GoFlag = 1;			// Tell the robots to Go.
									WaitForMutex = 1;	// We need the mutex to that.

									break;

				case ELocation:		break;
		
				// When EarthSat wants to stop a robot we will get here. We'll set a flag
				// handle this in the main loop.
				case EStop:			StopFlag = 1;			// Flag to handle this in main loop.
									WaitForMutex = 1;		// Need the mutex to send the message.
									StopRobot = RecData.Data[1];	// What robot to stop.
									break;

				case EBall:			break;

				// When EarthSat wants to resume a robot we will get here. We'll set a flag
				// handle this in the main loop.
				case EResume:		ResumeFlag = 1;			// Flag to handle this in main loop.
									WaitForMutex = 1;		// Need the mutex to send the message.
									ResumeRobot = RecData.Data[1];	// What robot to resume.
									break;

				// This message is the damage message that we send to the robots as part of
				// the project setup. It is not part of the actual task but to set up the
				// task. Although not mandatory we will likely use this before we ever begin
				// the task and once the task begins we will not use it again.
				case EDamage:		TransData.Data[0] = YourDamage;			// Send the damage command.
									TransData.Data[2] = RecData.Data[2];	// Send the actual damage.
									if(RecData.Data[1] == Seeker1Arg)		// Determine who we are
									{										// sending the data to and
										TransData.DestAddLSB = Seeker1;		// change the address to
										TransData.Data[1] = Seeker1Arg;		// reflect that.
								
										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);

										while(XBWriteAPI() == -1);			// Write the API.
									}
									else if(RecData.Data[1] == Seeker2Arg)
									{
										TransData.DestAddLSB = Seeker2;
										TransData.Data[1] = Seeker2Arg;

										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);	

										while(XBWriteAPI() == -1);			// Write the API.
									}
									else if(RecData.Data[1] == Reader1Arg)
									{
										TransData.DestAddLSB = Reader1;
										TransData.Data[1] = Reader1Arg;

										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);	

										while(XBWriteAPI() == -1);			// Write the API.
									}
									else if(RecData.Data[1] == Reader2Arg)
									{
										TransData.DestAddLSB = Reader2;
										TransData.Data[1] = Reader2Arg;

										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);	

										while(XBWriteAPI() == -1);			// Write the API.
									}
									// If the arguement is not one of the robot but something else
									// we won't know what to do so we will set an LED and send a
									// message to the terminal and make that the end of this
									// message.
									else
									{
										LEDOn(LED6);
										TIputs("EDamage: Unknown second arg.");
									}
									break;

				// This message is to dump our EEPROM data to EarthSat. The first packet contains our command,
				// our source ID, two bytes for the size, and then four bytes for data. Every other packet
				// will contain one byte for the command followed by seven bytes of data. We will read from the
				// EEPROM until we have reached a 0x00 in the first byte (as we don't have a command of 0x00
				// this indicates that there are no more stored commanded) of the message or we have reached 1024
				// sets. 
				case EEEPROM:		TransData.DestAddLSB = EarthSat;	// Sending to EarthSat.
									TransData.Data[0] = MEEPROM;		// Sending out EEPROM.
									TransData.Data[1] = MarsSatArg;		// Next arg is the source of the EEPROM.
									
									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);

									// First Determine our size. We do with by going through the EEPROM data
									// 8 bytes at a time. No command begins with 0x00 so if we find the first
									// byte in a set of 8 bytes to be 0x00 then we have reached the end of the list.
									// The maximum is 1024 sets of bytes. If we go that far we are done.
									size = 0;
									while(size < 1024)						// Won't go more than 1024 sets.
									{
										while(ReadEEPROM(size << 3, buffer, 8));	// Read the next eight bytes.
										if(buffer[0] == 0)					// If we read 0x00 in the first byte
											break;							// Then we are done with list.
										else
											size++;							// Otherwise increment and check the next set.
									}

									// Multiply size by 8.
									size = size << 3;

									// Assign the next two arguements to the size.
									TransData.Data[2] = (unsigned char)size;
									TransData.Data[3] = (unsigned char)(size >> 8);

									// This is a fun challenge. We have one eight byte array to read from
									// and one seven byte array to read to. When we reach the end of the seven
									// byte array then we write that to the XBeeAPI. When we reach the end of the
									// eight byte array (which is the array read from the EEPROM) then get
									// eight more bytes from the EEPROM. If we read a 0x00 from the first byte
									// in the next eight byte set then we are done. But, we need to check that
									// there maybe data we haven't sent to the XBee API because we have not
									// reached the limit of the seven byte array just yet. If there is such
									// data then send it.

									// Set counter1=4 because that is where we begin the seven byte array due
									// to the first packets contains the command, source, and two bytes for size.
									// Counter2=8 to force a read from the EEPROM as soon as we get into the loop.
									counter1=4;counter2=8;size=0;
									while(1)
									{
										// If we have reached the end of the eight byte array then read in new
										// eight bytes and determine if we are done by looking at the first byte.
										if(counter2 == 8)
										{
											while(ReadEEPROM(size << 3, buffer, 8));	// Reads 8 bytes from the EEPROM.
											counter2 = 0;						// Reset the index to the 8 byte array.
											size++;								// Increment the size. No more than 1024 sets.
											if(buffer[0] == 0 || size == 1024)	// If the first byte is 0x00 or we have 1024 sets.
											{
												if(counter1 != 1)				// Check to see if we need to do another XBee write.
													while(XBWriteAPI() == -1);	// Write the API.

												break;							// We are done so break out of the loop.
											}
										}

										// If we have reached the end of the seven byte array then we need to write
										// the data of that array to the XBee module. This is slightly deceiving as
										// we wait for counter1 to be 8 but remember that the first byte is always the
										// command, which leaves us with seven bytes for the data (unless it is the first
										// data packet). That is why we reset the counter1 index to 1.
										if(counter1 == 8)
										{
											while(XBWriteAPI() == -1);			// Write the API.
											counter1 = 1;
										}

										// Assign the buffer data to the XBee API data and increment respective indexes.
										TransData.Data[counter1] = buffer[counter2];
										counter1++;counter2++;												
									}

									break;

				// Here EarthSat has asked us for a robot's EEPROM. We need to first figure out if that robot can
				// talk to us. If it can then we send a message to it to dump its EEPROM. If it can't talk to us
				// then we need to relay the message to another robot. We will then get the robot's EEPROM data
				// from the relay robot.
				case EREEPROM:		// Build an array of zeros for clearing the EERPOM. We should clear
									// the EEPROM because there maybe other data in there from another
									// operation.
									for(index = 0;index < 64;index++)	// Clear the array.
										array[index] = 0;
						
									// Clear the EEPROM.
									for(index = 0x80;index < 0x100;index++)
										while(WriteEEPROM(index << 6, array, sizeof(array)));

									// Determine if we can talk to the robot.
									if( Robot[RecData.Data[1]].Status == OK || Robot[RecData.Data[1]].Status == NoMove ||
										Robot[RecData.Data[1]].Status == NoLocalCOM )	
									{
										TransData.DestAddLSB = RecData.Data[1] + 3;	// Determine which robot we are sending to.
										TransData.Data[0] = DumpEEPROM;				// Sending out Dump EEPROM command.
										TransData.Data[1] = RecData.Data[1];		// When sending to a robot the X arg is the robot.
									
										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);
 
										while(XBWriteAPI() == -1);			// Write the API.
									}

									// Be sure robot isn't dead.
									else if(Robot[RecData.Data[1]].Status == NoSatCOM)	
									{
										TransData.Data[0] = RelayDumpEEPROM;		// Sending out Relay Dump EEPROM command.
										TransData.Data[2] = RecData.Data[1];		// Robot we need to Relay to.

										// Now determine which robot we are going to send the relay command to.
										XArg = RecData.Data[1];
										while(1)
										{
											XArg++;			// Try the next robot.
											if(XArg == 4)
												XArg = 0;

											// Check to see if the next robot can talk to us and other robots.
											// If so then send the message to him.
											if(Robot[XArg].Status == OK || Robot[XArg].Status == NoMove)
											{
												TransData.DestAddLSB = XArg + 3;
												TransData.Data[1] = XArg;
												break;
											}
										}

										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);
 
										while(XBWriteAPI() == -1);			// Write the API.
									}

									// Else the robot is dead and we should never have gotten a dump EEPROM
									// message for him anyway. Turn on the error LED, message the user about
									// the error and get on with life.
									else
									{
										TransData.DestAddLSB = EarthSat;		// Send to EarthSat
										TransData.Data[0] = MEEPROM;			// Sending out Dump EEPROM command.
										TransData.Data[1] = RecData.Data[1];	// For the requested Robot.
										TransData.Data[2] = 0;					// Send a data size of 0.
										TransData.Data[3] = 0;

										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);
 
										while(XBWriteAPI() == -1);			// Write the API.
									}

									break;

				// -----------------------------------------------------------------------
				// Here are the responses and commands from the Robots.
				// -----------------------------------------------------------------------

				// When the robot sends us a message updating his location we get here. We
				// will save that data in the robot struct and then flag the data to be
				// sent to EarthSat and/or other robots in the main loop.
				case RLocation:		Robot[RecData.Data[1]].CurrentLocation = RecData.Data[2];	// Store data in struct.

									// Based on who sent us the data set a flag that indicates
									// we should sent a message to EarthSat and/or other robots.
									if(RecData.Data[1] == Seeker1Arg)
										UpdateLocSeeker1 = 1;
									else if(RecData.Data[1] == Seeker2Arg)
										UpdateLocSeeker2 = 1;
									else if(RecData.Data[1] == Reader1Arg)
										UpdateLocReader1 = 1;
									else if(RecData.Data[1] == Reader2Arg)
										UpdateLocReader2 = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When a robot feels the need to relay another robot's location (because the other
				// robot is unable to speak to us himself) we will get here. Simply put, we
				// will save that data in the robot struct and then flag the data to be
				// sent in the main loop.
				case RLocationRelay:// Store data in struct.
									Robot[RecData.Data[2]].CurrentLocation = RecData.Data[3];
									
									// Based on who sent us the data set a flag that indicates
									// we should sent a message to EarthSat and/or other robots.
									if(RecData.Data[2] == Seeker1Arg)
										UpdateLocSeeker1 = 1;
									else if(RecData.Data[2] == Seeker2Arg)
										UpdateLocSeeker2 = 1;
									else if(RecData.Data[2] == Reader1Arg)
										UpdateLocReader1 = 1;
									else if(RecData.Data[2] == Reader2Arg)
										UpdateLocReader2 = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When the robot sends us a message that a ball is found we get here. We
				// will save that data in the ball struct and then flag the data to be
				// sent to EarthSat and/or other robots in the main loop.
				case RBallFound:	// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
										Ball[RecData.Data[2] - 0x41].Status = There;
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										TIputs("RBallFound: Index out of bounds\n\r\0");
										break;
									}

									// Set a flag in the ball struct to indicate that we should
									// relay the ball's status to EarthSat. 
									Ball[RecData.Data[2] - 0x41].RelayStatus = 1;

									// Set a variable in the ball struct that says what robot
									// gave this ball information.
									Ball[RecData.Data[2] - 0x41].Robot = RecData.Data[1];

									// Set a flag so that in the main loop we will look to send the
									// message to EarthSat.
									UpdateBallThere = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When a robot feels the need to relay another robot's ball mess (because the other
				// robot is unable to speak to us himself) we will get here. Simply put, we
				// will save that data in the ball struct and then flag the data to be
				// sent in the main loop.
				case RBallFoundRelay:// Store data in struct.
									// Check the index is in bounds.
									if( ((RecData.Data[3] - 0x41) >= 0) && ((RecData.Data[3] - 0x41) <= 7) )
										Ball[RecData.Data[3] - 0x41].Status = There;
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										TIputs("RBallFoundRelay: Index out of bounds\n\r\0");
										break;
									}
									
									// Set a flag in the ball struct to indicate that we should
									// relay the ball's status to EarthSat. 
									Ball[RecData.Data[3] - 0x41].RelayStatus = 1;

									// Set a variable in the ball struct that says what robot
									// gave this ball information.
									Ball[RecData.Data[2] - 0x41].Robot = RecData.Data[2];

									// Set a flag so that in the main loop we will look to send the
									// message to EarthSat.
									UpdateBallThere = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When the robot sends us a message that a ball is found we get here. We
				// will save that data in the ball struct and then flag the data to be
				// sent to EarthSat and/or other robots in the main loop.
				case RBallNotFound:	// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
										Ball[RecData.Data[2] - 0x41].Status = NotThere;
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										TIputs("RBallNotFound: Index out of bounds\n\r\0");
										break;
									}

									// Set a flag in the ball struct to indicate that we should
									// relay the ball's status to EarthSat. 
									Ball[RecData.Data[2] - 0x41].RelayStatus = 1;

									// Set a variable in the ball struct that says what robot
									// gave this ball information.
									Ball[RecData.Data[2] - 0x41].Robot = RecData.Data[1];

									// Set a flag so that in the main loop we will look to send the
									// message to EarthSat.
									UpdateBallThere = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When a robot feels the need to relay another robot's ball mess (because the other
				// robot is unable to speak to us himself) we will get here. Simply put, we
				// will save that data in the ball struct and then flag the data to be
				// sent in the main loop.
				case RBallNotFoundRelay:// Store data in struct.
									// Check the index is in bounds.
									if( ((RecData.Data[3] - 0x41) >= 0) && ((RecData.Data[3] - 0x41) <= 7) )
										Ball[RecData.Data[3] - 0x41].Status = NotThere;
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										TIputs("RBallNotFoundRelay: Index out of bounds\n\r\0");
										break;
									}
									
									// Set a flag in the ball struct to indicate that we should
									// relay the ball's status to EarthSat. 
									Ball[RecData.Data[3] - 0x41].RelayStatus = 1;

									// Set a variable in the ball struct that says what robot
									// gave this ball information.
									Ball[RecData.Data[2] - 0x41].Robot = RecData.Data[2];

									// Set a flag so that in the main loop we will look to send the
									// message to EarthSat.
									UpdateBallThere = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When the robot sends us a message that if has ball IR Data we get here. We
				// will save that data in the ball struct and then flag the data to be
				// sent to EarthSat and/or other robots in the main loop.
				case RBallData:		// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
										Ball[RecData.Data[2] - 0x41].IRData = RecData.Data[3];
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										TIputs("RBallData: Index out of bounds\n\r\0");
										break;
									}

									// Set a flag in the ball struct to indicate that we should
									// relay the ball's IR Data to EarthSat. 
									Ball[RecData.Data[2] - 0x41].RelayData = 1;

									// Set a variable in the ball struct that says what robot
									// gave this ball information.
									Ball[RecData.Data[2] - 0x41].Robot = RecData.Data[1];

									// Set a flag so that in the main loop we will look to send the
									// message to EarthSat.
									UpdateBallData = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When a robot feels the need to relay another robot's ball mess (because the other
				// robot is unable to speak to us himself) we will get here. Simply put, we
				// will save that data in the ball struct and then flag the data to be
				// sent in the main loop.
				case RBallDataRelay:// Store data in struct.
									// Check the index is in bounds.
									if( ((RecData.Data[3] - 0x41) >= 0) && ((RecData.Data[3] - 0x41) <= 7) )
										Ball[RecData.Data[3] - 0x41].IRData = RecData.Data[4];
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										TIputs("RBallDataRelay: Index out of bounds\n\r\0");
										break;
									}
									
									// Set a flag in the ball struct to indicate that we should
									// relay the ball's IR Data to EarthSat. 
									Ball[RecData.Data[3] - 0x41].RelayData = 1;

									// Set a variable in the ball struct that says what robot
									// gave this ball information.
									Ball[RecData.Data[3] - 0x41].Robot = RecData.Data[2];

									// Set a flag so that in the main loop we will look to send the
									// message to EarthSat.
									UpdateBallData = 1;

									// Set the flag so that the main loop will store the data
									// in EEPROM and send the data via XBee.
									WaitForMutex = 1;

									break;

				// When we reach this case we need to relay the EERPOM data from the robots
				// to EarthSat. So first we will get the EEPROM data from the robot and store
				// it in the upper half of our EERPOM. We will then send the upper half of
				// our EEPROM to EarthSat. Just as when we send the MarsSat data to EarthSat
				// when we get data from the robot we get 7 byte chunks because the first
				// byte is the command but we store 8 bytes at a time which make this a
				// good time.
				case REEPROMRelay:
				case REEPROM:		// Begin by getting the robot data and storing it in the upper
									// part of our EEPROM.
									XArg = RecData.Data[1];			// This arguement is the robot the data is from.
									size = RecData.Data[2];			// Determine the size.
									size += (unsigned int)RecData.Data[3] << 8;

									// While we have not reached the end of the EEPROM read robot data
									// from the XBee receive buffer and store that data in our EEPROM.
									counter1 = 4;counter2 = 0;EEPROMRobotAddress = 0x2000;counter3=0;
									while(EEPROMRobotAddress < 0x4000)
									{
										// Store Robot data in a local buffer for now and increment the counters.
										buffer[counter2] = RecData.Data[counter1];
										counter2++;counter1++;

										// If we have filled the local buffer then write the data to the EEPROM.
										if(counter2 == 8)
										{
											// Store robot data.
											while(WriteEEPROM(EEPROMRobotAddress, buffer, sizeof(buffer)));
											EEPROMRobotAddress += sizeof(buffer);

											counter2 = 0;				// Clear the index counter.

											counter3 += sizeof(buffer);	// Increment the counter.

											// Determine if we are done getting data from the robot. If so, tell
											// Rec ISR to now record data and break out of the loop.
											if(counter3 == size)
											{
												// If we are done then we do want the Rec ISR to store the data.
												RecEEPROM = 0;

												// If we have read all the data from the robot and written all the
												// data to the our EEPROM then we are done and we should break out.
												break;
											}
										}

										// If we reached the end of the Rec buffer from the XBee module then
										// wait for more data from the robot.
										if(counter1 == 8)
										{
											
											// If we are not done then we need to wait for more data from the robot.
											RecEEPROM = 1;				// Tell the Rec ISR not to store the data.
											while(XBMessRec == 0);		// Spin around here waiting for data.
											XBMessRec = 0;				// Clear the flag.
											counter1 = 1;				// First byte is command so we start the data at 1.
										}
									}

									// Now that we have grabbed all the data from the robot we need to read that
									// data back out of the EEPROM and send it to EarthSat.
									TransData.DestAddLSB = EarthSat;	// Sending to EarthSat.
									TransData.Data[0] = MEEPROM;		// Sending out EEPROM.
									TransData.Data[1] = XArg;			// Next arg is the source of the EEPROM.

									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);

									// Assign the next two arguements to the size.
									TransData.Data[2] = (unsigned char)size;
									TransData.Data[3] = (unsigned char)(size >> 8);

									// Set counter1=4 because that is where we begin the seven byte array due
									// to the first packets contains the command, source, and two bytes for size.
									// Counter2=8 to force a read from the EEPROM as soon as we get into the loop.
									counter1=4;counter2=8;EEPROMRobotAddress = 0x2000;
									while(1)
									{
										// If we have reached the end of the eight byte array then read in new
										// eight bytes and determine if we are done by looking at the first byte.
										if(counter2 == 8)
										{
											while(ReadEEPROM(EEPROMRobotAddress, buffer, sizeof(buffer)));	// Reads 8 bytes from the EEPROM.
											counter2 = 0;							// Reset the index to the 8 byte array.
											EEPROMRobotAddress += sizeof(buffer);	// Increment the size.
											if(buffer[0] == 0 || EEPROMRobotAddress == 0x4000)	// If the first byte is 0x00 or we have too much data.
											{
												if(counter1 != 1)				// Check to see if we need to do another XBee write.
													while(XBWriteAPI() == -1);	// Write the API.

												break;							// We are done so break out of the loop.
											}
										}

										// If we have reached the end of the seven byte array then we need to write
										// the data of that array to the XBee module. This is slightly deceiving as
										// we wait for counter1 to be 8 but remember that the first byte is always the
										// command, which leaves us with seven bytes for the data (unless it is the first
										// data packet). That is why we reset the counter1 index to 1.
										if(counter1 == 8)
										{
											while(XBWriteAPI() == -1);			// Write the API.
											counter1 = 1;
										}

										// Assign the buffer data to the XBee API data and increment respective indexes.
										TransData.Data[counter1] = buffer[counter2];
										counter1++;counter2++;												
									}

									break;

				default:			LEDOn(LED6);
									TIputs("Main: Unknown command");
									break;
			}
		}

		// This message is for when we have pressed
		// button1.
		if(Button1Pressed)
		{
			Button1Pressed = 0;	// Clear the flag.
			HumanInt();
		}

		// This message is for when we have pressed
		// button2.
		if(Button2Pressed)
		{
			Button2Pressed = 0;	// Clear the flag.
		}

	} // End while(1)
}// End main()
