/*************************************************************
* This file contains the functions for the HumanInterface.
* These will include functions to test LEDs, test EEPROM,
* display USPulse data, display IR data, display IRDA data,
* command motors, dump EEPROM data, and send/receive data via
* the XB module.
*
* Written by: Shane Avery
**************************************************************/
#include<p33FJ128GP206.h>
#include "Robot.h"
#include "IOFuncs.h"

// This function will test the LEDs by turning each on for a
// short amount of time. Since the LEDs are used for debug it
// is useful to make sure that they are in fact working each
// time they are turned on. 
//
// NOTE: We intend to use this function one time after power
// up. Thus, we can re-use timer1 after this. In fact, the
// RS232 Terminal/IRDA uses timer1 so we in fact should not
// call this function after power up.
void LEDTest()
{
	T1CONbits.TCKPS = 3;	// Set prescale to 1:256.
	PR1 = 0xBFFF;			// Set the period register.
	TMR1 = 0;				// Clear the timer.
	IFS0bits.T1IF = 0;		// Clear the interrupt flag.
	T1CONbits.TON = 1;		// Turn timer on.

	// Now turn on and off LEDs one at a time to determine
	// if they work.
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOn(LED0);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED0);
	LEDOn(LED1);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED1);
	LEDOn(LED2);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED2);
	LEDOn(LED3);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED3);
	LEDOn(LED4);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED4);
	LEDOn(LED5);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED5);
	LEDOn(LED6);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LED6);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;

	// Lastly turn all LEDs on and then off just
	// to be sure. It helps. Trust me.
	LEDOn(LEDAll);
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	LEDOff(LEDAll);	

	T1CONbits.TON = 0;		// Turn timer off.
	T1CONbits.TCKPS = 0;	// Set prescale to 1:1.
}

// Here is a function to test the EEPROM. We will first
// write a counter and then read it back. Then we will
// write all ones and read it back. Lastly we will
// write all zeros and read it back. We will output
// progress and results to the terminal.
void TestEEPROM()
{
	char error = 0;
	int counter1, counter2;
	int counter3;
	unsigned char array[64];

	// &&&&&&&&&&&&&&&&&&&&&&
	// Write/Read a counter
	// &&&&&&&&&&&&&&&&&&&&&&
	TIputs("Writing count\0");
	error = 0;
	for(counter1=0;counter1<0x100;counter1++)
	{
		for(counter2=0;counter2<64;counter2++)
			array[counter2] = counter2 + (counter1 << 6);

		while(WriteEEPROM(counter1 << 6, array, sizeof(array)));

		counter3 = counter1 & 0x1f;
		if(counter3 == 0)
			TIputc('.');
	}
	TIputs("Done!\r\n\0");
	
	TIputs("Reading count\0");
	for(counter1=0;counter1<0x100;counter1++)
	{
		while(ReadEEPROM(counter1 << 6, array, sizeof(array)));
		for(counter2=0;counter2<64;counter2++)
			if(array[counter2] != ((counter2 + (counter1 << 6)) & 0xff))
				error = 1;

		counter3 = counter1 & 0x1f;
		if(counter3 == 0)
			TIputc('.');
	}
	if(error)
		TIputs("Error!\r\n\0");
	else
		TIputs("Success!\r\n\0");

	// &&&&&&&&&&&&&&&&&&&&&&
	// Write/Read all ones
	// &&&&&&&&&&&&&&&&&&&&&&
	TIputs("Writing ones \0");
	error = 0;
	for(counter1=0;counter1<64;counter1++)
		array[counter1]=0xff;

	for(counter1=0;counter1<0x100;counter1++)
	{
		while(WriteEEPROM(counter1 << 6, array, sizeof(array)));

		counter3 = counter1 & 0x1f;
		if(counter3 == 0)
			TIputc('.');
	}
	TIputs("Done!\r\n\0");

	TIputs("Reading ones \0");
	for(counter1=0;counter1<0x100;counter1++)
	{
		while(ReadEEPROM(counter1 << 6, array, sizeof(array)));
		for(counter2=0;counter2<64;counter2++)
			if(array[counter2] != 0xff)
				error = 1;

		counter3 = counter1 & 0x1f;
		if(counter3 == 0)
			TIputc('.');
	}
	if(error)
		TIputs("Error!\r\n\0");
	else
		TIputs("Success!\r\n\0");

	// &&&&&&&&&&&&&&&&&&&&&&
	// Write/Read all zeros
	// &&&&&&&&&&&&&&&&&&&&&&
	TIputs("Writing zeros\0");
	error = 0;
	for(counter1=0;counter1<64;counter1++)
		array[counter1]=0;

	for(counter1=0;counter1<0x100;counter1++)
	{
		while(WriteEEPROM(counter1 << 6, array, sizeof(array)));
		
		counter3 = counter1 & 0x1f;
		if(counter3 == 0)
			TIputc('.');
	}
	TIputs("Done!\r\n\0");

	TIputs("Reading zeros\0");
	for(counter1=0;counter1<0x100;counter1++)
	{
		while(ReadEEPROM(counter1 << 6, array, sizeof(array)));
		for(counter2=0;counter2<64;counter2++)
			if(array[counter2] != 0)
				error = 1;

		counter3 = counter1 & 0x1f;
		if(counter3 == 0)
			TIputc('.');
	}
	if(error)
		TIputs("Error!\r\n\0");
	else
		TIputs("Success!\r\n\0");
}

// Here is the human interface function that will be
// called when button1 is pressed. This will provide
// text to the terminal that will help us develop the
// code for the robot.
void HumanInt()
{
	char string[64];
	char mess[64];
	char xbstring[8];
	char ch = 0;
	int index;
	unsigned int *AtoD = (unsigned int *) &_DMA_BASE;
	unsigned int WriteArray[32];
	unsigned char WriteArrayCh[32];
	char lr, fr;
	unsigned int speed;
	char Motor, Dir;
	unsigned char buffer[8];
	int counter;


	//      1234567890123456789012345678901234567890
	TIputs("\r\n\n\0");
	TIputs("Welcome to the Human Interface\r\n\n\0");
	TIputs("led                Test the LEDs\r\n\0");
	TIputs("ee                 Test the EEPROMs\r\n\0");
	TIputs("irda               Send/Rec IRDA data\r\n\0");
	TIputs("us                 Ultra Sonic data\r\n\0");
	TIputs("ir                 IR/Batt data\r\n\0");
	TIputs("xbr                Display 17 bytes of XB rec data\r\n\0");
	TIputs("xb args            Send 17 bytes of XB data.\r\n\0");
	TIputs("m args             Motor command. args=>motor,direction,speed\r\n\0");
	TIputs("dump               Dump EEPROM\r\n\0");
	TIputs("exit               Exit human interface\r\n\0");	

	while(1)
	{
		TIputc('>');		

		// Read in the string.
		for(index=0;ch != '\r';index++)
		{
			ch = TIgetc();		// Get the char.
			if(ch == 8)			// Check for a backspace
			{
				index-=2;		// If we have BS back up the index.
				if(index < -1)	// but never go smaller than -1 or we
					index = -1;	// will run out of array bounds.
				else
					TIputc(ch);			// Echo it back.
			}
			else
			{
				TIputc(ch);			// Echo it back.
				string[index] = ch;	// Store it in the string.
			}
		}			
		TIputc('\n');	// Send a new line.
		ch = 0;			// Set ch back to 0 so we don't loop 4ever.
	
		// NULL terminate string.
		string[index] = 0;
		index++;
	
		// Parse the string into command and arguements.
		// LED Test.
		if(strncmp("led", string, 3) == 0)
			LEDTest();
		// EEPROM Test.
		else if(strncmp("ee", string, 2) == 0)
			TestEEPROM();
		// Send/Rec IRDA data.
		else if(strncmp("irda", string, 4) == 0)
		{
			sprintf(mess, "IRDA retval = %x\r\n\0", GetIRDA());
			TIputs(mess);
		}
		// Get US counter value.
		else if(strncmp("us", string, 2) == 0)
		{
			sprintf(mess, "US retval = %x\r\n\0", USPulse());
			TIputs(mess);
		}
		// Display the current AtoD values.
		else if(strncmp("ir", string, 2) == 0)
		{
			sprintf(mess, "IR/Batt vals = %x %x %x %x %x\r\n\0", 
					AtoD[0], AtoD[1], AtoD[2], AtoD[3], AtoD[4]);
			TIputs(mess);
		}
		// Display the current XB receive array.
		else if(strncmp("xbr", string, 3) == 0)
		{
			sprintf(mess, "XBRX = %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x\r\n\0",
					XBRX[0], XBRX[1], XBRX[2], XBRX[3], XBRX[4], XBRX[5],
					XBRX[6], XBRX[7], XBRX[8], XBRX[9], XBRX[10], XBRX[11],
					XBRX[12], XBRX[13], XBRX[14], XBRX[15], XBRX[16]);
			TIputs(mess);
		}
		// Accept 17 bytes and write them to the XB module.
		else if(strncmp("xb", string, 2) == 0)
		{
			// Scan in the values from the string.
			sscanf(&string[3], "%x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x",
					&WriteArray[0], &WriteArray[1], &WriteArray[2],
					&WriteArray[3], &WriteArray[4], &WriteArray[5],
					&WriteArray[6], &WriteArray[7], &WriteArray[8],
					&WriteArray[9], &WriteArray[10], &WriteArray[11],
					&WriteArray[12], &WriteArray[13], &WriteArray[14],
					&WriteArray[15], &WriteArray[16]);
	
			// Copy the integer string to the char string.
			for(index = 0;index < 32;index++)
				WriteArrayCh[index] = WriteArray[index];
	
			// Write the array to the XB module. Currently we will
			// write 17 bytes but we can any amount we define later.
			// Keep trying to write data until we are successful.
			while(XBWrite(WriteArrayCh, 17) == -1);			
		}
		// Command motor.
		else if(strncmp("m", string, 1) == 0)
		{
			// Scan in the args.
			sscanf(&string[2], "%c %c %x", &lr, &fr, &speed);

			// Parse the args
			if(lr == 'l')
				Motor = Left;
			else
				Motor = Right;

			if(fr == 'f')
				Dir = Forward;
			else
				Dir = Reverse;

			// Command the motor. Keep trying until successful.
			while(CommandMotor(Motor, Dir, (unsigned char)speed) == -1);
			
		}
		// Display the number of files in the EEPROM.
		else if(strncmp("fnums", string, 5) == 0)
		{
			TIputs("Unimplemented!\r\n\0");
		}
		// Dump the EEPROM file data to the terminal.
		else if(strncmp("dump", string, 4) == 0)
		{
			counter=0;
			buffer[0] = 0xff;
			while(buffer[0] != 0)
			{
				while(ReadEEPROM(counter, buffer, 8));
				counter += 8;
				sprintf(string, "%x %x %x %x %x %x %x %x\r\n\0",
					buffer[0], buffer[1], buffer[2], buffer[3],
					buffer[4], buffer[5], buffer[6], buffer[7]);
				TIputs(string);
			}
		}
		// Exit the human interface.
		else if(strncmp("exit", string, 4) == 0)
			break;
		// If we just hit enter.
		else if(string[0] == '\r')
			TIputs("\r\0");
		// Unknown command.
		else
			TIputs("Unknown command\n\r\0");
	}
}
