/**********************************************************
* This code contains the IO functions used for the Robot. 
* IO includes LEDs, pushbuttons, Terminal, IRDA, EEPROM, 
* Ultrasonic, Motors, IR sensors, and XBee module.
* 
* Written by: Shane Avery
***********************************************************/
#include<p33FJ128GP206.h>
#include "Robot.h"
#include "IOFuncs.h"

// This function is used to turn LEDs on. Simply pass in
// what LEDs we want on.
void LEDOn(unsigned int LEDs)
{
	unsigned int temp = 0xffff;
	temp ^= LEDs;
	PORTB &= temp;	
}

// This function is used to turn LEDs off. Simply pass in
// what LEDs we want on. 
void LEDOff(unsigned int LEDs)
{
	PORTB |= LEDs;
}

// This function will use timer6 to wait for 100ms.
// It simply sets up timer6 and looks for the
// interrupt flag.
void Wait100ms()
{
	// Set up timer to wait for 100ms.
	PR6 = 0x3D09;				// Compared value.
	T6CONbits.TCKPS = 3;		// Set prescaler to 256;
	TMR6 = 0;					// Clear the timer.
	IFS2bits.T6IF = 0;			// Clear the interrupt.
	T6CONbits.TON = 1;			// Turn the timer on.

	// Wait for the timer interrupt to set.
	while(IFS2bits.T6IF == 0);	
	T6CONbits.TON = 1;		// Turn the timer off.
	IFS2bits.T6IF = 0;		// Clear the interrupt.
	TMR6 = 0;				// Clear the timer.	
}

// This function will write size bytes from the buffer starting
// at address to the EEPROM using I2C. It should noted that this
// is a page write and so no more than 64 bytes should be written
// at a time or else the data will be corrupted. We will first
// check to see if there is another I2C operation otherwize we
// will fill in some Global buffers and variables, start a transmit
// and let the ISR do the work. This will allow us to call this
// function and quickly return to the main loop. We only block
// if we do two writes in a row.
char WriteEEPROM(int address, char *buffer, unsigned char size)
{
	unsigned char counter;

	// Check to see if we are currently doing another I2C operation.
	// If we are then return with a value of one.
	if(I2CDone == 0)
		return 1;
	
	// Break the address up into MSB and LSB.
	I2CLSB = address & 0xff;
	I2CMSB = (address & 0xff00) >> 8;

	// Write the size of the write to the global variable.
	I2CSize = size;

	// Copy the buffer to the I2C Global Buffer.
	for(counter=0;counter < size;counter++)
		I2CBuffer[counter] = buffer[counter];
	
	// Set/Clear flags.
	I2CDone = 0;			// Clear the done flag. The ISR will set it.
	I2CRW = 0;				// We want a write so set this to 0.
	I2C1CONbits.SEN = 1;	// Initiate start.

	return 0;	
}

// This function will read size bytes from the EEPROM using I2C
// starting at address and store the bytes in the buffer. Note
// that the EEPROM only allows for a page size of 64 bytes.
// Thus, we should never ask for a read bigger than that.
// We will first check to see if there is another I2C operation 
// otherwize we will fill in some Global buffers and variables, 
// start a recieve and let the ISR do the work. Unlike WriteEEPROM
// this will block until we are done with the operation. This is
// not that big of a deal because we will only do a read to dump
// the entire memory and everyone in the system should be stopped
// at that point anyway.
char ReadEEPROM(int address, char *buffer, unsigned char size)
{
	char MSB, LSB;
	unsigned char counter;
	char Ready = 0;

	// Check to see if we are currently doing another I2C operation.
	// If we are then return with a value of one.
	if(I2CDone == 0)
		return 1;
	
	// Break the address up into MSB and LSB.
	I2CLSB = address & 0xff;
	I2CMSB = (address & 0xff00) >> 8;

	// Write the size of the write to the global variable.
	I2CSize = size;

	// Set/Clear flags.
	I2CDone = 0;			// Clear the done flag. The ISR will set it.
	I2CRW = 1;				// We want a read so set this to 1.
	I2C1CONbits.SEN = 1;	// Initiate start.

	while(I2CDone == 0);	// Wait for the ISR to be done.

	// Copy the I2C Global buffer to buffer.
	for(counter=0;counter<size;counter++)
		buffer[counter] = I2CBuffer[counter];
	
	return 0;
}

// This is the I2C ISR. This function gets called whenever anything
// happens on I2C which is quite annoying. Since I need to determine
// what happened each time I come and then take action we implement
// this as a state machine. Whenever there is a read or write I2C
// operation this ISR will handle the actual sending and receiving
// of I2C data. This will allow the EEPROMWrite function to be
// non-blocking which should improve our performance in the main loop.
void __attribute__((__interrupt__)) _MI2C1Interrupt (void)
{
	static int I2CState = 0;
	static char I2CCounter = 0;

	// State machine for this ISR.
	switch(I2CState)
	{
		// Once we have initiated a start condition we need to send the address.
		case 0:		I2CState = 1;		// Define the next state.
					I2CCounter = 0;		// Clear the buffer counter.
					I2C1TRN = 0xA0;		// Address EEPROM for a write.
					break;

		// Check to be sure that the EEPROM is ready for us. If the EEPROM
		// is busy writing the data we gave it to the EEPROM then it will
		// not be immediately ready for another operation is which case
		// if wil not ACK when addresses. If we get a NACK then we initiate
		// a stop and try again.
		case 1:		if(I2C1STATbits.ACKSTAT)	// If we got a NACK
					{
						I2C1CONbits.PEN = 1;	// Initiate stop.	
						I2CState = 2;			// Define the next state in which
					}							// we will initiate another start.
					else
					{							// Else we got an ACK.
						I2C1TRN = I2CMSB;		// Write the first word of the address.	
						I2CState = 3;			// Define next state.
					}
					break;	

		// In the case that the EEPROM is not ready for us yet we will initiate
		// another start condition and try again.
		case 2:		I2C1CONbits.SEN = 1;		// Initiate start.
					I2CState = 0;				// Go back to the beginning.
					break;

		// Send the address LSB that we wish to write.
		case 3:		I2C1TRN = I2CLSB;	// Write the LSB of the address.

					if(I2CRW == 0)		// Determine if we are at this ISR for
						I2CState = 4;	// a read or a write and determine our
					else				// next state accordingly. The function
						I2CState = 6;	// that first initiated the start defined
										// this value (it's a global variable).
					break;

		// Write the data in the I2C buffer to the EEPROM.
		case 4:		if(I2CCounter < I2CSize)	// Write as many bytes of data as
					{							// defined by the global var I2CSize.
						I2C1TRN = I2CBuffer[I2CCounter];
						I2CCounter++;
					}
					else						// Once we has written all the data
					{							// we initiate a stop.
						I2C1CONbits.PEN = 1;	
						I2CState = 5;			// Define the next state.
						I2CCounter = 0;			// Reset the buffer counter.
					}
					break;

		// Once we have initiated the last stop we come here where we will set the
		// I2CDone global variable that tells the main loop that we are done writing
		// I2C data to the EEPROM and we can begin a new I2C operation.
		case 5:		I2CDone = 1;	// Set the flag to indicate that we are done.
					I2CState = 0;	// Go back to the beginning.
					break;

		// If we are receiving data then we need to initiate a restart.
		case 6:		I2C1CONbits.ACKDT = 0;		// Send an ACK when we receive data.
					I2C1CONbits.RSEN = 1;		// Initiate restart.
					I2CState = 7;				// Define the next state.
					break;

		// Address the EEPROM for a read.
		case 7:		I2C1TRN = 0xA1;				// Address EEPROM for a read.
					I2CState = 8;				// Define the next state.
					break;

		// Initiate a receive.
		case 8:		I2C1CONbits.RCEN = 1;		// Initiate receive.
					I2CState = 9;				// Define the next state.
					break;

		// When the received data comes in we get to this state. Here we will
		// store the received data in the global I2C buffer. When we are done
		// we need to send a NACK to the EEPROM to let it know we are done.
		case 9:		I2CBuffer[I2CCounter] = I2C1RCV;	// Store the data.
					I2CCounter++;					// Increment counter.
					if(I2CCounter < I2CSize)		// Read as many bytes as defined
					{								// in the I2CSize variable.
						I2C1CONbits.ACKEN = 1;		// Initiate ACK.
						I2CState = 8;				// Define the next state.
					}
					else							// Else we are done receiving data.
					{
						I2C1CONbits.ACKDT = 1;		// Send a NACK after the last byte.
						I2C1CONbits.ACKEN = 1;		// Initiate NACK.
						I2CState = 10;				// Define the next state.
					}
					break;

		case 10:	I2C1CONbits.PEN = 1;		// Initiate stop.			
					I2CState = 5;				// Define the next state.
					break;
					
	}
	
	IFS1bits.MI2C1IF = 0;	// Clear the interrupt flag.
	
} 

// This function will write the ch byte to the
// Terminal/IRDA output.
void TIputc(char ch)
{
	unsigned char counter;
	unsigned int WRegTemp;
	unsigned int chTemp;

	T1CONbits.TCKPS = 0;	// Set prescale to 1:1.
	PR1 = 0x15B;			// Set the period register.
	TMR1 = 0;				// Clear the timer.
	IFS0bits.T1IF = 0;		// Clear the interrupt flag.
	T1CONbits.TON = 1;		// Turn timer on.

	asm("NOP");				// NOPs to meet timing for
	asm("NOP");				// the start bit.
	asm("NOP");
	asm("NOP");
	PORTDbits.RD0 = 0;		// Start bit.

	chTemp = ch;

	// Now output the eight data bits.
	for(counter = 0;counter < 8;counter++)
	{
		asm("MOV W13,%0" : "+r"(WRegTemp));	// Save W13 to variable.
		asm("MOV %0,W13" : "+r"(chTemp));	// Move chTemp to W13.

		asm("CheckT1IF:");					// Wait for Timer1 interrupt flag.
		asm("BTSS	IFS0,#3");				
		asm("BRA	CheckT1IF");

		asm("BTSS W13,#0");					// Check to see if the next bit
		asm("BRA	ChClr");				// should be set or clear.
		asm("BRA	ChSet");
		asm("ChClr:");						// If it should be cleared then
		asm("NOP");							// clear PORTD pin0.
		asm("BCLR	PORTD,#0");
		asm("BRA	W13Shift");
		asm("ChSet:");						// If it should be set then
		asm("BSET	PORTD,#0");				// set PORTD pin0.
		asm("BRA	W13Shift");
		asm("W13Shift:");					// Shift W13 to the right.
		asm("RRNC	W13,W13");

		asm("MOV W13,%0" : "+r"(chTemp));	// Save W13 to chTemp.
		asm("MOV %0,W13" : "+r"(WRegTemp)); // Restore W13 to what it was before.

		IFS0bits.T1IF = 0;	// Clear the interrupt flag.
	}

	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	PORTDbits.RD0 = 1;		// Stop bit.
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	while(!IFS0bits.T1IF);
	IFS0bits.T1IF = 0;
	T1CONbits.TON = 0;		// Turn timer off.
		
}

// This function writes a string to the Terminal/IRDA
// interface. String must NULL Terminate.
void TIputs(char *ptr)
{
	unsigned char counter;

	// Output the pointer until NULL terminates.
	for(counter=0;ptr[counter] != 0;counter++)
		TIputc(ptr[counter]);
}

// This function will get a byte from the Terminal/IRDA.
// NOTE: This function uses Timer1 just as TIPutc and
// LEDTest. Thus, none of those functions should be called
// from within this function.
char TIgetc()
{
	char ch;
	unsigned char counter;

	T1CONbits.TCKPS = 0;	// Set prescale to 1:1.
	PR1 = 0xAD;				// Set the period register to 1/2
	TMR1 = 0;				// Clear the timer.
	IFS0bits.T1IF = 0;		// Clear the interrupt flag.

	while(PORTDbits.RD1)	// Wait for the start bit.
		if(IFS1bits.T5IF)	// If we don't get a start bit before
			return 0xff;	// timeout then send 0xff.

	T1CONbits.TON = 1;		// Turn timer on.

	while(!IFS0bits.T1IF);	// Wait for timer interrupt flag.
	IFS0bits.T1IF = 0;		// Clear the timer interrupt flag.

	PR1 = 0x15B;			// Set the period register to full.

	// Grab all eight bits.
	for(counter=0;counter<8;counter++)
	{
		ch = ch >> 1;			// Shift it.

		while(!IFS0bits.T1IF);	// Wait for timer interrupt flag.
		IFS0bits.T1IF = 0;		// Clear the timer interrupt flag.
		
		if(PORTDbits.RD1)		// Set or clear the bit as needed.
			ch |= 0x80;
		else
			ch &= 0x7F;
	}	
	
	// Consume the stop bit.
	while(!IFS0bits.T1IF);	// Wait for timer interrupt flag.
	IFS0bits.T1IF = 0;		// Clear the timer interrupt flag.

	T1CONbits.TON = 0;		// Turn timer off.

	return ch;
}

// This function will pulse the Ultra Sonic sensor and then wait
// for the response. We will use Timer4 to determine how long
// before sound returned to the sensor and then return that
// value.
unsigned int USPulse()
{
	// First we need to setup Timer4 to time the pulse to the sensor.
	PR4 = 400;				// Compared value that is 10us.
	T4CONbits.TCKPS = 0;	// Set prescaler to 1;
	TMR4 = 0;				// Clear the timer.
	IFS1bits.T4IF = 0;		// Clear the interrupt.
	
	// Send the trigger pulse.
	PORTFbits.RF1 = 1;		// Raise the line to trigger the US sensor.
	T4CONbits.TON = 1;		// Turn the timer on.
	while(IFS1bits.T4IF == 0);	// Wait for Timer4 to compare.	
	PORTFbits.RF1 = 0;		// Lower the trigger pulse line.

	// Now prepare Timer4 for the echo pulse.
	T4CONbits.TON = 0;		// Turn the timer off.
	TMR4 = 0;				// Clear the timer.
	IFS1bits.T4IF = 0;		// Clear the interrupt.	
	PR4 = 0xffff;			// We want to timer to run a long time.
	T4CONbits.TCKPS = 2;	// Set prescaler to 64;

	// Wait for the pulse and then use Timer4 to determine
	// how long the pulse is.
	while(PORTFbits.RF0 == 0);	// Wait for pulse to set.
	T4CONbits.TON = 1;		// Turn the timer on.
	while(PORTFbits.RF0);	// Wait for the pulse to clear.
	T4CONbits.TON = 0;		// Turn the timer off.

	return TMR4;
}

// This function will return the byte from the IRDA. We will first
// send 0x00 to the ball to tell it that are here and we would
// like a byte. Then we will wait for the byte. We will use Timer5
// as a time out timer. If we time out the return value will be 0xff.
unsigned char GetIRDA()
{
	unsigned char IRDA;
	
	PORTDbits.RD3 = 0;		// First disable the IRDA shutdown.
	TIputc(0x00);			// Send 0x00 to tell Ball to send us a byte.
	TMR5 = 0;				// Clear the timer.
	IFS1bits.T5IF = 0;		// Clear the interrupt.
	T5CONbits.TON = 1;		// Turn the timer on.
	IRDA = TIgetc();		// Get the byte.
	T5CONbits.TON = 0;		// Turn the timer off.
	IFS1bits.T5IF = 0;		// Clear the interrupt.
	TMR5 = 0;				// Clear the timer.
	PORTDbits.RD3 = 1;		// Enable the IRDA shutdown.

	return IRDA;
}

// This function will write command to the motor controller.
// Motor is either Left or Right. Direction is Forward or Reverse.
// Value is the motor speed. Value must be between 0x00 and
// 0x7f. A value of 0x00 turn the motor off. If we go off
// forward then we coast. If we go off reverse then we brake.
// As with the XBWrite we will return a -1 if we are in the
// middle of writing a command and the control loop should
// try to send the command again later.
char CommandMotor(char Motor, char Direction, unsigned char Speed)
{
	char temp;
	
	// First check that we are not currently writing
	// to the motor controller.
	if(MotorTXSize == 0)
		MotorTXSize = 4;	// If not then set the size.
	else
		return -1;			// If so then return -1.

	MotorTX[0] = 0x80;	// Start byte is always 0x80.
	MotorTX[1] = 0x00;	// Device type is always 0x00.

	temp = Motor << 1;	// Motor number resides in bits 1-6.
	temp += Direction;	// Direction resides in bits0.
	MotorTX[2] = temp;

	MotorTX[3] = Speed;	// Motor speed.

	// Clear the MotorTXIndex.
	MotorTXIndex = 0;

	// Setup the U2TX interrupt.
	IFS1bits.U2TXIF = 0;	// First clear the interrupt flag.
	IEC1bits.U2TXIE = 1;	// Enable the interrupt.
	U2TXREG = MotorTX[0];	// Write the first byte.	

	return 0;
} 

// This is the ISR that gets called when the UART can receive
// another transmit byte. We use this function to send data to
// the Motor controller. This interrupted method is better because
// we don't block the CPU from doing something else as before when
// we just sat there waiting for the transmit to complete.
void __attribute__((__interrupt__)) _U2TXInterrupt (void)
{
	MotorTXIndex++;		// Increment the index.
	MotorTXSize--;		// Decrement size.

	// Check to see if we are done sending data.
	if(MotorTXSize == 0)
		IEC1bits.U2TXIE = 0;	// If we are disable the interrupt.
	else
		U2TXREG = MotorTX[MotorTXIndex];	// If not then write the next byte.
	
	IFS1bits.U2TXIF = 0;	// Clear the interrupt.
}

// This function will simply block waiting until we have the mutex.
// We will set the WaitForMutex flag which will tell the RX ISR
// that we want the mutex. Otherwise the ISR would just pass the
// mutex on to the next node.
void GetMutex()
{
	WaitForMutex = 1;	// Set a flag to indicate we want the mutex.
	while(!Mutex);		// Wait until we get the mutex.
	WaitForMutex = 0;	// Clear the flag. Otherwise we will grab the next mutex.
}

// This function will send the mutex to the next node. We will also clear
// the Mutex variable which tells our system that we currently have the mutex.
char ReleaseMutex()
{
	char DestAdd;

	DestAdd = MYADD+1;				// Compute the next address
	if(DestAdd == TooFar)			// to send the mutex to.
		DestAdd = EarthSat;

	TransData.DestAddLSB = DestAdd;	// Send the mutex to next XBee module.	
	TransData.Data[0] = 0xFF;		// Mutex command.

	if(XBWriteAPI() == 0)			// Try to write the release command.
	{
		Mutex = 0;					// If successful then clear the Mutex flag,
		LEDOff(LED2);				// turn off the LED, and return 0.
		return 0;
	}
	else
		return -1;					// If unsuccessful then return -1.
}

// This function will write the TransData struct to the XB module.
// It will check to see that the module is ready to receive
// data. If not then we will return with a value of -1.
// We will also check to see if we are currently sending
// data to the XB module. If we are then we will return 
// a value of -1. In addition we will check the status packet
// that comes back and if it was not successful then we will
// return a value of -1 .A return value of -1 indicates that we 
// can't write the buffer right now and control loop should 
// try sometime later. 
//
// This function should be called when we are in the API
// mode. It make some assumptions about what the TransData
// struct looks like and thus if there is a change to the
// struct this function should be verified for functionality.
// What's nice about this function is that the main loop only
// needs to define the destination address and the data. The
// rest was defined in the init function and the checksum
// is computed here. This should make the code in the main
// loop easier to read and write.
char XBWriteAPI()
{
	char index;
	unsigned char checksum = 0;

	// Check to see if we are currently writing something.
	if(XBTXSize == 0)
		XBTXSize = sizeof(TransData);	// If not then set the global size
	else								// variable to the size of the TX struct
		return -1;						// If we are then return -1.

	// Check to see if the CTS line is asserted.
	if(PORTFbits.RF6 == 1)	
		return -1;			// If not then return with a -1.

	// Compute the checksum.
	for(index=0;index<8;index++)
		checksum += TransData.Data[index];
	checksum += TransData.ID + TransData.DestAddMSB + TransData.DestAddLSB;
	checksum += TransData.FrameID + TransData.Options;
	checksum = 0xFF - checksum;
	TransData.Checksum = checksum;
	
	// Copy the TransData struct to XBTX buffer.
	memcpy(XBTX, &TransData, sizeof(TransData));

	// Clear the XBTXIndex.
	XBTXIndex = 0;
	
	// Clear the status flag.
	GotStatus = 0;
	
	// Setup the U1TX interrupt.
	IFS0bits.U1TXIF = 0;	// First clear the interrupt flag.
	IEC0bits.U1TXIE = 1;	// Enable the interrupt.
	U1TXREG = XBTX[0];		// Write the first byte.	

	// Wait for the status byte to come.
	while(GotStatus == 0);

	// Clear the status flag.
	GotStatus = 0;

	// Check the status. If it didn't equal zero then something
	// went wrong and we should resend.
	if(StatData->Status != 0x00)
		return -1;
		
	return 0;
}

// This function will write the ptr array to the XB module.
// It will check to see that the module is ready to receive
// data. If not then we will return with a value of -1.
// We will also check to see if we are currently sending
// data to the XB module. If we are then we will return 
// a value of -1. A return value of -1 indicates that
// we can't write the buffer right now and control loop
// should try sometime later.
//
// This is a general purpose XBWrite function that will
// write an arbitrary data. Writing to the API could use
// this but it would be easier use the XBWriteAPI function.
char XBWrite(char *ptr, char size)
{
	char index;

	// Check to see if we are currently writing something.
	if(XBTXSize == 0)
		XBTXSize = size;	// If not then set the global size
	else					// variable to the size arguement.
		return -1;			// If we are then return -1.

	// Check to see if the CTS line is asserted.
	if(PORTFbits.RF6 == 1)	
		return -1;			// If not then return with a -1.

	// Copy the ptr buffer to XBTX buffer.
	for(index = 0;index < size;index++)
		XBTX[index] = ptr[index];

	// Clear the XBTXIndex.
	XBTXIndex = 0;

	// Setup the U1TX interrupt.
	IFS0bits.U1TXIF = 0;	// First clear the interrupt flag.
	IEC0bits.U1TXIE = 1;	// Enable the interrupt.
	U1TXREG = XBTX[0];		// Write the first byte.	

	return 0;
}

// This is the ISR that gets called when the UART can receive
// another transmit byte. We use this function to send data to
// the XB module. This interrupted method is better because we
// don't block the CPU from doing something else as before when
// we just sat there waiting for the transmit to complete.
void __attribute__((__interrupt__)) _U1TXInterrupt (void)
{
	XBTXIndex++;	// Increment the index.
	XBTXSize--;		// Decrement size.

	// Check to see if we are done sending data.
	if(XBTXSize == 0)
		IEC0bits.U1TXIE = 0;		// If we are disable the interrupt.
	else
		U1TXREG = XBTX[XBTXIndex];	// If not then write the next byte.
	
	IFS0bits.U1TXIF = 0;	// Clear the interrupt.
}

// This function will read from the XB module and store the
// result in the array. Note that this function determines
// we are done with the reception based on the return character.
// Therefore, this function should only be called during AT mode.
// The interrupt will enable once we reach API mode which will
// render this function basically useless.
char XBRead(char *ptr)
{
	char index;
	char ch = 0;

	// Read the values from the XB module and store
	// them in an array.
	for(index = 0;ch != '\r';index++)
	{
		while(U1STAbits.URXDA == 0);	// Wait for data.
		ch = U1RXREG;					// Store the value.		
		ptr[index] = ch;
	}

	// Return the number of bytes read.
	return index;
}

// This is the ISR that gets called when the UART receives a byte
// from the XBee module. We will fill the receive buffer based
// on the size of the message (located in the 3rd byte of the
// received message). We will then set RTS so we don't get any
// more bytes and set a flag for the main loop to take action.
//
// We will determine within the ISR if the received data
// is an TX Status message in which case we will handle that
// message here in the ISR.
//
// In addition we will verify that the message received is
// good based on the checksum.
void __attribute__((__interrupt__)) _U1RXInterrupt (void)
{
	unsigned char checksum = 0;
	unsigned char index;

	XBRX[XBRXIndex] = U1RXREG;	// Store the byte in the global buffer.

	// Check that the first byte is the delimiter. If
	// it is not then turn on the error LED.
	if(XBRXIndex == 0)
	{			
		if(XBRX[XBRXIndex] != 0x7E)
		{
			LEDOn(LED6);
		}
	}

	// The third byte will tell us how large the message will be.
	if(XBRXIndex == 2)
		XBRXSize = XBRX[XBRXIndex] + 4;
			
	XBRXIndex++;			// Increment the index.

	// Check to see if we are done.
	if(XBRXSize == XBRXIndex)
	{
		XBRXIndex = 0;		// If we are then reset the index.
	
		// Check to see if the message is TX status message. If it is
		// then the delimiter, checksum, and status.
		if(XBRX[3] == 0x89)
		{
			// Check the delimiter.
			// If the delimiter is not 0x7E then turn on Error LED.
			if(StatData->Delimiter != 0x7E)
			{
				LEDOn(LED6);
			}

			// Check checksum.
			// If the checksum is incorrect then turn on Error LED.
			checksum = StatData->ID + StatData->FrameID + StatData->Status + StatData->Checksum;
			if(checksum != 0xFF)
			{
				LEDOn(LED4);
			}

			// Check that the status is OK.
			// If not then turn on an LED.
			if(StatData->Status != 0x00)
			{
				LEDOn(LED3);
			}

			// No matter what we will message the WriteAPI function that
			// we got a status message
			GotStatus = 1;	
		}

		// If the message if not an TX status message then
		// it is a message from another XB module. Check the delimiter
		// and checksum. Then check to see if it is a mutex. If don't
		// need the mutex then schedule it for release.
		else
		{
			// Copy the XBRX buffer to the RecData struct.
			memcpy(&RecData, XBRX, sizeof(RecData));

			// Check the delimiter.
			// If the delimiter is not 0x7E then turn on Error LED.
			if(RecData.Delimiter != 0x7E)
			{
				LEDOn(LED6);
			}

			// Check checksum.
			// If the checksum is incorrect then turn on Error LED.
			for(index = 0;index < 8;index++)
				checksum += RecData.Data[index];
			checksum += RecData.ID + RecData.SourceAddMSB + RecData.SourceAddLSB;
			checksum += RecData.RSSI + RecData.Options + RecData.Checksum;
			if(checksum != 0xFF)
			{
				LEDOn(LED5);
			}

			// Check to see if the message was a mutex.
			if(RecData.Data[0] == 0xFF)
			{
				LEDOn(LED2);		// Turn on an LED so we can see the mutex get passed around.

				// Determine if this is the first time we have received a mutex. If it is
				// then we will erase the EEPROM. Else determine if we want the mutex.
				// If we don't want the mutex then flag it for release in the main loop.
				if(FirstTimeMutex == 0)		// If this is the first mutex we see flag
					FirstTimeMutex = 1;		// an erase for the EEPROM.	
				else if(WaitForMutex)		// Check to see if we even want to mutex.
					Mutex = 1;				// If we do set a variable.
				else
					ReleaseMtx = 1;			// Else flag the release of the mutex.
			}
			else
			{
				if(XBMessRec == 1)	// Check to see if XBMessRec is clear. If not this
				{					// indicates that the main loop have not taken care
					LEDOn(LED6);	// of the previous RX packet and there is an error.
				}		

				// Set a flag to store data we get from the XBee module in the EEPROM.
				if(RecEEPROM == 0)
				{
					// Check to see if the last RecData packet was written. If not
					// then turn on an LED to indicate error.
					if(WriteEEPROMRX == 1)
					{
						LEDOn(LED6);
					}					

					WriteEEPROMRX = 1;
				}			

				XBMessRec = 1;		// If not a mutex then set a flag to 
									// take action in the main loop.
			}
		}
	}
	
	IFS0bits.U1RXIF = 0;	// Clear the interrupt flag.
}

// ISR for CN interrupt. This will be called whenever
// the pushbuttons are push or released.
void __attribute__((__interrupt__)) _CNInterrupt (void)
{	
	unsigned char IRDA;
	char string[32];
	IFS1bits.CNIF = 0;		// Clear the interrupt flag.

	// Set the message flags so that the main loop
	// can handle the message.
	if(PORTDbits.RD6 == 0)
		Button1Pressed = 1;

	if(PORTDbits.RD5 == 0)
		Button2Pressed = 1;

}

// Initialize the dsPIC. We will init the clock, IO pins,
// push buttons, EEPROM, DMA for AtoD transfer, AtoD which
// is the IR sensors and battery monitor, Timers, Serial to
// the motor controller, Motor controller, IRDA, XB
// module, and the message flags.
void Init()
{
	char index,index2;
	char buffer[8];

	// ---------------------------------------------------------
	// Initialize clock.
	// ---------------------------------------------------------
	// This will configure the PLL so that we run at 40 MIPS.
	CLKDIVbits.PLLPRE = 0;	
	PLLFBDbits.PLLDIV = 0x1E;
	CLKDIVbits.PLLPOST = 0;

	// Wait for the PLL to lock.
	while(OSCCONbits.LOCK == 0);

	// ---------------------------------------------------------
	// Initialize IO pins.
	// ---------------------------------------------------------
	// PORTB pins 0-4 are analog inputs.
	AD1PCFGL = 0xFFE0;
	AD1PCFGH = 0xFFFF;	

	// Set the inputs and outputs for PORTB.
	TRISB = 0x80FF;
	PORTB |= 0x7f00;	// Turn LEDs off.

	// Set the inputs and outputs for PORTD.
	PORTD = 0x0009;		// Hold MCP2122 in reset.
						// Hold IRDA in shutdown.
						// Hold RTS to XB low (asserted).
	TRISD = 0xFFE2;		// Term, IR serial, and RTS output.

	// Set the inputs and outputs for PORTF.
	TRISF = 0xFFD5;
	PORTFbits.RF1 = 0;	// Trigger for UltraSonic.
	asm("NOP");
	PORTFbits.RF3 = 1;	// XBee serial output.
	asm("NOP");
	PORTFbits.RF5 = 1;	// Motor serial output.

	// Set the inputs and outputs for PORTG.
	TRISG = 0xFFFc;
	PORTGbits.RG0 = 0;	// Hold motors in reset.
	asm("NOP");
	PORTGbits.RG1 = 0;	// Unprotect EEPROM.

	// ---------------------------------------------------------
	// Initialize interrupts
	// ---------------------------------------------------------
	INTCON1bits.NSTDIS = 1;	// We don't want nested interrupts.

	// ---------------------------------------------------------
	// Initialize the CN pins for push button interrupt.
	// ---------------------------------------------------------
	CNEN1 = 0xC000;			// Enable CN14 and CN15.
	IEC1bits.CNIE = 1;		// Enable CN interrupts.

	// ---------------------------------------------------------
	// Initialize the I2C for the EEPROM.
	// ---------------------------------------------------------
	I2C1BRG = 0x3f;			// 400K BAUD rate.
	IEC1bits.MI2C1IE = 1;	// Enable interrupt.
	I2C1CONbits.I2CEN = 1;	// Turn I2C on.

	// ---------------------------------------------------------
	// Configure the DMA. This will be configured such that the
	// AtoD data will be stored in the DMA buffer as they are
	// sampled. This is helpful in that there is no CPU
	// intervention needed during this time.
	// ---------------------------------------------------------
	DMA0REQ = 13;			// IRQ number set to ADC1.
	DMA0CNT = 4;			// Indicates we want 5 DMA transfer at time.

	// Load the PAD register with the address of the AtoD conversion
	// result register.
	DMA0PAD = (volatile unsigned int) &ADC1BUF0;
	DMA0CONbits.CHEN = 1;	// Enable (turn on) the DMA channel.	

	// ---------------------------------------------------------
	// Configure the AtoD.
	// Here is the math for this. We set the sample time for 
	// the AtoD to 1 TAD. The conversion time is determined by
	// the controller and is 12 TAD. So each conversion will
	// take 13 TAD. 1 TAD was defined by us to be 1.6us. Thus,
	// One channel is converted every 20.8us (or ~48K samples
	// per second). Thus, five channels will be converted every
	// 104us (or 9615 times a second).
	// ---------------------------------------------------------
	AD1CON1bits.SSRC = 7;		// Internal counter ends sampling.
	AD1CON1bits.ASAM = 1;		// Sampling begins immediatly after last conversion.
	AD1CON2bits.SMPI = 4;		// Increment DMA address after 5 conversions.
	AD1CON2bits.CSCNA = 1;		// Scan inputs.
	AD1CON3bits.ADCS = 0x3f;	// TAD = 64 * Tcy = 64 * 25ns = 1.6us
	AD1CON3bits.SAMC = 1;		// 1 TAD for Sample time.
	AD1CSSL = 0x1f;				// Input scan channels 0-4.
	AD1CON1bits.ADON = 1;		// Turn AtoD on.

	// --------------------------------------------------------
	// Configure the serial port for motor communication.
	// We will talk to the motors serially at 9600 BAUD.
	// First we will need to take the motor controller out of
	// reset and then wait 100ms.
	// --------------------------------------------------------
	// First set up the serial port.
	U2BRG = 259;			// Set BAUD rate generator to send at 9600.
	U2MODEbits.UARTEN = 1;	// Enable UART.
	U2STAbits.UTXEN = 1;	// Enable transmitter.
	
	// Clear the global size variable for the motor
	MotorTXSize = 0;

	// Take motors out of reset.
	PORTGbits.RG0 = 1;		// Take the motors out of rest.

	// Set up timer to wait for 100ms.
	PR3 = 0x3D09;			// Compared value.
	T3CONbits.TCKPS = 3;	// Set prescaler to 256;
	TMR3 = 0;				// Clear the timer.
	IFS0bits.T3IF = 0;		// Clear the interrupt.
	T3CONbits.TON = 1;		// Turn the timer on.
	while(IFS0bits.T3IF == 0);	// Wait for the timer.
	T3CONbits.TON = 1;		// Turn the timer off.
	IFS0bits.T3IF = 0;		// Clear the interrupt.
	TMR3 = 0;				// Clear the timer.

	// -------------------------------------------------------
	// Now we will take the IRDA module out of reset and
	// setup Timer5 for timeout usage.
	// -------------------------------------------------------
	PORTDbits.RD4 = 1;		// Take IRDA out of reset.
	PR5 = 0xFFFF;			// Compared value for Timer5.
	T5CONbits.TCKPS = 2;	// Set prescalar to 64.
	
	// ---------------------------------------------------------
	// Initialize the UART for communication with the XB module.
	// Then we will initialize the XB module itself.
	// ---------------------------------------------------------
	// Set up the serial port.
	U1BRG = 259;			// Set BAUD rate generator to send at 9600.
	U1MODEbits.UARTEN = 1;	// Enable UART.
	U1STAbits.UTXEN = 1;	// Enable transmitter.

	// Set up the XB module.
	// Set the global size variables to zero.
	XBTXSize = 0;
	XBRXSize = 0;

	// First get into AT mode.
	XBWrite("+++", 3);		// Get into AT mode by +++.
	XBRead(XBRX);			// Get the OK or ERROR.
	if(strncmp(XBRX, "OK", 2) != 0)	// If we don't get OK then
	{								// turn on an LED.
		LEDOn(LED6);
	}				

	// Define our address based on what robot we are.
	sprintf(XBTX, "ATMY %i\r", MYADD);
	XBWrite(XBTX, 7);
	XBRead(XBRX);
	if(strncmp(XBRX, "OK", 2) != 0)
	{
		LEDOn(LED6);
	}	

	// We want RTS flow control.
	sprintf(XBTX, "ATD6 1\r");
	XBWrite(XBTX, 7);
	XBRead(XBRX);
	if(strncmp(XBRX, "OK", 2) != 0)
	{
		LEDOn(LED6);
	}

	// We want CCA threshold to be 0x50.
	sprintf(XBTX, "ATCA 50\r");
	XBWrite(XBTX, 8);
	XBRead(XBRX);
	if(strncmp(XBRX, "OK", 2) != 0)
	{
		LEDOn(LED6);
	}

	// Tell XB module we want to be in API mode.
	sprintf(XBTX, "ATAP 1\r");
	XBWrite(XBTX, 7);
	XBRead(XBRX);
	if(strncmp(XBRX, "OK", 2) != 0)
	{
		LEDOn(LED6);
	}
	
	// Explicitly leave the AT mode.
	sprintf(XBTX, "ATCN\r");
	XBWrite(XBTX, 5);
	XBRead(XBRX);
	if(strncmp(XBRX, "OK", 2) != 0)
	{
		LEDOn(LED6);
	}

	// Enable the U1RX interrupt now because we will never
	// use the XBRead function again and all U1RX data will
	// now be handled by the ISR.
	IFS0bits.U1RXIF = 0;	// Clear the interrupt flag.
	IEC0bits.U1RXIE = 1;	// Enable the U1RX interrupt.

	// --------------------------------------------------------
	// Configure the 16bit timer.
	// This timer will be set to fire 100 times a second. It
	// will be used to look at the AtoD values to determine if
	// we see a line. While we are at it we will check the
	// battery voltage as well.
	// At 40 MIPS we need 0x61A80 clock ticks for 10ms. If we
	// have a prescaler of 64 then the timer value needs
	// to be 0x186A.
	// --------------------------------------------------------
	PR2 = 0x186A;			// Compared value.
	T2CONbits.TCKPS = 2;	// Set prescaler to 64.
	TMR2 = 0;				// Clear the timer.
	IEC0bits.T2IE = 1;		// Enable the interrupt.
	T2CONbits.TON = 1;		// Turn the timer on.

	// --------------------------------------------------------
	// Clear the global variables that are the messages to the
	// main loop.
	// --------------------------------------------------------
	Button1Pressed = 0;
	Button2Pressed = 0;
	XBMessRec = 0;
	MotorOn = 0;
	MotorState = 0;
	Timer2Fired = 0;
	Mutex = 0;
	WaitForMutex = 0;
	ReleaseMtx = 0;
	GotStatus = 0;
	FirstTimeMutex = 0;
	FirstTimeMutexDone = 0;
	EEPROMAddress = 0;
	WriteEEPROMRX = 0;

	// --------------------------------------------------------
	// Initialize the TX data structs.
	// --------------------------------------------------------
	TransData.Delimiter = 0x7E;
	TransData.SizeMSB = 0;
	TransData.SizeLSB = 13;
	TransData.ID = 1;
	TransData.FrameID = 1;
	TransData.DestAddMSB = 0;
	TransData.DestAddLSB = 0;
	TransData.Options = 0;
	TransData.Checksum = 0;
	for(index = 0;index < 8;index++)
		TransData.Data[index] = 0;

	// --------------------------------------------------------
	// Initialize the Robot structs.
	// --------------------------------------------------------
	Robot[0].TypeOfRobot = Seeker;
	Robot[1].TypeOfRobot = Seeker;
	Robot[2].TypeOfRobot = Reader;
	Robot[3].TypeOfRobot = Reader;

	Robot[0].CurrentLocation = 2;
	Robot[1].CurrentLocation = 3;
	Robot[2].CurrentLocation = 1;
	Robot[3].CurrentLocation = 4;

	for(index = 0;index < 4;index++)
	{
		Robot[index].Status = OK;
		Robot[index].Confused = 0;
	}

	Robot[0].Address = Seeker1;
	Robot[1].Address = Seeker2;
	Robot[2].Address = Reader1;
	Robot[3].Address = Reader2;

	// --------------------------------------------------------
	// Initialize the Ball structs.
	// --------------------------------------------------------
	for(index = 0;index < 8;index++)
	{
		Ball[index].BallLetter = 0x41 + index;
		Ball[index].IRData = 0;
		Ball[index].Status = Unknown;
		Ball[index].RelayStatus = 0;
		Ball[index].RelayData = 0;
		Ball[index].Robot = 0;
		for(index2 = 0;index2 < 4;index2++)
		{
			Ball[index].RelayStatRobot[index2] = 0;
			Ball[index].RelayDataRobot[index2] = 0;
		}
	}

	// --------------------------------------------------------
	// Determine our EEPROM address.
	// --------------------------------------------------------
	EEPROMAddress = 0;						// Initially set address to zero.
	while(EEPROMAddress < 8192)				// Won't go more than 8192 bytes.
	{
		while(ReadEEPROM(EEPROMAddress, buffer, 8));	// Read the next eight bytes.
		if(buffer[0] == 0)					// If we read 0x00 in the first byte
			break;							// Then we are done with list.
		else
			EEPROMAddress += 8;				// Otherwise increment and check the next set.
	}

	if(EEPROMAddress == 8192)
		EEPROMAddress = 0;

	// --------------------------------------------------------
	// Now we will turn on LED0 to indicate that the dsPIC was
	// initialized properly.
	// --------------------------------------------------------
	LEDOn(LED0);
	
}
