/**************************************************************
* Herein lies the main loop for the robot. This file will 
* contain the code for the timer2 interrupt which tells us
* when we should look at the IR sensors for black lines.
* It will also contain the main loop which will process 
* messages that have been flagged by various ISRs.
*
* Written by: Shane Avery
****************************************************************/
#include<p33FJ128GP206.h>
#include "Robot.h"
#include "IOFuncs.h"
#include "HumanInt.h"

// -------------------------------------------------------
// Here are the declarations for the global extern
// variables in the Robot.h header file.
// -------------------------------------------------------
#ifdef RED1
const char MYADD = 3;	// XB address.
const char MYARG = 0;	// Struct index.
#endif

#ifdef RED2
const char MYADD = 4;	// XB address.
const char MYARG = 1;	// Struct index.
#endif

#ifdef BLUE1
const char MYADD = 5;	// XB address.
const char MYARG = 2;	// Struct index.
#endif

#ifdef BLUE2
const char MYADD = 6;	// XB address.
const char MYARG = 3;	// Struct index.
#endif

// Here are the const definitions for the LEDs.
const unsigned int LED0 = 0x100;	// dsPIC Initialized OK.
const unsigned int LED1 = 0x200;	// Robot status complete.
const unsigned int LED2 = 0x400;	// We have the mutex.
const unsigned int LED3 = 0x800;
const unsigned int LED4 = 0x1000;
const unsigned int LED5 = 0x2000;	
const unsigned int LED6 = 0x4000;	// General Purpose Error (red).
const unsigned int LEDAll = 0x7F00;

// Here are the const definitions for motor directions.
const char Forward = 1;
const char Reverse = 0;
const char Left = 3;
const char Right = 2;

// Global buffers for XBTX and XBRX.
unsigned char XBTX[17];
unsigned char XBRX[17];
unsigned char XBTXSize;
unsigned char XBRXSize;
unsigned char XBTXIndex;
unsigned char XBRXIndex;

// Global buffers and variables for Motor Controller.
unsigned char MotorTX[4];
unsigned char MotorTXSize;
unsigned char MotorTXIndex;

// Global buffers for I2C EEPROM.
unsigned char I2CMSB,I2CLSB;
unsigned char I2CBuffer[64];
unsigned char I2CRW = 0;
unsigned char I2CDone = 1;
unsigned char I2CSize;

// Set the StatusData pointer variable equal to
// the address of the XB Receiver buffer.
struct StatusData *StatData = (struct StatusData *)XBRX;
struct ReceivedData RecData;
struct TransmitData TransData;

// Here are the structs to hold data about all robots and balls.
struct RobotStruct Robot[4];
struct BallStruct Ball[8];

// Global buffer that define the messages to the main loop.
char Button1Pressed = 0;
char Button2Pressed = 0;
char XBMessRec = 0;
char MotorOn = 0;
int  MotorState = 0;
char Timer2Fired = 0;
char Mutex = 0;
char WaitForMutex = 0;
char ReleaseMtx = 0;
char GotStatus = 0;
char FirstTimeMutex = 0;
char FirstTimeMutexDone = 0;
int EEPROMAddress = 0;
char RecEEPROM = 0;
char WriteEEPROMRX = 0;

// ---------------------------------------------------------
// End of extern global variable definitions
// ---------------------------------------------------------

// These defines make the code easier to read and write.
// AtoD[0] = OutsideRight
// AtoD[1] = OutsideLeft
// AtoD[2] = InsideRight
// AtoD[3] = InsideLeft
#define OutRight AtoD[0]
#define OutLeft AtoD[1]
#define InRight AtoD[2]
#define InLeft AtoD[3]
#define SeesBlack > 0x120
#define SeesAllBlack > 0x1c0
#define SeesWhite < 0x100
#define SeesAllWhite < 0x70
#define SeesSomeWhite < 0x180

// This will define the threshold value for if there is a ball there
// or not.
#define USThreshold 0x280

// Using these defines helps us grow and shrink the states without
// having to constantly renumber them. Don't forget that when we
// make a change here we need to make a change to the corresponding
// command defines in robot.h that are used to determine what action
// action to take in the decision array.
#define StraightState 1
#define DecisionState 2
#define LeftState 40
#define RightState 10
#define CorrectLState 100
#define CorrectRState 70
#define FlipState 130
#define FindBallState 200
#define ReadBallState 230
#define StopState 500
#define WaitState 504

// ISR for Timer2. This will be called 100 times a second.
// We will check the battery voltage and set a flag for
// the main loop. We could just poll for the interrupt
// in the main loop but this gives us the flexability
// to do something within the ISR if we need to.
void __attribute__((__interrupt__)) _T2Interrupt (void)
{
	static unsigned char count = 0;
	unsigned int *AtoD = (unsigned int *) &_DMA_BASE;

	IFS0bits.T2IF = 0;		// Clear the interrupt flag.

	Timer2Fired = 1;		// Set the TimerFired flag.

	// Check to battery voltage and turn on the red LED
	// to indicate that it is time to change the battery.
	if(AtoD[4] < 0x327)
		LEDOn(LED6);
}

// This is the main loop for the system. Here we will first
// initialize the dsPIC and its IO and then fall into process
// loop which will process message flags set by interrupts.
int main()
{
	unsigned char temp;

	#ifdef RED1
	unsigned char LeftSpeed = 0x48;
	unsigned char RightSpeed = 0x48;
	#endif

	#ifdef RED2
	unsigned char LeftSpeed = 0x48;
	unsigned char RightSpeed = 0x4c;
	#endif

	#ifdef BLUE1
	unsigned char LeftSpeed = 0x48;
	unsigned char RightSpeed = 0x48;
	#endif

	#ifdef BLUE2
	unsigned char LeftSpeed = 0x48;
	unsigned char RightSpeed = 0x40;
	#endif

	unsigned char FastSpeed = 0x60;
	static unsigned char count = 0;
	unsigned int index = 0;
	unsigned char IgnoreFlag = 0;
	unsigned int *AtoD = (unsigned int *) &_DMA_BASE;
	unsigned char array[64];
	unsigned char MotorStateBallIndex = 0;
	char MotorStateBallArray[4] = {0,1,2,3};
	unsigned char BallArrayIndex = 0;
	unsigned char BallArrayCounter = 0;
	unsigned char BallDecIndex = 0;

	int counter1,counter2,counter3;
	unsigned char buffer[8];
	int size;
	unsigned char XArg;
	unsigned int EEPROMRobotAddress;

	unsigned char UpdateLocation = 0;
	unsigned char UpdateLocState = 0;
	unsigned char LocationIndex = 0;
	unsigned char RelayUpdateLocStateS1 = 0,RelayUpdateLocStateS2 = 0;
	unsigned char RelayUpdateLocStateR1 = 0,RelayUpdateLocStateR2 = 0;
	unsigned char RelayUpdateLocSeeker1 = 0,RelayUpdateLocSeeker2 = 0;
	unsigned char RelayUpdateLocReader1 = 0,RelayUpdateLocReader2 = 0;
	unsigned char RelayLocNoSatCOMS1[2] = {0xff,0xff},RelayLocNoSatCOMS2[2] = {0xff,0xff};
	unsigned char RelayLocNoSatCOMR1[2] = {0xff,0xff},RelayLocNoSatCOMR2[2] = {0xff,0xff};
	unsigned char RelayLocNoSatCOMStateS1 = 0,RelayLocNoSatCOMStateS2 = 0;
	unsigned char RelayLocNoSatCOMStateR1 = 0,RelayLocNoSatCOMStateR2 = 0;

	unsigned char UpdateBallThere = 0;
	unsigned char UpdateBTState = 0;
	char BallToUpdate = 'A';
	unsigned char BallTUFound = 0;
	unsigned char UpdateBTRelay = 0;
	unsigned char UpdateBTRState = 0;
	unsigned char UpdateBTRIndex = 0;
	unsigned char UpdateBTRRelay = 0;
	unsigned char UpdateBTRRState = 0;
	unsigned char UpdateBTRRIndex = 0;
	unsigned char UpdateBTRRIndex2 = 0;

	unsigned char UpdateBallData = 0;
	unsigned char UpdateBDState = 0;
	unsigned char BallIRData = 0x69;
	unsigned char UpdateBDRelay = 0;
	unsigned char UpdateBDRState = 0;
	unsigned char UpdateBDRIndex = 0;
	unsigned char UpdateBDRRelay = 0;
	unsigned char UpdateBDRRState = 0;
	unsigned char UpdateBDRRIndex = 0;
	unsigned char UpdateBDRRIndex2 = 0;

	unsigned char GoFlag = 0;
	unsigned char GoState = 0;
	unsigned char RelayGoIndex = 0;
	unsigned char RelayGoFlag = 0;
	unsigned char RelayGoState = 0;

	unsigned char StopFlag = 0;
	unsigned char StopStateM = 0;
	unsigned char RelayStopFlag = 0;
	unsigned char RelayStopState = 0;
	unsigned char RelayStopRobot = 0;
	
	unsigned char ResumeFlag = 0;
	unsigned char ResumeState = 0;
	unsigned char RelayResumeFlag = 0;
	unsigned char RelayResumeState = 0;
	unsigned char RelayResumeRobot = 0;

	unsigned int  DecArray[32];
	unsigned char LocArray[32];

	char string[32];
	
	Init();

	// Main loop will process messages flagged from
	// the ISRs. We could just do interrupt polling
	// and it would be the same thing but this allows
	// us to do work in the ISR if we so wish to.
	while(1)
	{
		// --------------------------------------------------------------------
		// When EarthSat wants us to resume we will set a flag which will land
		// us here.
		// --------------------------------------------------------------------
		if(ResumeFlag)
		{
			// State machine will start both motors.
			switch(ResumeState)
			{
				// Case 0 will try to start the left motor. If we fail then we will
				// just break out to the main loop which will find its way here
				// to try again later. If we succeed then try the right motor.
				case 0:	
						if(CommandMotor(Left, Forward, LeftSpeed) == 0)	// Start to start Left motor.
							ResumeState = 1;		// If successful try the right motor.
						break;
	
				// Case 1 will try to start the right motor. If we fail then we will
				// just break out to the main loop which will find its way here
				// to try again later. If we succeed put MotorState in the
				// StaightState state and then we are done.
				case 1:	if(CommandMotor(Right, Forward, RightSpeed) == 0)	// Try to start Right motor.
						{
							ResumeState = 0;	// If successful then reset state machine.
							ResumeFlag = 0;		// Clear the flag that got us here.
							MotorState = StraightState;		// Put MotorState in StraightState
						}
						break;
			}
		}

		// --------------------------------------------------------------------
		// When EarthSat wants us to stop we will set a flag which will land
		// us here.
		// --------------------------------------------------------------------
		if(StopFlag)
		{
			// State machine will reverse stop both motors.
			switch(StopStateM)
			{
				// Case 0 will put the MotorState state machine in a wait state
				// and then try to stop the left motor. If we fail then we will
				// just break out to the main loop which will find its way here
				// to try again later. If we succeed then try the right motor.
				case 0:	MotorState = WaitState;		// Put MotorState in WaitState
						if(CommandMotor(Left, Reverse, 0x00) == 0)	// Try to stop Left motor.
							StopStateM = 1;		// If successful try the right motor.
						break;
	
				// Case 1 will try to stop the right motor. If we fail then we will
				// just break out to the main loop which will find its way here
				// to try again later. If we succeed then we are done.
				case 1:	if(CommandMotor(Right, Reverse, 0x00) == 0)	// Try to stop Right motor.
						{
							StopStateM = 0;		// If successful then reset state machine.
							StopFlag = 0;		// Clear the flag that got us here.
						}
						break;
			}
		}

		// -----------------------------------------------------------------------
		// When MarsSat tells us to go or we decide it for ourselves we get here.
		// -----------------------------------------------------------------------
		if(GoFlag)
		{
			// This state machine will turn on the Motors if we are a Seeker robot.
			switch(GoState)
			{
				// Determine if we can move. If not then we are done.
				case 0:	if(Robot[MYARG].Status == NoMove)
							GoFlag = 0;		// Clear the flag that got us here.
						else
							GoState = 1;	// Turn on the motors.

						break;

				// Case 1 will try to turn on the left motor. If we don't succeed then
				// do nothing so that the main loop will find its way here later to try again.
				case 1:	// Try to turn the left motor on. If we don't succeed then
						// do nothing so that the main loop will find its way here
						// later to try again.
						if(CommandMotor(Left, Forward, LeftSpeed) == 0)
							GoState = 2;	// If we succeed then turn on the right motor.

						break;
	
				// Case 2 will try to turn on the right motor. If we don't succeed then
				// do nothing so that the main loop will find its way here later to try again.
				case 2:	if(CommandMotor(Right, Forward, RightSpeed) == 0)	// Try to turn on motor.
						{
								GoState = 0;	// If we succeed then reset the State Machine.
								GoFlag = 0;		// Clear the flag that got us here.
								MotorOn = 1;	// Set a flag that begins the control state machine.
						}
						break;	
			}
		}

		// If the flag for the first we receive a mutex has been set
		// then check to see if we have erased the EEPROM. If we haven't
		// then erase the EEPROM and release the mutex.
		if(FirstTimeMutex && (FirstTimeMutexDone == 0))
		{
			// Build an array of zeros for clearing the EERPOM.
			for(index = 0;index < 64;index++)	// Clear the array.
				array[index] = 0;

			// Clear the EEPROM.
			for(index = 0;index < 0x100;index++)
				while(WriteEEPROM(index << 6, array, sizeof(array)));

			FirstTimeMutexDone = 1;	// Set the flag so we don't do this again.
			EEPROMAddress = 0;		// Reset the address in case we got data
									// before we got the first mutex.
			ReleaseMtx = 1;			// Release the mutex.
		}

		// We get here when we both want the mutex and have the mutex. Once we
		// are here we will determine what is was we want to send. We put this
		// in the main loop so the robot can continue to move around and not
		// be affected by a blocking function call of some sort.
		if(WaitForMutex && Mutex)
		{
			// --------------------------------------------------------------------
			// If we have received the relay Resume flag we will relay the Resume 
			// command to the robot MarsSat told us to.
			// --------------------------------------------------------------------
			if(RelayResumeFlag)
			{
				// State machine that will build up the TX struct, try to write the
				// TX data to EEPROM, and then send the TX data to the XBee module.
				switch(RelayResumeState)
				{
					// Build up the TX struct and try to write that data to the
					// EEPROM. If we fail then do nothing and break out in which case
					// the main loop will find its way here again to try later. If we
					// succeed then right away try to send the TX message via XBee.
					case 0: // Fill in the TransData struct.
							TransData.DestAddLSB = RelayResumeRobot+3;		// Robot we are sending to.		
							TransData.Data[0] = RRResume;					// Go message.		
							TransData.Data[1] = RelayResumeRobot;			// Robot we are sending to.	
	
							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								RelayResumeState = 1;						// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
								{
									RelayResumeState = 0;		// If successful then reset state machine.
									RelayResumeFlag = 0;		// Clear the flag that got us here.
								}
							}

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine and increment the index to check the next robot.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayResumeState = 0;		// Reset the state machine.
								RelayResumeFlag = 0;		// Clear the flag that got us here.					
							}											

							break;	
				}
			}

			// --------------------------------------------------------------------
			// If we have received the relay Stop flag we will relay the Stop command
			// to the robot MarsSat told us to.
			// --------------------------------------------------------------------
			if(RelayStopFlag)
			{
				// State machine that will build up the TX struct, try to write the
				// TX data to EEPROM, and then send the TX data to the XBee module.
				switch(RelayStopState)
				{
					// Build up the TX struct and try to write that data to the
					// EEPROM. If we fail then do nothing and break out in which case
					// the main loop will find its way here again to try later. If we
					// succeed then right away try to send the TX message via XBee.
					case 0: // Fill in the TransData struct.
							TransData.DestAddLSB = RelayStopRobot+3;		// Robot we are sending to.		
							TransData.Data[0] = RRStop;						// Go message.		
							TransData.Data[1] = RelayStopRobot;				// Robot we are sending to.	
	
							// Now try to write the TX data to the EEPROM. If we fail then do nothing and
							// break out to the main loop. If we succeed then immediatly try to send the
							// message via the XBee module.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed.
								RelayStopState = 1;							// Increment the state machine.
								EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
								if(XBWriteAPI() == 0)						// Try to write to XBee.
								{
									RelayStopState = 0;		// If successful then reset state machine.
									RelayStopFlag = 0;		// Clear the flag that got us here.
								}
							}

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine and increment the index to check the next robot.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayStopState = 0;		// Reset the state machine.
								RelayStopFlag = 0;		// Clear the flag that got us here.					
							}											

							break;	
				}
			}
	
			// --------------------------------------------------------------------
			// If we have received the relay Go flag we will relay the Go command
			// to robots who can't talk to MarsSat.
			// --------------------------------------------------------------------
			else if(RelayGoFlag)
			{
				// State machine that will build up the TX struct, try to write the
				// TX data to EEPROM, and then send the TX data to the XBee module.
				// It will do this for every robot that need the Go command relayed
				// to it.
				switch(RelayGoState)
				{
					// First thing in Case 0 is to determine if we are done. If not then
					// check the indexed robot to see if it needs to be relayed to. If
					// so then build up the TX struct and try to write that data to the
					// EEPROM. If we fail then do nothing and break out in which case
					// the main loop will find its way here again to try later. If we
					// succeed then right away try to send the TX message via XBee.
					case 0: if(RelayGoIndex == 4)	// Check to see if we are done.
							{
								RelayGoIndex = 0;	// Clear the index.
								RelayGoFlag = 0;	// Clear the flag.
								break;
							}

							// Check to see if the indexed robot needs to get relayed to. 
							if(Robot[RelayGoIndex].Status == NoSatCOM)
							{
								// If so then fill in the TransData struct.
								TransData.DestAddLSB = RelayGoIndex + 3;		// Robot we are sending to.
								TransData.Data[0] = RRGo;						// Go message.		
								TransData.Data[1] = RelayGoIndex;				// Robot we are sending to.	
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									RelayGoState = 1;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										RelayGoState = 0;		// If successful then reset state machine.
										RelayGoIndex++;			// Increment the robot index.
									}
								}
							}
							// If the robot doesn't need to get relayed to then increment index so say
							// we can try the next robot.
							else
								RelayGoIndex++;		// Increment the robot relay index.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine and increment the index to check the next robot.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayGoState = 0;	// Reset the state machine.
								RelayGoIndex++;		// Increment the robot relay index.					
							}											

							break;	
				}
			}

			// ---------------------------------------------------------------------
			// If we need to relay the Ball IR data to to another robot then we 
			// set a flag and get here.
			// ---------------------------------------------------------------------
			else if(UpdateBDRRelay)
			{
				// State machine that will check each ball to determine if we should
				// relay its IR data to another Robot.
				switch(UpdateBDRRState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the balls one at a time
					// to determine if we should relay its IR Data. If we should then
					// build up the TX struct and try to write it to the EEPROM. If we
					// fail at then then break out to the main loop which will find its
					// way back here to try again later. If we succeed then try to write
					// to the XBee module.
					case 0:	if(UpdateBDRRIndex2 == 4)		// If we have checked all the robot flags
							{								// for this ball.
								UpdateBDRRIndex2 = 0;		// Clear the robot flag index.
								UpdateBDRRIndex++;			// Increment the ball index.

								if(UpdateBDRRIndex == 8)	// If we have checked all the balls then
								{							// we are done.
									UpdateBDRRelay = 0;		// Clear flag that go us here.
									UpdateBDRRIndex = 0;	// Reset ball index.
									break;
								}
							}

							// Check the robot relay flag of the UpdateBDRRIndex Ball. If the flag is
							// is set then we need to relay the IR data to that robot.
							if(Ball[UpdateBDRRIndex].RelayDataRobot[UpdateBDRRIndex2])	
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = UpdateBDRRIndex2 + 3;	// Robot we are sending to.
								TransData.Data[0] = RRIRData;					// IR Data message.		
								TransData.Data[1] = UpdateBDRRIndex2;			// Robot we are sending to.	
								TransData.Data[2] = UpdateBDRRIndex + 0x41;		// Which ball.
								TransData.Data[3] = Ball[UpdateBDRRIndex].IRData; // What IR Data.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDRRState = 1;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateBDRRState = 0;		// If successful then reset state machine.
										Ball[UpdateBDRRIndex].RelayDataRobot[UpdateBDRRIndex2] = 0;	// Clear the robot relay flag.
										UpdateBDRRIndex2++;			// Increment the robot relay index.
									}
								}
								break;
							}

							// If we should not relay the IR Data of this ball to this robot then increment
							// the robot relay index to check to see if we should relay the IR data to the next
							// robot for this ball.
							else	
								UpdateBDRRIndex2++;	// Increment index to check next robot relay flag in this ball.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine, clear this ball's robot relay flag, and increment the index
					// to check the next robot relay flag in this ball.
					case 1:	if(XBWriteAPI() == 0)
							{
								UpdateBDRRState = 0;	// Reset the state machine.
								Ball[UpdateBDRRIndex].RelayDataRobot[UpdateBDRRIndex2] = 0;	// Clear the robot relay flag.
								UpdateBDRRIndex2++;		// Increment the robot relay index.					
							}											

							break;
						
				}
			}

			// ---------------------------------------------------------------------
			// If we need to relay the Ball IR Data to MarsSat on behalf of another
			// robot then we set a flag and get here.
			// ---------------------------------------------------------------------
			else if(UpdateBDRelay)
			{
				// State machine that will check each ball to determine if we should
				// relay its IR Data to MarsSat.
				switch(UpdateBDRState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the balls one at a time
					// to determine if we should relay its IR Data. If we should then
					// build up the TX struct and try to write it to the EEPROM. If we
					// fail at then then break out to the main loop which will find its
					// way back here to try again later. If we succeed then try to write
					// to the XBee module.
					case 0:	if(UpdateBDRIndex == 8)		// If we are done.
							{
								UpdateBDRelay = 0;		// Clear flag.
								UpdateBDRIndex = 0;		// Reset index.
								break;
							}

							// Check the relay flag of the next ball to determine if we should
							// relay the ball's IR Data.
							if(Ball[UpdateBDRIndex].RelayData)	
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = MarsSat;				// Sending to MarsSat.	
								TransData.Data[0] = RBallDataRelay;			// Message is IR Data relayed.	
								TransData.Data[1] = MYARG;					// Coming from us.
								TransData.Data[2] = Ball[UpdateBDRIndex].Robot;	// Say which robot we send this for.			
								TransData.Data[3] = UpdateBDRIndex + 0x41;		// Which ball.
								TransData.Data[4] = Ball[UpdateBDRIndex].IRData;// What ball Data.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBDRState = 1;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateBDRState = 0;		// If successful then reset state machine.
										Ball[UpdateBDRIndex].RelayData = 0;		// Clear this ball's relay flag.
										UpdateBDRIndex++;						// Increment index to check next ball.
									}
								}
								break;
							}

							// If we should not relay the status of this ball then increment the index to
							// check the relay flag of the next ball.
							else	
								UpdateBDRIndex++;	// Increment index to check next ball.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine, clear this ball's relay flag, and increment the index
					// to check the next ball.
					case 1:	if(XBWriteAPI() == 0)
							{
								UpdateBDRState = 0;						// Reset the state machine.
								Ball[UpdateBDRIndex].RelayData = 0;		// Clear relay flag.
								UpdateBDRIndex++;						// Increment index to try the next
							}											// ball.

							break;
					
				}
			}

			// -------------------------------------------------------------------------------
			// If this robot determines that the ball data and a flag is set and we get here. 
			// -------------------------------------------------------------------------------
			else if(UpdateBallData)
			{
				// This state machine will send the ball data to all the other robots that
				// we can talk to and the MarsSat if we can talk to it.
				switch(UpdateBDState)
				{
					case 0:	// Case 0 is slightly special in that the first thing we do is check
							// to see that we can send to other robots. If not then skip right
							// to sending data to MarsSat. If so then determine if we can send
							// to the first robot and if so then do so.
							if(Robot[MYARG].Status == NoLocalCOM || Robot[MYARG].Status == Dead)
							{
								UpdateBDState = 8;	// If we can't send robots then try to send
								break;				// MarsSat.
							}

							// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Seeker1)
							{
								UpdateBDState = 2;
								break;
							}
							
							// Be sure that Seeker1 can talk to us. If not then move to the next robot.
							if(Robot[Seeker1Arg].Status == NoLocalCOM || Robot[Seeker1Arg].Status == Dead)
							{
								UpdateBDState = 2;
								break;
							}
							
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Seeker1;			// Sending to Seeker1.
							TransData.Data[0] = RRIRData;			// IR Data Message.
							TransData.Data[1] = Seeker1Arg;			// Sending to Seeker1.				
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							TransData.Data[3] = BallIRData;			// What IR Data.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBDState = 2;		// If we succeed move to sending to the next bot.
								else
									UpdateBDState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 1:	if(XBWriteAPI() == 0)
									UpdateBDState = 2; 
							break;
	
					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 2:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Seeker2)
							{
								UpdateBDState = 4;
								break;
							}
							
							// Be sure that Seeker2 can talk to us. If not then move to the next robot.
							if(Robot[Seeker2Arg].Status == NoLocalCOM || Robot[Seeker2Arg].Status == Dead)
							{
								UpdateBDState = 4;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Seeker2;			// Sending to Seeker2.
							TransData.Data[0] = RRIRData;			// IR Data Message.
							TransData.Data[1] = Seeker2Arg;			// Sending to Seeker2.				
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							TransData.Data[3] = BallIRData;			// What IR Data.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBDState = 4;		// If we succeed move to sending to the next bot.
								else
									UpdateBDState = 3;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 3:	if(XBWriteAPI() == 0)
									UpdateBDState = 4; 
							break;
	
					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 4:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Reader1)
							{
								UpdateBDState = 6;
								break;
							}
							
							// Be sure that Reader1 can talk to us. If not then move to the next robot.
							if(Robot[Reader1Arg].Status == NoLocalCOM || Robot[Reader1Arg].Status == Dead)
							{
								UpdateBDState = 6;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Reader1;			// Sending to Reader1.
							TransData.Data[0] = RRIRData;			// IR Data Message.
							TransData.Data[1] = Reader1Arg;			// Sending to Reader1.				
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							TransData.Data[3] = BallIRData;			// What IR Data.
							
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBDState = 6;		// If we succeed move to sending to the next bot.
								else
									UpdateBDState = 5;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 5:	if(XBWriteAPI() == 0)
									UpdateBDState = 6; 
							break;

					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 6:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Reader2)
							{
								UpdateBDState = 8;
								break;
							}
							
							// Be sure that Reader2 can talk to us. If not then move to MarsSat.
							if(Robot[Reader2Arg].Status == NoLocalCOM || Robot[Reader2Arg].Status == Dead)
							{
								UpdateBDState = 8;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Reader2;			// Sending to Reader2.		
							TransData.Data[0] = RRIRData;			// IR Data Message.
							TransData.Data[1] = Reader2Arg;			// Sending to Reader2.				
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							TransData.Data[3] = BallIRData;			// What IR Data.
							
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBDState = 8;		// If we succeed move to sending to MarsSat.
								else
									UpdateBDState = 7;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to MarsSat.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 7:	if(XBWriteAPI() == 0)
									UpdateBDState = 8; 
							break;

					// Here we send the data to MarsSat. First we will check that we can talk to
					// MarsSat. If we can't then set/clear flags and increment the LocationIndex.
					// If we can then build up the TX struct and try to write it to EEPROM. If 
					// unsuccessful then break out and the main loop will come back here and try 
					// again later. If we succeed then immediatly try to write the TX data to the 
					// XBee module.	
					case 8:	if(Robot[MYARG].Status == NoSatCOM || Robot[MYARG].Status == Dead)
 							{	// Get here if we can't talk to MarsSat.
								UpdateBDState = 0;		// Reset this state machine.
								UpdateBallData = 0;	// No loger want to be here to clear that flag.
								break;					// Break to main loop.
							}

							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = MarsSat;			// Sending to MarsSat.
							TransData.Data[0] = RBallData;			// IR Data Message.
							TransData.Data[1] = MYARG;				// Message comes from us.				
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							TransData.Data[3] = BallIRData;			// What IR Data.
	
							// Attempt to write the Transmitted data to the EEPROM. If we are unsuccessful 
							// then we just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)	// If we succeed set/clear flags and increment the LocationIndex.
								{							
									UpdateBDState = 0;		// Reset this state machine.
									UpdateBallData = 0;		// No loger want to be here to clear that flag.
								}
								else
									UpdateBDState = 9;		// If we fail then go to state that continually retries.
							}
							
							break;

					// Retry sending the XBee message. If we succeed then set/clear flags 
					// and increment the LocationIndex. 
					case 9:	if(XBWriteAPI() == 0)
							{
								UpdateBDState = 0;		// Reset this state machine.
								UpdateBallData = 0;		// No loger want to be here to clear that flag.
							}
							break;

				}// End Switch
			}// End If UpdateBallData

			// ---------------------------------------------------------------------
			// If we need to relay the Ball status data (found or NOT found) to
			// to another robot then we set a flag and get here.
			// ---------------------------------------------------------------------
			else if(UpdateBTRRelay)
			{
				// State machine that will check each ball to determine if we should
				// relay its status to another Robot.
				switch(UpdateBTRRState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the balls one at a time
					// to determine if we should relay its status. If we should then
					// build up the TX struct and try to write it to the EEPROM. If we
					// fail at then then break out to the main loop which will find its
					// way back here to try again later. If we succeed then try to write
					// to the XBee module.
					case 0:	if(UpdateBTRRIndex2 == 4)		// If we have checked all the robot flags
							{								// for this ball.
								UpdateBTRRIndex2 = 0;		// Clear the robot flag index.
								UpdateBTRRIndex++;			// Increment the ball index.

								if(UpdateBTRRIndex == 8)	// If we have checked all the balls then
								{							// we are done.
									UpdateBTRRelay = 0;		// Clear flag that go us here.
									UpdateBTRRIndex = 0;	// Reset ball index.
									break;
								}
							}

							// Check the robot relay flag of the UpdateBTRRIndex Ball. If the flag is
							// is set then we need to relay the data to that robot.
							if(Ball[UpdateBTRRIndex].RelayStatRobot[UpdateBTRRIndex2])	
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = UpdateBTRRIndex2 + 3;	// Robot we are sending to.
								if(Ball[UpdateBTRRIndex].Status == There)		// Determine if we send found
									TransData.Data[0] = RRBallFound;			// or NOT found.
								else
									TransData.Data[0] = RRBallNotFound;
								TransData.Data[1] = UpdateBTRRIndex2;			// Robot we are sending to.	
								TransData.Data[2] = UpdateBTRRIndex + 0x41;		// Which ball.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTRRState = 1;						// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateBTRRState = 0;		// If successful then reset state machine.
										Ball[UpdateBTRRIndex].RelayStatRobot[UpdateBTRRIndex2] = 0;	// Clear the robot relay flag.
										UpdateBTRRIndex2++;			// Increment the robot relay index.
									}
								}
								break;
							}

							// If we should not relay the status of this ball to this robot then increment
							// the robot relay index to check to see if we should relay the data to the next
							// robot for this ball.
							else	
								UpdateBTRRIndex2++;	// Increment index to check next robot relay flag in this ball.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine, clear this ball's robot relay flag, and increment the index
					// to check the next robot relay flag in this ball.
					case 1:	if(XBWriteAPI() == 0)
							{
								UpdateBTRRState = 0;	// Reset the state machine.
								Ball[UpdateBTRRIndex].RelayStatRobot[UpdateBTRRIndex2] = 0;	// Clear the robot relay flag.
								UpdateBTRRIndex2++;		// Increment the robot relay index.					
							}											

							break;
						
				}
			}

			// ---------------------------------------------------------------------
			// If we need to relay the Ball status data (found or NOT found) to
			// MarsSat on behalf of another robot then we set a flag and get here.
			// ---------------------------------------------------------------------
			else if(UpdateBTRelay)
			{
				// State machine that will check each ball to determine if we should
				// relay its status to MarsSat.
				switch(UpdateBTRState)
				{
					// First check to see if we are done. If so then clear the flag
					// and reset the index. If not then check the balls one at a time
					// to determine if we should relay its status. If we should then
					// build up the TX struct and try to write it to the EEPROM. If we
					// fail at then then break out to the main loop which will find its
					// way back here to try again later. If we succeed then try to write
					// to the XBee module.
					case 0:	if(UpdateBTRIndex == 8)		// If we are done.
							{
								UpdateBTRelay = 0;		// Clear flag.
								UpdateBTRIndex = 0;		// Reset index.
								break;
							}

							// Check the relay flag of the next ball to determine if we should
							// relay the ball's status.
							if(Ball[UpdateBTRIndex].RelayStatus)	
							{
								// Fill in the TransData struct.
								TransData.DestAddLSB = MarsSat;				// Sending to MarsSat.	
								if(Ball[UpdateBTRIndex].Status == There)	// Determine if we send found
									TransData.Data[0] = RBallFoundRelay;	// or NOT found.
								else
									TransData.Data[0] = RBallNotFoundRelay;
								TransData.Data[1] = MYARG;					// Coming from us.
								TransData.Data[2] = Ball[UpdateBTRIndex].Robot;	// Say which robot we send this for.			
								TransData.Data[3] = UpdateBTRIndex + 0x41;		// Which ball.
	
								// Now try to write the TX data to the EEPROM. If we fail then do nothing and
								// break out to the main loop. If we succeed then immediatly try to send the
								// message via the XBee module.
								if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
								{
									// If we succeed.
									UpdateBTRState = 1;							// Increment the state machine.
									EEPROMAddress += sizeof(TransData.Data);	// Increment the EEPROM address.
									if(XBWriteAPI() == 0)						// Try to write to XBee.
									{
										UpdateBTRState = 0;		// If successful then reset state machine.
										Ball[UpdateBTRIndex].RelayStatus = 0;	// Clear this ball's relay flag.
										UpdateBTRIndex++;						// Increment index to check next ball.
									}
								}
								break;
							}

							// If we should not relay the status of this ball then increment the index to
							// check the relay flag of the next ball.
							else	
								UpdateBTRIndex++;	// Increment index to check next ball.

							break;

					// Retry sending the XBee message. If we fail then just break out to the main loop
					// which will find its way here to try again later. If we succeed then reset
					// the state machine, clear this ball's relay flag, and increment the index
					// to check the next ball.
					case 1:	if(XBWriteAPI() == 0)
							{
								UpdateBTRState = 0;						// Reset the state machine.
								Ball[UpdateBTRIndex].RelayStatus = 0;	// Clear relay flag.
								UpdateBTRIndex++;						// Increment index to try the next
							}											// ball.

							break;
					
				}
			}

			// -------------------------------------------------------------------------------
			// If this robot determines that the ball is either there or not there and a flag
			// is set and we get here. 
			// -------------------------------------------------------------------------------
			else if(UpdateBallThere)
			{
				// This state machine will send the ball status to all the other robots that
				// we can talk to and the MarsSat if we can talk to it.
				switch(UpdateBTState)
				{
					case 0:	// Case 0 is slightly special in that the first thing we do is check
							// to see that we can send to other robots. If not then skip right
							// to sending data to MarsSat. If so then determine if we can send
							// to the first robot and if so then do so.
							if(Robot[MYARG].Status == NoLocalCOM || Robot[MYARG].Status == Dead)
							{
								UpdateBTState = 8;	// If we can't send robots then try to send
								break;				// MarsSat.
							}

							// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Seeker1)
							{
								UpdateBTState = 2;
								break;
							}
							
							// Be sure that Seeker1 can talk to us. If not then move to the next robot.
							if(Robot[Seeker1Arg].Status == NoLocalCOM || Robot[Seeker1Arg].Status == Dead)
							{
								UpdateBTState = 2;
								break;
							}
							
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Seeker1;			// Sending to Seeker1.
							if(BallTUFound)							// Determine if we send a found
								TransData.Data[0] = RRBallFound;	// or a NOT found message.
							else
								TransData.Data[0] = RRBallNotFound;	
							TransData.Data[1] = Seeker1Arg;			// Sending to Seeker1.				
							TransData.Data[2] = BallToUpdate;		// Which Ball.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBTState = 2;		// If we succeed move to sending to the next bot.
								else
									UpdateBTState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 1:	if(XBWriteAPI() == 0)
									UpdateBTState = 2; 
							break;
	
					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 2:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Seeker2)
							{
								UpdateBTState = 4;
								break;
							}
							
							// Be sure that Seeker2 can talk to us. If not then move to the next robot.
							if(Robot[Seeker2Arg].Status == NoLocalCOM || Robot[Seeker2Arg].Status == Dead)
							{
								UpdateBTState = 4;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Seeker2;			// Sending to Seeker2.
							if(BallTUFound)							// Determine if we send a found
								TransData.Data[0] = RRBallFound;	// or NOT found message.
							else
								TransData.Data[0] = RRBallNotFound;	
							TransData.Data[1] = Seeker2Arg;			// Sending to Seeker2.
							TransData.Data[2] = BallToUpdate;		// Which Ball.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBTState = 4;		// If we succeed move to sending to the next bot.
								else
									UpdateBTState = 3;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 3:	if(XBWriteAPI() == 0)
									UpdateBTState = 4; 
							break;
	
					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 4:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Reader1)
							{
								UpdateBTState = 6;
								break;
							}
							
							// Be sure that Reader1 can talk to us. If not then move to the next robot.
							if(Robot[Reader1Arg].Status == NoLocalCOM || Robot[Reader1Arg].Status == Dead)
							{
								UpdateBTState = 6;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Reader1;			// Sending to Reader1.
							if(BallTUFound)							// Determine if we send a found
								TransData.Data[0] = RRBallFound;	// or NOT found.
							else
								TransData.Data[0] = RRBallNotFound;	
							TransData.Data[1] = Reader1Arg;			// Sending to Reader2.
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBTState = 6;		// If we succeed move to sending to the next bot.
								else
									UpdateBTState = 5;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 5:	if(XBWriteAPI() == 0)
									UpdateBTState = 6; 
							break;

					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 6:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Reader2)
							{
								UpdateBTState = 8;
								break;
							}
							
							// Be sure that Reader2 can talk to us. If not then move to MarsSat.
							if(Robot[Reader2Arg].Status == NoLocalCOM || Robot[Reader2Arg].Status == Dead)
							{
								UpdateBTState = 8;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Reader2;			// Sending to Reader2.		
							if(BallTUFound)							// Determine if we send a found
								TransData.Data[0] = RRBallFound;	// or NOT found.
							else
								TransData.Data[0] = RRBallNotFound;	
							TransData.Data[1] = Reader2Arg;			// Sending to Reader2.					
							TransData.Data[2] = BallToUpdate;		// Which Ball.
							
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateBTState = 8;		// If we succeed move to sending to MarsSat.
								else
									UpdateBTState = 7;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to MarsSat.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 7:	if(XBWriteAPI() == 0)
									UpdateBTState = 8; 
							break;

					// Here we send the data to MarsSat. First we will check that we can talk to
					// MarsSat. If we can't then set/clear flags and increment the LocationIndex.
					// If we can then build up the TX struct and try to write it to EEPROM. If 
					// unsuccessful then break out and the main loop will come back here and try 
					// again later. If we succeed then immediatly try to write the TX data to the 
					// XBee module.	
					case 8:	if(Robot[MYARG].Status == NoSatCOM || Robot[MYARG].Status == Dead)
 							{	// Get here if we can't talk to MarsSat.
								UpdateBTState = 0;		// Reset this state machine.
								UpdateBallThere = 0;	// No loger want to be here to clear that flag.
								break;					// Break to main loop.
							}

							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = MarsSat;			// Sending to MarsSat.
							if(BallTUFound)							// Determine if we send a found
								TransData.Data[0] = RBallFound;		// or NOT found.
							else
								TransData.Data[0] = RBallNotFound;	
							TransData.Data[1] = MYARG;				// Coming from us.
							TransData.Data[2] = BallToUpdate;		// Which Ball.
	
							// Attempt to write the Transmitted data to the EEPROM. If we are unsuccessful 
							// then we just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)	// If we succeed set/clear flags and increment the LocationIndex.
								{							
									UpdateBTState = 0;		// Reset this state machine.
									UpdateBallThere = 0;	// No loger want to be here to clear that flag.
								}
								else
									UpdateBTState = 9;		// If we fail then go to state that continually retries.
							}
							
							break;

					// Retry sending the XBee message. If we succeed then set/clear flags 
					// and increment the LocationIndex. 
					case 9:	if(XBWriteAPI() == 0)
							{
								UpdateBTState = 0;		// Reset this state machine.
								UpdateBallThere = 0;	// No loger want to be here to clear that flag.
							}
							break;

				}// End Switch
			}// End If UpdateBallThere

			// ----------------------------------------------------------------
			// Here we will relay the location of a robot to MarsSat because 
			// that robot was unable to tell MarsSat his own self.
			// ----------------------------------------------------------------
			else if(RelayUpdateLocSeeker1)
			{
				// Here we have a simple state machine that will tell MarsSat the new
				// location of the robot. This would occur if the robot was able to speak
				// to other robots but not to MarSat. Thus, we decided that we should
				// do it for him.
				switch(RelayUpdateLocStateS1)
				{
					// In case 0 we will fill up the TX struct and try to write the data
					// to the EEPROM. If we fail then we don't change state or clear flags
					// but bail out to the main loop. As a result the main loop will come
					// back here later. If we succeed then we will try to write the TX 
					// to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = MarsSat;			// Sending to MarsSat.
							TransData.Data[0] = RLocationRelay;		// Relay location command.
							TransData.Data[1] = MYARG;				// Indicate this came from us.			
							TransData.Data[2] = Seeker1Arg;			// Location of Seeker1.
							TransData.Data[3] = Robot[Seeker1Arg].CurrentLocation;	// Actual location data.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// We succeed in writing to the XBee module.
									RelayUpdateLocStateS1 = 0;	// Reset the state machine.
									RelayUpdateLocSeeker1 = 0;	// Clear flag that got us here.
								}
								else
									UpdateLocState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we clear the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing
					// the TX struct because at this point we knwo we have the mutex and have
					// had it since case 0 when we did build up the TX struct.
 					case 1:	if(XBWriteAPI() == 0)
							{
								RelayUpdateLocStateS1 = 0;	// Reset the state machine.
								RelayUpdateLocSeeker1 = 0;	// Clear the flag that go us here.
							}
				}
			}

			else if(RelayUpdateLocSeeker2)
			{
				// Here we have a simple state machine that will tell MarsSat the new
				// location of the robot. This would occur if the robot was able to speak
				// to other robots but not to MarSat. Thus, we decided that we should
				// do it for him.
				switch(RelayUpdateLocStateS2)
				{
					// In case 0 we will fill up the TX struct and try to write the data
					// to the EEPROM. If we fail then we don't change state or clear flags
					// but bail out to the main loop. As a result the main loop will come
					// back here later. If we succeed then we will try to write the TX 
					// to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = MarsSat;			// Send to MarsSat.
							TransData.Data[0] = RLocationRelay;		// Relay Location Command.
							TransData.Data[1] = MYARG;				// Indicate this came from us. 
							TransData.Data[2] = Seeker2Arg;			// Location of Seeker2.
							TransData.Data[3] = Robot[Seeker2Arg].CurrentLocation;	// Actual location data.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// We succeed in writing to the XBee module.
									RelayUpdateLocStateS2 = 0;	// Reset the state machine.
									RelayUpdateLocSeeker2 = 0;	// Clear the flag that got us here.
								}
								else
									UpdateLocState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we clear the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing
					// the TX struct because at this point we knwo we have the mutex and have
					// had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayUpdateLocStateS2 = 0;	// Reset the state machine.
								RelayUpdateLocSeeker2 = 0;	// Clear the flag that got us here.
							}
				}		
			}

			else if(RelayUpdateLocReader1)
			{
				// Here we have a simple state machine that will tell MarsSat the new
				// location of the robot. This would occur if the robot was able to speak
				// to other robots but not to MarSat. Thus, we decided that we should
				// do it for him.
				switch(RelayUpdateLocStateR1)
				{
					// In case 0 we will fill up the TX struct and try to write the data
					// to the EEPROM. If we fail then we don't change state or clear flags
					// but bail out to the main loop. As a result the main loop will come
					// back here later. If we succeed then we will try to write the TX 
					// to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = MarsSat;			// Send to MarsSat.
							TransData.Data[0] = RLocationRelay;		// Relay Location Command.
							TransData.Data[1] = MYARG;				// Indicate this came from us.
							TransData.Data[2] = Reader1Arg;			// Location of Reader1.
							TransData.Data[3] = Robot[Reader1Arg].CurrentLocation;	// Acutal location data.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// We succeed in writing to the XBee module.
									RelayUpdateLocStateR1 = 0;	// Reset the state machine.
									RelayUpdateLocReader1 = 0;	// Clear the flag that got us here.
								}
								else
									UpdateLocState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we clear the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing
					// the TX struct because at this point we knwo we have the mutex and have
					// had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayUpdateLocStateR1 = 0;	// Reset the state machine.
								RelayUpdateLocReader1 = 0;	// Clear the flag that got us here.
							}
				}			
			}

			else if(RelayUpdateLocReader2)
			{
				// Here we have a simple state machine that will tell MarsSat the new
				// location of the robot. This would occur if the robot was able to speak
				// to other robots but not to MarSat. Thus, we decided that we should
				// do it for him.
				switch(RelayUpdateLocStateR2)
				{
					// In case 0 we will fill up the TX struct and try to write the data
					// to the EEPROM. If we fail then we don't change state or clear flags
					// but bail out to the main loop. As a result the main loop will come
					// back here later. If we succeed then we will try to write the TX 
					// to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = MarsSat;			// Sending to MarsSat.
							TransData.Data[0] = RLocationRelay;		// Relay Location Command.
							TransData.Data[1] = MYARG;				// Indicate this came from us.
							TransData.Data[2] = Reader2Arg;			// Location of Reader2.
							TransData.Data[3] = Robot[Reader2Arg].CurrentLocation;	// Actual location data.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// We succeed in writing to the XBee module.
									RelayUpdateLocStateR2 = 0;	// Reset the state machine.
									RelayUpdateLocReader2 = 0;	// Clear the flag that got us here.
								}
								else
									UpdateLocState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we clear the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing
					// the TX struct because at this point we knwo we have the mutex and have
					// had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayUpdateLocStateR2 = 0;	// Reset the state machine.
								RelayUpdateLocReader2 = 0;	// Clear the flag that got us here.
							}
				}	
			}

			// ---------------------------------------------------------------------
			// Here we will relay the location of a robot to another robot. This
			// condition arises when the robot with the location update is unable
			// to talk to other robots (so he sends the data to MarsSat only) and
			// the robot is unable to talk to MarsSat. As a result the only way
			// he we will get the data is if we send it to him when we get it from
			// MarsSat.
			// ---------------------------------------------------------------------
			
			// The flags for robot to robot location relay are interesting so we will
			// discuss them here. When there is nothing to do the flag is set to 0xff.
			// When there is something to do then the flag is the robot arguement value
			// (which is between 0-3). An interesting condition could arise in which
			// two robot that had NoSatCOM errors both report location change to MarsSat
			// at the same time. Thus, MarsSat will report both to us at the same time.
			// And thus, we may have to relay both to another robot at the same time.
			// To make it such that we can have two that happen at the same time the
			// flags for a robot will be a two byte array. We will know that we should
			// send two based on whether the value is 0xff or something else. The value
			// in the array corresponds to what robot's location needs to be sent.

			else if(RelayLocNoSatCOMS1[0] != 0xff || RelayLocNoSatCOMS1[1] != 0xff)
			{
				// State machine that will relay the locations of one or two robots.
				switch(RelayLocNoSatCOMStateS1)
				{
					// In case 0 we will fill up the TX struct with the robot location data
					// of the first index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = Seeker1;		// Send to Seeker1.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Seeker1Arg;		// Send to Seeker1.							
							TransData.Data[2] = RelayLocNoSatCOMS1[0];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMS1[0]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{	
									RelayLocNoSatCOMS1[0] = 0xff;		// Reset flag to 0xff.
									if(RelayLocNoSatCOMS1[1] == 0xff)	// Determine if we need to relay another location.
										RelayLocNoSatCOMStateS1 = 0;	
									else
										RelayLocNoSatCOMStateS1 = 2;	
								}
								else
									RelayLocNoSatCOMStateS1 = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and determine if we should relay another location. We don't have to worry 
					// about reconstructing the TX struct because at this point we know we have 
					// the mutex and have had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMS1[0] = 0xff;			// Reset flag to 0xff.
								if(RelayLocNoSatCOMS1[1] == 0xff)		// Determine if we need to relay another location. 
									RelayLocNoSatCOMStateS1 = 0;
								else
									RelayLocNoSatCOMStateS1 = 2;	
							}
							break;

					// In case 2 we will fill up the TX struct with the robot location data
					// of the second index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 2:	// Fill in struct.	
							TransData.DestAddLSB = Seeker1;		// Send to Seeker1.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Seeker1Arg;		// Send to Seeker1.					
							TransData.Data[2] = RelayLocNoSatCOMS1[1];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMS1[1]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// If successful.
									RelayLocNoSatCOMStateS1 = 0;		// Reset state machine.
									RelayLocNoSatCOMS1[1] = 0xff;		// Reset the flag to 0xff.
								}
								else
									RelayLocNoSatCOMStateS1 = 3;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 3 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing 
					// the TX struct because at this point we know we have the mutex and have 
					// had it since case 0 when we did build up the TX struct.
					case 3:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMStateS1 = 0;			// Reset state machine.
								RelayLocNoSatCOMS1[1] = 0xff;			// Reset the flag to 0xff.
							}
							break;
				}
			}

			else if(RelayLocNoSatCOMS2[0] != 0xff || RelayLocNoSatCOMS2[1] != 0xff)
			{
				// State machine that will relay the locations of one or two robots.
				switch(RelayLocNoSatCOMStateS2)
				{
					// In case 0 we will fill up the TX struct with the robot location data
					// of the first index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = Seeker2;		// Send to Seeker2.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Seeker2Arg;		// Send to Seeker2.
							TransData.Data[2] = RelayLocNoSatCOMS2[0];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMS2[0]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{	
									RelayLocNoSatCOMS2[0] = 0xff;		// Reset flag to 0xff.
									if(RelayLocNoSatCOMS2[1] == 0xff)	// Determine if we need to relay another location.
										RelayLocNoSatCOMStateS2 = 0;
									else
										RelayLocNoSatCOMStateS2 = 2;	
								}
								else
									RelayLocNoSatCOMStateS2 = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and determine if we should relay another location. We don't have to worry 
					// about reconstructing the TX struct because at this point we know we have 
					// the mutex and have had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMS2[0] = 0xff;			// Reset flag to 0xff.
								if(RelayLocNoSatCOMS2[1] == 0xff)		// Determine if we need to relay another location.
									RelayLocNoSatCOMStateS2 = 0;
								else
									RelayLocNoSatCOMStateS2 = 2;	
							}
							break;

					// In case 2 we will fill up the TX struct with the robot location data
					// of the second index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 2:	// Fill in struct.	
							TransData.DestAddLSB = Seeker2;		// Send to Seeker2.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Seeker2Arg;		// Send to Seeker2.					
							TransData.Data[2] = RelayLocNoSatCOMS2[1];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMS2[1]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// If successful.
									RelayLocNoSatCOMStateS2 = 0;		// Reset state machine.
									RelayLocNoSatCOMS2[1] = 0xff;		// Reset the flag to 0xff.
								}
								else
									RelayLocNoSatCOMStateS2 = 3;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 3 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing 
					// the TX struct because at this point we know we have the mutex and have 
					// had it since case 0 when we did build up the TX struct.
					case 3:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMStateS2 = 0;			// Reset state machine.
								RelayLocNoSatCOMS2[1] = 0xff;			// Reset the flag to 0xff.
							}
							break;
				}
			}

			else if(RelayLocNoSatCOMR1[0] != 0xff || RelayLocNoSatCOMR1[1] != 0xff)
			{
				// State machine that will relay the locations of one or two robots.
				switch(RelayLocNoSatCOMStateR1)
				{
					// In case 0 we will fill up the TX struct with the robot location data
					// of the first index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = Reader1;		// Send to Reader1.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Reader1Arg;		// Send to Reader1.
							TransData.Data[2] = RelayLocNoSatCOMR1[0];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMR1[0]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{	
									RelayLocNoSatCOMR1[0] = 0xff;		// Reset flag to 0xff.
									if(RelayLocNoSatCOMR1[1] == 0xff)	// Determine if we need to relay another location.
										RelayLocNoSatCOMStateR1 = 0;
									else
										RelayLocNoSatCOMStateR1 = 2;	
								}
								else
									RelayLocNoSatCOMStateR1 = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and determine if we should relay another location. We don't have to worry 
					// about reconstructing the TX struct because at this point we know we have 
					// the mutex and have had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMR1[0] = 0xff;			// Reset flag to 0xff.
								if(RelayLocNoSatCOMR1[1] == 0xff)		// Determine if we need to relay another location.
									RelayLocNoSatCOMStateR1 = 0;
								else
									RelayLocNoSatCOMStateR1 = 2;	
							}
							break;

					// In case 2 we will fill up the TX struct with the robot location data
					// of the second index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 2:	// Fill in struct.	
							TransData.DestAddLSB = Reader1;		// Send to Reader1.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Reader1Arg;		// Send to Reader1.
							TransData.Data[2] = RelayLocNoSatCOMR1[1];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMR1[1]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// If successful.
									RelayLocNoSatCOMStateR1 = 0;		// Reset state machine.
									RelayLocNoSatCOMR1[1] = 0xff;		// Reset the flag to 0xff.
								}
								else
									RelayLocNoSatCOMStateR1 = 3;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 3 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing 
					// the TX struct because at this point we know we have the mutex and have 
					// had it since case 0 when we did build up the TX struct.
					case 3:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMStateR1 = 0;			// Reset state machine.
								RelayLocNoSatCOMR1[1] = 0xff;			// Reset the flag to 0xff.
							}
							break;
				}
			}

			else if(RelayLocNoSatCOMR2[0] != 0xff || RelayLocNoSatCOMR2[1] != 0xff)
			{
				// State machine that will relay the locations of one or two robots.
				switch(RelayLocNoSatCOMStateR2)
				{
					// In case 0 we will fill up the TX struct with the robot location data
					// of the first index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 0:	// Fill in struct.	
							TransData.DestAddLSB = Reader2;		// Send to Reader2.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Reader2Arg;		// Send to Reader2.
							TransData.Data[2] = RelayLocNoSatCOMR2[0];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMR2[0]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// If successful.	
									RelayLocNoSatCOMR2[0] = 0xff;		// Reset flag to 0xff.
									if(RelayLocNoSatCOMR2[1] == 0xff)	// Determine if we need to relay another location.
										RelayLocNoSatCOMStateR2 = 0;
									else
										RelayLocNoSatCOMStateR2 = 2;	
								}
								else
									RelayLocNoSatCOMStateR2 = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 1 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and determine if we should relay another location. We don't have to worry 
					// about reconstructing the TX struct because at this point we know we have 
					// the mutex and have had it since case 0 when we did build up the TX struct.
					case 1:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMR2[0] = 0xff;			// Reset flag to 0xff.
								if(RelayLocNoSatCOMR2[1] == 0xff)		// Determine if we need to relay another location.
									RelayLocNoSatCOMStateR2 = 0;
								else
									RelayLocNoSatCOMStateR2 = 2;	
							}
							break;

					// In case 2 we will fill up the TX struct with the robot location data
					// of the second index flag and try to write the data to the EEPROM. If 
					// we fail then we don't change state or clear flags but bail out to 
					// the main loop. As a result the main loop will come back here later. 
					// If we succeed then we will try to write the TX to the XBee module.
					case 2:	// Fill in struct.	
							TransData.DestAddLSB = Reader2;		// Send to Reader2.
							TransData.Data[0] = RRLocRelay;		// Relay Location Command.
							TransData.Data[1] = Reader2Arg;		// Send to Reader2.
							TransData.Data[2] = RelayLocNoSatCOMR2[1];	// Which robot's location.
							TransData.Data[3] = Robot[RelayLocNoSatCOMR2[1]].CurrentLocation;	// Actual location.
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
								{
									// If successful.
									RelayLocNoSatCOMStateR2 = 0;		// Reset state machine.
									RelayLocNoSatCOMR2[1] = 0xff;		// Reset the flag to 0xff.
								}
								else
									RelayLocNoSatCOMStateR2 = 3;		// If we fail then go to state that continually retries.
							}
							break;

					// Case 3 will try to send the TX struct to the XBee module. If it fails
					// then we do nothing which will eventually lead us back here to try again
					// sometime later. If we succeed then we reset the flag that got us here
					// and reset the state machine. We don't have to worry about reconstructing 
					// the TX struct because at this point we know we have the mutex and have 
					// had it since case 0 when we did build up the TX struct.
					case 3:	if(XBWriteAPI() == 0)
							{
								RelayLocNoSatCOMStateR2 = 0;			// Reset state machine.
								RelayLocNoSatCOMR2[1] = 0xff;			// Reset the flag to 0xff.
							}
							break;
				}
			}

			// If we need to tell the world that we are at a new location then
			// we go here.
			else if(UpdateLocation)
			{
				// Simple state machine helps us leave here and continue servicing
				// messages in t\he main loop when we can no longer write to XBee
				// or EEPROM.
				switch(UpdateLocState)
				{
					case 0:	// Case 0 is slightly special in that the first thing we do is check
							// to see that we can send to other robots. If not then skip right
							// to sending data to MarsSat. 
							if(Robot[MYARG].Status == NoLocalCOM || Robot[MYARG].Status == Dead)
							{
								UpdateLocState = 8;
								break;
							}

							// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Seeker1)
							{
								UpdateLocState = 2;
								break;
							}
							
							// If we can send to Seeker1 then be sure that Seeker1 can talk to us.
							if(Robot[Seeker1Arg].Status == NoLocalCOM || Robot[Seeker1Arg].Status == Dead)
							{
								UpdateLocState = 2;
								break;
							}
							
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Seeker1;	
							TransData.Data[0] = RRLocation;	
							TransData.Data[1] = Seeker1Arg;							
							TransData.Data[2] = LocArray[LocationIndex];
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateLocState = 2;		// If we succeed move to sending to the next bot.
								else
									UpdateLocState = 1;		// If we fail then go to state that continually retries.
							}
							break;

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 1:	if(XBWriteAPI() == 0)
									UpdateLocState = 2; 
							break;
	
					
					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 2:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Seeker2)
							{
								UpdateLocState = 4;
								break;
							}
							
							// If we can send to Seeker1 then be sure that Seeker1 can talk to us.
							if(Robot[Seeker2Arg].Status == NoLocalCOM || Robot[Seeker2Arg].Status == Dead)
							{
								UpdateLocState = 4;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Seeker2;	
							TransData.Data[0] = RRLocation;	
							TransData.Data[1] = Seeker2Arg;							
							TransData.Data[2] = LocArray[LocationIndex];
	
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateLocState = 4;		// If we succeed move to sending to the next bot.
								else
									UpdateLocState = 3;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 3:	if(XBWriteAPI() == 0)
									UpdateLocState = 4; 
							break;
	
					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 4:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Reader1)
							{
								UpdateLocState = 6;
								break;
							}
							
							// If we can send to Seeker1 then be sure that Seeker1 can talk to us.
							if(Robot[Reader1Arg].Status == NoLocalCOM || Robot[Reader1Arg].Status == Dead)
							{
								UpdateLocState = 6;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Reader1;	
							TransData.Data[0] = RRLocation;	
							TransData.Data[1] = Reader1Arg;							
							TransData.Data[2] = LocArray[LocationIndex];
							
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateLocState = 6;		// If we succeed move to sending to the next bot.
								else
									UpdateLocState = 5;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to the next bot.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 5:	if(XBWriteAPI() == 0)
									UpdateLocState = 6; 
							break;

					// Check the next robot to be sure that we can to it and if we can then build up
					// the TX struct and try to write it to EEPROM. If unsuccessful then break out
					// and the main loop will come back here and try again later. If we succeed then
					// immediatly try to write the TX data to the XBee module.
					case 6:	// It makes no sense to send a message to ourselves so check for that.
							if(MYADD == Reader2)
							{
								UpdateLocState = 8;
								break;
							}
							
							// If we can send to Seeker1 then be sure that Seeker1 can talk to us.
							if(Robot[Reader2Arg].Status == NoLocalCOM || Robot[Reader2Arg].Status == Dead)
							{
								UpdateLocState = 8;
								break;
							}
								
							// If he can then fill out the struct to do just that.	
							TransData.DestAddLSB = Reader2;	
							TransData.Data[0] = RRLocation;	
							TransData.Data[1] = Reader2Arg;							
							TransData.Data[2] = LocArray[LocationIndex];
							
							// Attempt to write the data to the EEPROM. If we are unsuccessful then we
							// just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)
									UpdateLocState = 8;		// If we succeed move to sending to MarsSat.
								else
									UpdateLocState = 7;		// If we fail then go to state that continually retries.
							}
							break;	

					// Retry sending the XBee messages. If we succeed then we try to send to MarsSat.
					// If not then we stay on this state for another retry. No matter what we break to the
					// main loop to service other messages.
					case 7:	if(XBWriteAPI() == 0)
									UpdateLocState = 8; 
							break;

					// Here we send the data to MarsSat. First we will check that we can talk to
					// MarsSat. If we can't then set/clear flags and increment the LocationIndex.
					// If we can then build up the TX struct and try to write it to EEPROM. If 
					// unsuccessful then break out and the main loop will come back here and try 
					// again later. If we succeed then immediatly try to write the TX data to the 
					// XBee module.	
					case 8:	if(Robot[MYARG].Status == NoSatCOM || Robot[MYARG].Status == Dead)
 							{	// Get here if we can't talk to MarsSat.
								UpdateLocState = 0;		// Reset this state machine.
								UpdateLocation = 0;		// No loger want to be here to clear that flag.
								LocationIndex++;		// Increment the LocationIndex.
								break;					// Break to main loop.
							}
								
							// If we can send to MarsSat then fill out the struct to do just that.
							TransData.DestAddLSB = MarsSat;		// Send to MarsSat.
							TransData.Data[0] = RLocation;		// Sending location message.
							TransData.Data[1] = MYARG;			// First arg is always us.
							TransData.Data[2] = LocArray[LocationIndex];	// Location Data.
	
							// Attempt to write the Transmitted data to the EEPROM. If we are unsuccessful 
							// then we just break to the main loop and we will try again later.
							if(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)) == 0)
							{
								// If we succeed then increment the address and immediatly try to write
								// to the XBee module.
								EEPROMAddress += sizeof(TransData.Data);
								if(XBWriteAPI() == 0)	// If we succeed set/clear flags and increment the LocationIndex.
								{							
									UpdateLocState = 0;		// Reset this state machine.
									UpdateLocation = 0;		// No loger want to be here to clear that flag.
									LocationIndex++;		// Increment the LocationIndex.
								}
								else
									UpdateLocState = 9;		// If we fail then go to state that continually retries.
							}
							
							break;

					// Retry sending the XBee message. If we succeed then set/clear flags 
					// and increment the LocationIndex. 
					case 9:	if(XBWriteAPI() == 0)
							{
								UpdateLocState = 0;		// Reset this state machine.
								UpdateLocation = 0;		// No loger want to be here to clear that flag.
								LocationIndex++;		// Increment the LocationIndex.
							}
							break;

				}// End Switch
			}// End If UpdateLocation

			else
			{
				WaitForMutex = 0;		// We no longer want the mutex.
				ReleaseMtx = 1; 		// Schedule the release of the mutex.  	
			}	
		}

		// If the mutex has been flagged for release then call a function to
		// release the mutex.
		if(ReleaseMtx)
		{
			if(ReleaseMutex() == 0)		// Attempt to release the mutex.
				ReleaseMtx = 0;		// If successful then clear the flag.
		}

		// If we are flagged to do so store any data we get from the XBee module in 
		// the EERPOM, increment the EEPROM address, and clear the flag. 
		if(WriteEEPROMRX)
		{
			// Attemp to write the EEPROM data.
			if(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)) == 0)
			{	
					// If successful then increment the EEPROM address and clear flag.
					EEPROMAddress += sizeof(RecData.Data);
					WriteEEPROMRX = 0;
			}
		}
		
		// This message indicates that timer2
		// have fired and we should determine what
		// action to take with the motors if any.
		if(Timer2Fired)
		{
			// Clear the flag.
			Timer2Fired = 0;
			
			switch(MotorState)
			{
				// Wait for us to tell uC that it is time to move.
				case 0:		if(MotorOn)
							{
								MotorState = StraightState;
								index = 0;
								IgnoreFlag = 0;
								if(MYARG == Seeker1Arg || MYARG == Reader1Arg)
									MotorStateBallIndex = 0;
								else
									MotorStateBallIndex = 4;

								// Build the LocArray, DecArray, and the BallArray for each robot based
								// on who is operational and where the balls are. The Decision array tells
								// the robot what action to take when he gets to an intersection. The
								// Location array tells the robot what location to tell the other robots
								// and MarsSat. If the value is 0xff then we don't send a location. The
								// Ball Array is only for readers and it tells them which balls to go
								// and get data from because they are there.
								if(MYARG == Seeker1Arg)
								{
									DecArray[0]  = TurnLeft;  DecArray[1]  = TurnLeft;  DecArray[2]  = TurnRight;
									DecArray[3]  = FindBall;  DecArray[4]  = Ignore;    DecArray[5]  = FindBall;
									DecArray[6]  = TurnRight; DecArray[7]  = Ignore;    DecArray[8]  = TurnLeft;
									DecArray[9]  = FindBall;  DecArray[10] = Ignore;    DecArray[11] = FindBall;
									DecArray[12] = TurnLeft;  DecArray[13] = TurnLeft; 

									LocArray[0]  = 5;         LocArray[1]  = 9;         LocArray[2]  = 11;
									LocArray[3]  = 0xff;	  LocArray[4]  = 0xff;      LocArray[5]  = 0xff;
									LocArray[6]  = 0xff;      LocArray[7]  = 9;         LocArray[8]  = 12;
									LocArray[9]  = 0xff;	  LocArray[10] = 0xff;      LocArray[11] = 0xff;
									LocArray[12] = 0xff;      LocArray[13] = 9;

									// If Seeker2 is dead or not moving then we need to go look at his balls.
									if(Robot[Seeker2Arg].Status == Dead || Robot[Seeker2Arg].Status == NoMove)
									{
										DecArray[14] = Ignore; 	  DecArray[15] = Ignore;    DecArray[16] = TurnRight;
										DecArray[17] = TurnRight; DecArray[18] = FindBall;  DecArray[19] = Ignore;
										DecArray[20] = FindBall;  DecArray[21] = TurnRight; DecArray[22] = Ignore;
										DecArray[23] = TurnLeft;  DecArray[24] = FindBall;  DecArray[25] = Ignore;
										DecArray[26] = FindBall;  DecArray[27] = TurnLeft;  DecArray[28] = TurnRight;
										DecArray[29] = Ignore;    DecArray[30] = TurnRight; DecArray[31] = AllStop;

										LocArray[14] = 5;         LocArray[15] = 6;         LocArray[16] = 10;
										LocArray[17] = 13;        LocArray[18] = 0xff;		LocArray[19] = 0xff;
										LocArray[20] = 0xff;      LocArray[21] = 0xff;      LocArray[22] = 10;
										LocArray[23] = 14;        LocArray[24] = 0xff;      LocArray[25] = 0xff;
										LocArray[26] = 0xff;      LocArray[27] = 0xff;      LocArray[28] = 10;
										LocArray[29] = 6;         LocArray[30] = 5;			LocArray[31] = 7;
									}

									// Else Seeker2 is fine and we just go home.
									else
									{
										DecArray[14] =  TurnLeft; DecArray[15] =  AllStop;
									
										LocArray[14] = 5;         LocArray[15] = 7;
									}
								}

								else if(MYARG == Seeker2Arg)
								{
									DecArray[0]  = TurnLeft;  DecArray[1]  = TurnRight; DecArray[2]  = TurnRight;
									DecArray[3]  = FindBall;  DecArray[4]  = Ignore;    DecArray[5]  = FindBall;
									DecArray[6]  = TurnRight; DecArray[7]  = Ignore;    DecArray[8]  = TurnLeft;
									DecArray[9]  = FindBall;  DecArray[10] = Ignore;    DecArray[11] = FindBall;
									DecArray[12] = TurnLeft;  DecArray[13] = TurnRight; 

									LocArray[0]  = 6;         LocArray[1]  = 10;        LocArray[2]  = 13;
									LocArray[3]  = 0xff;	  LocArray[4]  = 0xff;      LocArray[5]  = 0xff;
									LocArray[6]  = 0xff;      LocArray[7]  = 10;        LocArray[8]  = 14;
									LocArray[9]  = 0xff;	  LocArray[10] = 0xff;      LocArray[11] = 0xff;
									LocArray[12] = 0xff;      LocArray[13] = 10;

									// If Seeker1 is dead or not moving then we need to go look at his balls.
									if(Robot[Seeker1Arg].Status == Dead || Robot[Seeker1Arg].Status == NoMove)
									{
										DecArray[14] = Ignore; 	  DecArray[15] = Ignore;    DecArray[16] = TurnLeft;
										DecArray[17] = TurnRight; DecArray[18] = FindBall;  DecArray[19] = Ignore;
										DecArray[20] = FindBall;  DecArray[21] = TurnRight; DecArray[22] = Ignore;
										DecArray[23] = TurnLeft;  DecArray[24] = FindBall;  DecArray[25] = Ignore;
										DecArray[26] = FindBall;  DecArray[27] = TurnLeft;  DecArray[28] = TurnLeft;
										DecArray[29] = Ignore;    DecArray[30] = TurnRight; DecArray[31] = AllStop;

										LocArray[14] = 6;         LocArray[15] = 5;         LocArray[16] = 9;
										LocArray[17] = 11;        LocArray[18] = 0xff;		LocArray[19] = 0xff;
										LocArray[20] = 0xff;      LocArray[21] = 0xff;      LocArray[22] = 9;
										LocArray[23] = 12;        LocArray[24] = 0xff;      LocArray[25] = 0xff;
										LocArray[26] = 0xff;      LocArray[27] = 0xff;      LocArray[28] = 9;
										LocArray[29] = 5;         LocArray[30] = 6;			LocArray[31] = 8;
									}

									// Else Seeker1 is fine and we just go home.
									else
									{
										DecArray[14] =  TurnLeft; DecArray[15] =  AllStop;
									
										LocArray[14] = 6;         LocArray[15] = 8;
									}
								}

								else if(MYARG == Reader1Arg)
								{	
									// If there is at least one ball on my side.
									if( Ball[A].Status == There || Ball[B].Status == There ||
										Ball[C].Status == There || Ball[D].Status == There )
									{
										DecArray[BallDecIndex] = TurnLeft; LocArray[BallDecIndex] = 5; BallDecIndex++;
										
										// If either Ball A or Ball B is there then turn left.
										if(Ball[A].Status == There || Ball[B].Status == There)
										{
											DecArray[BallDecIndex] = TurnLeft; LocArray[BallDecIndex] = 9; BallDecIndex++;

											// If Ball A is there then turn right and read it.
											if(Ball[A].Status == There)
											{
												DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 11;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;  LocArray[BallDecIndex] = 0xff; BallDecIndex++;

												// If Ball B is there then read it.
												if(Ball[B].Status == There)
												{
													DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												}
												// If Ball B is not there then move on.
												else
												{
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												}
											}

											// Else Ball A is not there so we read Ball B
											else
											{
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 11;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
											}

											// If either Ball C or Ball D is there then move forward.
											if(Ball[C].Status == There || Ball[D].Status == There)
											{
												DecArray[BallDecIndex] = Ignore;	LocArray[BallDecIndex] = 9; BallDecIndex++;

												// If Ball C is there then turn and read the ball.
												if(Ball[C].Status == There)
												{
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 12;   BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												
													// Check that Ball D is there. If so then read it and start home.
													if(Ball[D].Status == There)
													{
														DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
													}

													// If Ball D is not there then start home.
													else
													{
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
													}
												}

												// Else Ball C is not there so we turn to Ball D. We will read D and start home.
												else
												{
													DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 12;   BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
												}

											}

											// If Ball C or D is not there then turn right to go home.
											else
											{
												DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 9; BallDecIndex++;
											}
										}

										// Else there must be at least one ball at C or D so turn right.
										else
										{
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 9;		BallDecIndex++;

											// If Ball C is there then turn and read the ball.
											if(Ball[C].Status == There)
											{
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 12;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
											
												// Check that Ball D is there. If so then read it and start home.
												if(Ball[D].Status == There)
												{
													DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
												}
	
												// If Ball D is not there then start home.
												else
												{
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
												}
											}
	
											// Else Ball C is not there so we turn to Ball D. We will read D and start home.
											else
											{
												DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 12;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
											}
										}

										// Check to see if my buddy is functional.
										if(Robot[Reader2Arg].Status == Dead || Robot[Reader2Arg].Status == NoMove)
										{
											// Check to see if there any balls over there that needs to be read
											if(	Ball[E].Status == There || Ball[F].Status == There ||
												Ball[G].Status == There || Ball[H].Status == There )
											{
												DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 5;    BallDecIndex++;
												DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 6;    BallDecIndex++;

												// If either Ball E or Ball F is there then turn right.
												if(Ball[E].Status == There || Ball[F].Status == There)
												{
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10; BallDecIndex++;
		
													// If Ball E is there then turn right and read it.
													if(Ball[E].Status == There)
													{
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 13;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;  LocArray[BallDecIndex] = 0xff; BallDecIndex++;
		
														// If Ball F is there then read it.
														if(Ball[F].Status == There)
														{
															DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														}
														// If Ball F is not there then move on.
														else
														{
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														}
													}
		
													// Else Ball E is not there so we read Ball F
													else
													{
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 13;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													}
		
													// If either Ball G or Ball H is there then move forward.
													if(Ball[G].Status == There || Ball[H].Status == There)
													{
														DecArray[BallDecIndex] = Ignore;	LocArray[BallDecIndex] = 10; BallDecIndex++;
		
														// If Ball G is there then turn and read the ball.
														if(Ball[G].Status == There)
														{
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 14;   BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														
															// Check that Ball H is there. If so then read it and start home.
															if(Ball[H].Status == There)
															{
																DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;    BallDecIndex++;
															}
		
															// If Ball H is not there then start home.
															else
															{
																DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
															}
														}
		
														// Else Ball G is not there so we turn to Ball H. We will read H and start home.
														else
														{
															DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 14;   BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
														}
		
													}
		
													// If Ball G or H is not there then turn left to go home.
													else
													{
														DecArray[BallDecIndex] = TurnLeft;	LocArray[BallDecIndex] = 10; BallDecIndex++;
													}
												}
		
												// Else there must be at least one ball at G or H so turn left.
												else
												{
													DecArray[BallDecIndex] = TurnLeft; LocArray[BallDecIndex] = 10;	BallDecIndex++;
		
													// If Ball G is there then turn and read the ball.
													if(Ball[G].Status == There)
													{
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 14;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													
														// Check that Ball H is there. If so then read it and start home.
														if(Ball[H].Status == There)
														{
															DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
														}
			
														// If Ball H is not there then start home.
														else
														{
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
														}
													}
			
													// Else Ball G is not there so we turn to Ball H. We will read H and start home.
													else
													{
														DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 14;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
													}
												}

												// Stop as we are done.
												DecArray[BallDecIndex] = AllStop; LocArray[BallDecIndex] = 6;
												
											}
			
											// Else there are not any balls over there so we can go home.
											else
											{
												DecArray[BallDecIndex] = AllStop; 	LocArray[BallDecIndex] = 5;
											}
										}
										
										// Else he is functional so I don't need to go to his end. We can go home.
										else
										{
											DecArray[BallDecIndex] = AllStop; 	LocArray[BallDecIndex] = 5;
										}
									}

									// Else all the balls must be on the other side.
									else
									{
										// Check that our buddy is dead or can't move. If he is functional
										// then we are done.
										if(Robot[Reader2Arg].Status == Dead || Robot[Reader2Arg].Status == NoMove)
										{
											// If all the balls are on the other side then there is no conditional logic.
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 5;		BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 6;		BallDecIndex++;
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 13;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 10;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 14;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;	BallDecIndex++;
											DecArray[BallDecIndex] = AllStop; 	LocArray[BallDecIndex] = 6;
										}

										// We are done.
										else
										{
											// Stop because we have no balls to look for and our friend
											// is able to read the data from the others.
											StopFlag = 1;			// Set the stop flag so that the main loop stops the motors.
											MotorState = WaitState;	// Goto the WaitState as we are done.
										}
									}
								}

								else if(MYARG == Reader2Arg)
								{
									// If there is at least one ball on my side.
									if( Ball[E].Status == There || Ball[F].Status == There ||
										Ball[G].Status == There || Ball[H].Status == There )
									{
										DecArray[BallDecIndex] = TurnLeft; LocArray[BallDecIndex] = 6; BallDecIndex++;
										
										// If either Ball E or Ball F is there then turn right.
										if(Ball[E].Status == There || Ball[F].Status == There)
										{
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10; BallDecIndex++;

											// If Ball E is there then turn right and read it.
											if(Ball[E].Status == There)
											{
												DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 13;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;  LocArray[BallDecIndex] = 0xff; BallDecIndex++;

												// If Ball F is there then read it.
												if(Ball[F].Status == There)
												{
													DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												}
												// If Ball F is not there then move on.
												else
												{
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												}
											}

											// Else Ball E is not there so we read Ball F
											else
											{
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 13;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
											}

											// If either Ball G or Ball H is there then move forward.
											if(Ball[G].Status == There || Ball[H].Status == There)
											{
												DecArray[BallDecIndex] = Ignore;	LocArray[BallDecIndex] = 10; BallDecIndex++;

												// If Ball G is there then turn and read the ball.
												if(Ball[G].Status == There)
												{
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 14;   BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												
													// Check that Ball H is there. If so then read it and start home.
													if(Ball[H].Status == There)
													{
														DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
													}

													// If Ball H is not there then start home.
													else
													{
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
													}
												}

												// Else Ball G is not there so we turn to Ball H. We will read H and start home.
												else
												{
													DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 14;   BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
												}

											}

											// If Ball G or H is not there then turn left to go home.
											else
											{
												DecArray[BallDecIndex] = TurnLeft;	LocArray[BallDecIndex] = 10; BallDecIndex++;
											}
										}

										// Else there must be at least one ball at G or H so turn left.
										else
										{
											DecArray[BallDecIndex] = TurnLeft; LocArray[BallDecIndex] = 10;		BallDecIndex++;

											// If Ball G is there then turn and read the ball.
											if(Ball[G].Status == There)
											{
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 14;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
											
												// Check that Ball H is there. If so then read it and start home.
												if(Ball[H].Status == There)
												{
													DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
												}
	
												// If Ball H is not there then start home.
												else
												{
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
												}
											}
	
											// Else Ball G is not there so we turn to Ball H. We will read H and start home.
											else
											{
												DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 14;   BallDecIndex++;
												DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
												DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 10;   BallDecIndex++;
											}
										}

										// Check to see if my buddy is functional.
										if(Robot[Reader1Arg].Status == Dead || Robot[Reader1Arg].Status == NoMove)
										{
											// Check to see if there any balls over there that needs to be read
											if(	Ball[A].Status == There || Ball[B].Status == There ||
												Ball[C].Status == There || Ball[D].Status == There )
											{
												DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 6;    BallDecIndex++;
												DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 5;    BallDecIndex++;

												// If either Ball A or Ball B is there then turn left.
												if(Ball[A].Status == There || Ball[B].Status == There)
												{
													DecArray[BallDecIndex] = TurnLeft; LocArray[BallDecIndex] = 9; BallDecIndex++;
		
													// If Ball A is there then turn right and read it.
													if(Ball[A].Status == There)
													{
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 11;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;  LocArray[BallDecIndex] = 0xff; BallDecIndex++;
		
														// If Ball B is there then read it.
														if(Ball[B].Status == There)
														{
															DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														}
														// If Ball B is not there then move on.
														else
														{
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														}
													}
		
													// Else Ball A is not there so we read Ball B
													else
													{
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 11;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													}
		
													// If either Ball C or Ball D is there then move forward.
													if(Ball[C].Status == There || Ball[D].Status == There)
													{
														DecArray[BallDecIndex] = Ignore;	LocArray[BallDecIndex] = 9; BallDecIndex++;
		
														// If Ball C is there then turn and read the ball.
														if(Ball[C].Status == There)
														{
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 12;   BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														
															// Check that Ball D is there. If so then read it and start home.
															if(Ball[D].Status == There)
															{
																DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
															}
		
															// If Ball D is not there then start home.
															else
															{
																DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
																DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
															}
														}
		
														// Else Ball C is not there so we turn to Ball D. We will read D and start home.
														else
														{
															DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 12;   BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
														}
		
													}
		
													// If Ball C or D is not there then turn right to go home.
													else
													{
														DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 9; BallDecIndex++;
													}
												}
		
												// Else there must be at least one ball at C or D so turn right.
												else
												{
													DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 9;	BallDecIndex++;
		
													// If Ball C is there then turn and read the ball.
													if(Ball[C].Status == There)
													{
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 12;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
													
														// Check that Ball D is there. If so then read it and start home.
														if(Ball[D].Status == There)
														{
															DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;    BallDecIndex++;
														}
			
														// If Ball D is not there then start home.
														else
														{
															DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff; BallDecIndex++;
															DecArray[BallDecIndex] = TurnLeft;  LocArray[BallDecIndex] = 9;    BallDecIndex++;
														}
													}
			
													// Else Ball C is not there so we turn to Ball D. We will read H and start home.
													else
													{
														DecArray[BallDecIndex] = TurnRight;	LocArray[BallDecIndex] = 12;   BallDecIndex++;
														DecArray[BallDecIndex] = ReadBall;	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff; BallDecIndex++;
														DecArray[BallDecIndex] = TurnLeft;  LocArray[BallDecIndex] = 9;    BallDecIndex++;
													}
												}

												// Stop as we are done.
												DecArray[BallDecIndex] = AllStop; LocArray[BallDecIndex] = 5;
												
											}
			
											// Else there are not any balls over there so we can go home.
											else
											{
												DecArray[BallDecIndex] = AllStop; 	LocArray[BallDecIndex] = 6;
											}
										}
										
										// Else he is functional so I don't need to go to his end. We can go home.
										else
										{
											DecArray[BallDecIndex] = AllStop; 	LocArray[BallDecIndex] = 6;
										}
									}

									// Else all the balls must be on the other side.
									else
									{
										// Check that our buddy is dead or can't move. If he is functional
										// then we are done.
										if(Robot[Reader1Arg].Status == Dead || Robot[Reader1Arg].Status == NoMove)
										{
											// If all the balls are on the other side then there is no conditional logic.
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 6;		BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 5;		BallDecIndex++;
											DecArray[BallDecIndex] = TurnLeft;  LocArray[BallDecIndex] = 9;		BallDecIndex++;
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 11;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnRight; LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 9;		BallDecIndex++;
											DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 12;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = Ignore; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = ReadBall; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 0xff;	BallDecIndex++;
											DecArray[BallDecIndex] = TurnLeft; 	LocArray[BallDecIndex] = 9;		BallDecIndex++;
											DecArray[BallDecIndex] = AllStop; 	LocArray[BallDecIndex] = 5;
										}

										// We are done.
										else
										{
											// Stop because we have no balls to look for and our friend
											// is able to read the data from the others.
											StopFlag = 1;			// Set the stop flag so that the main loop stops the motors.
											MotorState = WaitState;	// Goto the WaitState as we are done.
										}
									}
								}


								// If we are a reader then build up the Ball Array.
								if(Robot[MYARG].TypeOfRobot == Reader)
								{
									// Start at Ball A if we are Reader1 and Ball E if Reader2
									if(MYARG == Reader1Arg)
										BallArrayIndex = 0;
									else
										BallArrayIndex = 4;

									// Fill in the BallArray. We will go through the balls one a time to
									// determine if we should add it to the array or not. Note that there
									// should not be more that four balls because that is the size of the 
									// ball array.
									MotorStateBallIndex = 0;	// Reset the Index.
									for(BallArrayCounter=0;BallArrayCounter<8;BallArrayCounter++)
									{
										// If this ball is there then add it to the array and increment
										// the ball index.
										if(Ball[BallArrayIndex].Status == There)
										{
											MotorStateBallArray[MotorStateBallIndex] = BallArrayIndex;
											MotorStateBallIndex++;
										}

										// Increment the ball index that says what ball we are looking at.
										// If we get to 8 then we need to roll over to zero as there are
										// only eight balls.
										BallArrayIndex++;
										if(BallArrayIndex == 8)
											BallArrayIndex = 0;
									}

									// Reset the Ball index as it is used in the control loop.
									MotorStateBallIndex = 0;
								}
							}
							break;

				// Here we look for any black line. The first two look
				// for intersections (possible places to turn) and
				// the last two look for the center line which will
				// keep us straight.
				case StraightState:		if(OutRight SeesBlack || OutLeft SeesBlack)
										{
											if(IgnoreFlag == 0)
												MotorState = DecisionState;
											else
											{
												if(InLeft SeesBlack)
													MotorState = CorrectRState;
												else if(InRight SeesBlack)
													MotorState = CorrectLState;
											}
										} 
										else if(OutRight SeesAllWhite && OutLeft SeesAllWhite)
										{
											IgnoreFlag = 0;
									 		if(InLeft SeesBlack)
												MotorState = CorrectRState;
											else if(InRight SeesBlack)
												MotorState = CorrectLState;
										}
										else
										{
									 		if(InLeft SeesBlack)
												MotorState = CorrectRState;
											else if(InRight SeesBlack)
												MotorState = CorrectLState;
										}
										break;

				case DecisionState:		if(LocArray[LocationIndex] == 0xFF)
											LocationIndex++;
										else
										{
											UpdateLocation = 1;
											WaitForMutex = 1;
										}
								

										if(DecArray[index] == TurnLeft)
											MotorState = LeftState;
										else if(DecArray[index] == TurnRight)
											MotorState = RightState;
										else if(DecArray[index] == Flip)
											MotorState = FlipState;
										else if(DecArray[index] == FindBall)
											MotorState = FindBallState;
										else if(DecArray[index] == ReadBall)
											MotorState = ReadBallState;
										else if(DecArray[index] == Ignore)
										{
											IgnoreFlag = 1;
											MotorState = StraightState;
										}
										else if(DecArray[index] == AllStop)
											MotorState = StopState;
										index++;
										break;

				// ####################################
				// Here we will turn right.
				// ####################################
				// Command the right motor to brake and the left motor to go forward.
				case RightState:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
										MotorState = RightState+1;
									break;

				case RightState+1:	if(CommandMotor(Right, Reverse, 0x00) == 0)
										MotorState = RightState+2;
									break;

				// Wait until the inner left sensor sees black, then white, and then black.
				case RightState+2:	if(InLeft SeesBlack)
									{
										MotorState = RightState+3;
									};
									break;

				case RightState+3:	if(InLeft SeesWhite)
									{
										MotorState = RightState+4;
									}
									break;

				case RightState+4:	if(InLeft SeesBlack)
									{
										MotorState = RightState+5;
									}
									break;

				// Then back up until the inner left sensor sees white (it helps
				// straighten us out).
				case RightState+5:	if(CommandMotor(Left, Reverse, LeftSpeed) == 0)
										MotorState = RightState+6;
									break;

				case RightState+6:	if(InLeft SeesWhite)
										MotorState = RightState+7;
									break;

				// Then command the motors to go forward and carry on.
				case RightState+7:	if(CommandMotor(Right, Forward, RightSpeed) == 0)
										MotorState = RightState+8;
									break;

				case RightState+8:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
									{
										MotorState = StraightState;
									}
									break;	

				// ####################################
				// Here we will turn left.
				// ####################################
				// Command the left motor to brake and the right motor to go forward.
				case LeftState:		if(CommandMotor(Right, Forward, RightSpeed) == 0)
										MotorState = LeftState+1;
									break;

				case LeftState+1:	if(CommandMotor(Left, Reverse, 0x00) == 0)
										MotorState = LeftState+2;
									break;

				// Wait until the inner right sensor sees black, then white, and then black.
				case LeftState+2:	if(InRight SeesBlack)
									{
										MotorState = LeftState+3;
									};
									break;

				case LeftState+3:	if(InRight SeesWhite)
									{
										MotorState = LeftState+4;
									};
									break;

				case LeftState+4:	if(InRight SeesBlack)
									{
										MotorState = LeftState+5;
									};
									break;

				// Then back up until the inner right sensor sees white (it helps
				// straighten us out).
				case LeftState+5:	if(CommandMotor(Right, Reverse, RightSpeed) == 0);
										MotorState = LeftState+6;
									break;

				case LeftState+6:	if(InRight SeesWhite)
										MotorState = LeftState+7;
									break;

				// Then command the motors to go forward and carry on.
				case LeftState+7:	if(CommandMotor(Right, Forward, RightSpeed) == 0)
										MotorState = LeftState+8;
									break;

				case LeftState+8:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
									{
										MotorState = StraightState;
									}
									break;	

				// #################################################################
				// This is the case when the inner left sees black. This indicates
				// that we need to straighten out.
				// #################################################################
				// Make the right motor go faster to straighten us out.
				case CorrectRState:	if(CommandMotor(Right, Forward, FastSpeed) == 0)
										MotorState = CorrectRState+1;
									break;

				/* Got IT! Problem is that we don't see the outside black (which will be an ignore)
				* until we are correcting. Then we goto the decision which is to ignore. But,
				* we never changed the FastSpeed. */
			

				// Check for intersections and check to see if we have straightened out.
				case CorrectRState+1:	if(OutRight SeesBlack || OutLeft SeesBlack)
										{
											if(IgnoreFlag == 0)
											{
												if(DecArray[index] == Ignore)
												{
													if(CommandMotor(Right, Forward, RightSpeed) == 0)
														MotorState = DecisionState;
												}
												else
													MotorState = DecisionState;
		
											}
											else if(InLeft SeesWhite)
												MotorState = CorrectRState+2;
										}
										else if(InLeft SeesWhite)
											MotorState = CorrectRState+2;
										break;

				// If we have straightened out then we need make the right motor
				// back to the normal speed.
				case CorrectRState+2:	if(CommandMotor(Right, Forward, RightSpeed) == 0)
											MotorState = CorrectRState+3;
										break;

			
				case CorrectRState+3:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
										{
											MotorState = StraightState;	
										}
										break;	

				// #################################################################
				// This is the case when the inner right sees black. This indicates
				// that we need to straighten out.
				// #################################################################
				// Make the left motor go faster to straighten us out.
				case CorrectLState:		if(CommandMotor(Left, Forward, FastSpeed) == 0)
											MotorState = CorrectLState+1;
										break;

				// Check for intersections and check to see if we have straightened out.
				case CorrectLState+1:	if(OutRight SeesBlack || OutLeft SeesBlack)
										{
											if(IgnoreFlag == 0)
											{
												if(DecArray[index] == Ignore)
												{
													if(CommandMotor(Left, Forward, LeftSpeed) == 0)
														MotorState = DecisionState;
												}
												else
													MotorState = DecisionState;
		
											}
											else if(InRight SeesWhite)
												MotorState = CorrectLState+2;
										}
										else if(InRight SeesWhite)
											MotorState = CorrectLState+2;
										break;

				// If we have straightened out then we need make the right motor
				// back to the normal speed.
				case CorrectLState+2:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
											MotorState = CorrectLState+3;
										break;

				case CorrectLState+3:	if(CommandMotor(Right, Forward, RightSpeed) == 0)
										{
											MotorState = StraightState;
										}	
										break;


				// ##################################################################
				// This is when we need to perform a 180 degree turn
				// ##################################################################
				// Begin by stopping the motors.
				case FlipState:		if(CommandMotor(Left, Reverse, 0x00) == 0)
										MotorState = FlipState+1;
									break;

				case FlipState+1:	if(CommandMotor(Right, Reverse, 0x00) == 0)
										MotorState = FlipState+2;
									break;	

				// Then we will wait 500ms.
				case FlipState+2:	count++;
									if(count == 50)
									{
										count = 0;
										MotorState = FlipState+3;
									}
									break;

				// Then we will turn left in place (spin). The +12 in reverse helps us here.
				case FlipState+3:	if(CommandMotor(Right, Reverse, RightSpeed+12) == 0)
										MotorState = FlipState+4;
									break;

				case FlipState+4:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
										MotorState = FlipState+5;
									break;

				// Be sure that first we see white.
				case FlipState+5:	if(OutRight SeesWhite)
										MotorState = FlipState+6;
									break;


				// We will turn until the outside right sees black.
				case FlipState+6:	if(OutRight SeesBlack)
										MotorState = FlipState+7;
									break;


				// Then we will back up the left side until we see black.
				case FlipState+7:	if(CommandMotor(Left, Reverse, LeftSpeed) == 0)
										MotorState = FlipState+8;
									break;
				
				case FlipState+8:	if(CommandMotor(Right, Reverse, 0x00) == 0)
									MotorState = FlipState+9;
									break;

				case FlipState+9:	if(OutLeft SeesBlack)
										MotorState = FlipState+10;
									break;

				// Then we will drive both motors in reverse until the outside
				// left sees white.
				case FlipState+10:	if(CommandMotor(Left, Reverse, LeftSpeed) == 0)
										MotorState = FlipState+11;
									break;

				case FlipState+11:	if(CommandMotor(Right, Reverse, RightSpeed) == 0)
										MotorState = FlipState+12;
									break;

				case FlipState+12:	if(OutLeft SeesAllWhite)
									{
										MotorState = FlipState+13;
									}
									break;

				// Then we will stop and wait 500ms.
				case FlipState+13:	if(CommandMotor(Left, Reverse, 0x00) == 0)
										MotorState = FlipState+14;
									break;

				case FlipState+14:	if(CommandMotor(Right, Reverse, 0x00) == 0)
										MotorState = FlipState+15;
									break;
	
				case FlipState+15:	count++;
									if(count == 50)
									{
										count = 0;
										MotorState = FlipState+16;
									}
									break;	

				// Then we will determine where the outside right is. We assume
				// he is either over the line or past the line.
				case FlipState+16:	if(OutRight SeesAllWhite)
										MotorState = FlipState+17;
									else
										MotorState = FlipState+26;
									break;

				// If he is past the line then move forward until he sees black.
				case FlipState+17:	if(CommandMotor(Left, Reverse, 0x00) == 0)
									{
										MotorState = FlipState+18;
									}
									break;

				case FlipState+18:	if(CommandMotor(Right, Forward, RightSpeed) == 0)
										MotorState = FlipState+19;
									break;				
				
				case FlipState+19:	if(OutRight SeesBlack)
										MotorState = FlipState+20;
									break;

				// Then move the left side forward until the outside left sees black.
				case FlipState+20:	if(CommandMotor(Left, Forward, LeftSpeed) == 0)
										MotorState = FlipState+21;
									break;

				case FlipState+21:	if(CommandMotor(Right, Forward, 0x00) == 0)
										MotorState = FlipState+22;
									break;	

				case FlipState+22:	if(OutLeft SeesBlack)
										MotorState = FlipState+23;
									break;	

				// Then move both motors forward until we are sure they are both
				// seeing black and then make a right turn.
				case FlipState+23:	if(CommandMotor(Right, Forward, LeftSpeed) == 0)
										MotorState = FlipState+24;
									break;

				case FlipState+24:	if(CommandMotor(Left, Forward, RightSpeed) == 0)
										MotorState = FlipState+25;
									break;
		
				case FlipState+25:	if(OutLeft SeesBlack && OutRight SeesBlack)
									{
										MotorState = RightState;
									}
									break;

				// If we is over the line then we back up until he sees white and act
				// as though he overshot the line.
				case FlipState+26:	if(CommandMotor(Left, Reverse, 0x00) == 0)
										MotorState = FlipState+27;
									break;

				case FlipState+27:	if(CommandMotor(Right, Reverse, RightSpeed) == 0)
										MotorState = FlipState+28;
									break;
			
				case FlipState+28:	if(OutRight SeesAllWhite)
										MotorState = FlipState+17;
									break;

				// ##################################################################
				// FindBallState is where we will determine if there is a ball in
				// front of us or not.
				// ##################################################################
				// First we will stop the motors
				case FindBallState:		if(CommandMotor(Left, Reverse, 0x00) == 0)
											MotorState = FindBallState+1;
										break;

				case FindBallState+1:	if(CommandMotor(Right, Reverse, 0x00) == 0)
											MotorState = FindBallState+2;
										break;

				// Wait for 500ms.
				case FindBallState+2:	count++;
										if(count == 50)
										{
											count = 0;
											MotorState = FindBallState+3;
										}
										break;

				// Determine if there is a ball there or not and then flip around.
				case FindBallState+3:	GetMutex();
										if(USPulse() < USThreshold)
											BallTUFound = 1;
										else
											BallTUFound = 0;
										while(ReleaseMutex() == -1);

										BallToUpdate = Ball[MotorStateBallIndex].BallLetter;
										UpdateBallThere = 1;
										WaitForMutex = 1;
										
										MotorStateBallIndex++;
										if(MotorStateBallIndex == 8)
											MotorStateBallIndex = 0;

										MotorState = FlipState;
										break;

				// ##################################################################
				// ReadBallState will read the data from the ball in front of us.
				// ##################################################################
				// First we will stop the motors
				case ReadBallState:		if(CommandMotor(Left, Reverse, 0x00) == 0)
											MotorState = ReadBallState+1;
										break;

				case ReadBallState+1:	if(CommandMotor(Right, Reverse, 0x00) == 0)
											MotorState = ReadBallState+2;
										break;

				// Wait for 500ms.
				case ReadBallState+2:	count++;
										if(count == 50)
										{
											count = 0;
											MotorState = ReadBallState+3;
										}
										break;

				// Read the ball data and then flip around.
				case ReadBallState+3:	GetMutex();
										BallIRData = GetIRDA();
										while(ReleaseMutex() == -1);

										BallToUpdate = Ball[MotorStateBallArray[MotorStateBallIndex]].BallLetter;
										UpdateBallData = 1;
										WaitForMutex = 1;

										MotorStateBallIndex++;
										if(MotorStateBallIndex == 8)
											MotorStateBallIndex = 0;

										MotorState = FlipState;
										break;

				// ##################################################################
				// Stop State is an all stop state that will be used for debugging
				// ##################################################################
				case StopState:		if(CommandMotor(Left, Reverse, 0x00) == 0)
										MotorState = StopState+1;
									break;

				case StopState+1:	if(CommandMotor(Right, Reverse, 0x00) == 0)
										MotorState = StopState+2;
									break;

				case StopState+2:	MotorState = StopState+2;
									break;	

				// ##################################################################
				// Wait state is just loop unto itself.
				// ##################################################################
				case WaitState:		MotorState = WaitState;
									break;

			} // End Switch
		}		

		// This message is to indicate that the
		// XB module has sent us data for us to
		// process.
		if(XBMessRec && WriteEEPROMRX == 0)
		{	
			XBMessRec = 0;		// Clear the flag.

			// Here we will look at what data we received and
			// determine what to do.
			switch(RecData.Data[0])
			{
				// ---------------------------------------------------------------------------------
				// Here begins the MarsSat to robot messages.
				// Note that the first arguement when receiving from MarsSat is the robot and the 
				// first arguement when sending to MarsSat is the robot.
				// ---------------------------------------------------------------------------------
				// This is when MarsSat asks us if we are here. We will reply
				// with an ACK and our status as long as we are not dead or we
				// have lost sat com.
				case AreYouThereRobots:	if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoLocalCOM ||
									       Robot[MYARG].Status == NoMove)		// Determine if we can talk to MarsSat.
										{
										 	TransData.DestAddLSB = MarsSat;		// Sending to MarsSat.
											TransData.Data[0] = IAmHere;		// Telling MarsSat that we are here.
											TransData.Data[1] = MYARG;			// First Arg is always our robot number.
											TransData.Data[2] = Robot[MYARG].Status;	// Send our actual status.
											
											// Write the Transmitted data to the EEPROM.
											while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
											EEPROMAddress += sizeof(TransData.Data);
											
											while(XBWriteAPI() == -1);					// Write the API.
										}

									LEDOn(LED1);	// Turn on an LED to indicate that we got this message.

									break;

				// Here is where MarsSat has asked us if we can talk to another robot.
				// We will try to send an AreYouThere message to the robot to see if
				// he can talk to us.
				case Talk:			temp = RecData.Data[2]+3;			// Calculate who we send to.
									TransData.DestAddLSB = temp;
									TransData.Data[0] = RRAreYouThere;	// Ask the robot if he is there.
									TransData.Data[1] = temp-3;			// First Arg when sending to robot is RX robot.
									
									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);
									
									while(XBWriteAPI() == -1);			// Write the API.

									Wait100ms();	// Wait100ms for the robot to get back to us.

									// Check to see if we got a message. If did then send 
									// the status of the robot to MarsSat.
									if(XBMessRec == 1)
										TransData.Data[3] = RecData.Data[2];
									// If we didn't get a message we can assume the robot
									// is dead so that is what we will send to MarsSat.
									else				
										TransData.Data[3] = Dead;

									XBMessRec = 0;	// Clear the flag;

									// Since we are not in the main loop to service the RX EEPROM write
									// we need to do it here. Thus we will write the RX data, increment 
									// the EEPROM address, and clear the flag. This is slightly dangerous
									// as we will block until we can do this but for our implementation
									// this command occurs before we begin moving and thus it is not
									// imperative that we service the other messages in the main loop.
									while(WriteEEPROM(EEPROMAddress, RecData.Data, sizeof(RecData.Data)));	
									EEPROMAddress += sizeof(RecData.Data);	
									WriteEEPROMRX = 0;

									TransData.DestAddLSB = MarsSat;	// Sending to MarsSat.
									TransData.Data[0] = ICanTalk;	// Saying that we can talk to robot.
									TransData.Data[1] = MYARG;		// First Arg is always our robot number.
									TransData.Data[2] = temp-3;		// What robot.
									
									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);
									
									while(XBWriteAPI() == -1);		// Write the API.
									break;

				// When the MarsSat has gathered all the status data of all the robots then
				// he will send all that data to the robots in turn using this message.
				case MRDamage:		Robot[RecData.Data[2]].Status = RecData.Data[3];
									break;

				// In the event that one of the other robots has had his SatCOM damaged
				// MarsSat may ask us to relay all of the damage info using this message.
				case RelayDamage:	TransData.DestAddLSB = RecData.Data[2]+3;	// Calculate dest address.
									TransData.Data[0] = RRDamReport;		// Sending the damage report.
									TransData.Data[1] = RecData.Data[2];	// First Arg when sending to robot is RX robot.
									TransData.Data[2] = Robot[0].Status;	// Send Seeker1 status.
									TransData.Data[3] = Robot[1].Status;	// Send Seeker2 status.
									TransData.Data[4] = Robot[2].Status;	// Send Reader1 status.
									TransData.Data[5] = Robot[3].Status;	// Send Reader2 status.

									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);
									
									while(XBWriteAPI() == -1);				// Write the API.
									break;

				// This is the Go command from MarsSat that tells us all status information passing is done
				// and we should begin the task. We will set a flag so that we can handle this in the main
				// loop.
				case Go:			if(Robot[MYARG].TypeOfRobot == Seeker)
										GoFlag = 1;			// Flag to say that we want to turn on Motors if we are a Seeker.

									// Determines if we are the robot to relay the data.
									while(Robot[RelayGoIndex].Status == Dead || 		// Determine the which Robot
										  Robot[RelayGoIndex].Status == NoSatCOM ||		// should relay the data.
										  Robot[RelayGoIndex].Status == NoLocalCOM)
										RelayGoIndex++;
		
									// Now check to see if that robot is us.
									if(RelayGoIndex == MYARG)
									{
										RelayGoFlag = 1;	// If it is us then set a flag so relay in the main loop.
										WaitForMutex = 1;	// We will need the mutex for that.
									}
		
									RelayGoIndex = 0;			// Clear the index.
									break;

				// Here MarsSat has relayed to us the ball status data. It has done this because either
				// we had LocalCOM damage and could not get the message and/or the other robot had LocalCOM damage
				// and couldn't tell us. Once we store the message we need to determine if one of the remaining
				// robots needs our help to send it the location info.
				// An interesting condition could occur here. We could get up to two (but not more than two) messages
				// to relay to a single robot at the same time. Very unlikely but it could happen. What we do here
				// is flag within the ball struct itself what robot we should relay to. Then we set a flag that tells
				// the main loop that we want to relay the ball data to another robot.
				case BallNOTFound:
				case BallFound:		// Update that a ball is found.
									// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
									{
										if(RecData.Data[0] == BallFound)
											Ball[RecData.Data[2] - 0x41].Status = There;
										else
											Ball[RecData.Data[2] - 0x41].Status = NotThere;
									}
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										break;
									}
								
									// Now determine if anyone else needs data relayed to them 
									// and if so determine if we are the ones that should do that.
									// First determine if the original robot was sick with NoLocalCOM
									if(Robot[RecData.Data[3]].Status != NoLocalCOM)
										break;		// If he wasn't sick then break we are done.

									// Store which robot we got the data from in a temp variable.
									temp = RecData.Data[3];

									// Determine if we can/should relay data to other robots.
									while(1)
									{
										temp++;			// Incrementing temp checks the next robot.
										if(temp == 4)	// If we reach 4 then we need to roll over.
											temp = 0;
	
										// Check to see if this us. If it is then be sure that we are able
										// to send to other robots.
										if(temp == MYARG)
										{
											// Check to be sure that we can send to the robot. If we can, flag
											// which robot should have the message relayed to him so that
											// we can do this in the main loop.
											if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
											{
												// If Seeker1 is not us and not the robot that has updated Ball status and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Seeker1Arg != MYARG && Seeker1Arg != RecData.Data[3] && Robot[Seeker1Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayStatRobot[Seeker1Arg] = 1;
												
												// If Seeker2 is not us and not the robot that has updated Ball status and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Seeker2Arg != MYARG && Seeker2Arg != RecData.Data[3] && Robot[Seeker2Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayStatRobot[Seeker2Arg] = 1;
												
												// If Reader1 is not us and not the robot that has an updated Ball status and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Reader1Arg != MYARG && Reader1Arg != RecData.Data[3] && Robot[Reader1Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayStatRobot[Reader1Arg] = 1;

												// If Reader2 is not us and not the robot that has an updated Ball status and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Reader2Arg != MYARG && Reader2Arg != RecData.Data[3] && Robot[Reader2Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayStatRobot[Reader2Arg] = 1;
	
												UpdateBTRRelay = 1;			// Flag to say that we want to relay ball data to another robot.
												WaitForMutex = 1;			// Flag that we want the mutex.
												break;
											}
											// If we are unable to send messages to other robots then we are done. 
											// Break out to the main loop.
											else
												break;
										}
	
										// Check to see if the next robot that is responsible for relaying the
										// message is able to do so. If he is able then we are done because he
										// will relay the messages. Break out to the main loop.
										else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
											break;
									}

									break;

				// Here MarsSat has relayed to us the ball IR data. It has done this because either
				// we had LocalCOM damage and could not get the message and/or the other robot had LocalCOM damage
				// and couldn't tell us. Once we store the message we need to determine if one of the remaining
				// robots needs our help to send it the location info.
				// An interesting condition could occur here. We could get up to two (but not more than two) messages
				// to relay to a single robot at the same time. Very unlikely but it could happen. What we do here
				// is flag within the ball struct itself what robot we should relay to. Then we set a flag that tells
				// the main loop that we want to relay the ball data to another robot.
				case IRData:		// Update the Ball IR Data.
									// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
									{
											Ball[RecData.Data[2] - 0x41].IRData = RecData.Data[3];
									}
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										break;
									}
								
									// Now determine if anyone else needs data relayed to them 
									// and if so determine if we are the ones that should do that.
									// First determine if the original robot was sick with NoLocalCOM
									if(Robot[RecData.Data[4]].Status != NoLocalCOM)
										break;		// If he wasn't sick then break we are done.

									// Store which robot we got the data from in a temp variable.
									temp = RecData.Data[4];

									// Determine if we can/should relay data to other robots.
									while(1)
									{
										temp++;			// Incrementing temp checks the next robot.
										if(temp == 4)	// If we reach 4 then we need to roll over.
											temp = 0;
	
										// Check to see if this us. If it is then be sure that we are able
										// to send to other robots.
										if(temp == MYARG)
										{
											// Check to be sure that we can send to the robot. If we can, flag
											// which robot should have the message relayed to him so that
											// we can do this in the main loop.
											if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
											{
												// If Seeker1 is not us and not the robot that updated Ball IR Data and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Seeker1Arg != MYARG && Seeker1Arg != RecData.Data[4] && Robot[Seeker1Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayDataRobot[Seeker1Arg] = 1;
												
												// If Seeker2 is not us and not the robot that updated Ball IR Data and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Seeker2Arg != MYARG && Seeker2Arg != RecData.Data[4] && Robot[Seeker2Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayDataRobot[Seeker2Arg] = 1;
												
												// If Reader1 is not us and not the robot that updated Ball IR Data and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Reader1Arg != MYARG && Reader1Arg != RecData.Data[4] && Robot[Reader1Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayDataRobot[Reader1Arg] = 1;

												// If Reader2 is not us and not the robot that updated Ball IR Data and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Reader2Arg != MYARG && Reader2Arg != RecData.Data[4] && Robot[Reader2Arg].Status == NoSatCOM)
													Ball[RecData.Data[2] - 0x41].RelayDataRobot[Reader2Arg] = 1;
	
												UpdateBDRRelay = 1;			// Flag to say that we want to relay ball IR data to another robot.
												WaitForMutex = 1;			// Flag that we want the mutex.
												break;
											}
											// If we are unable to send messages to other robots then we are done. 
											// Break out to the main loop.
											else
												break;
										}
	
										// Check to see if the next robot that is responsible for relaying the
										// message is able to do so. If he is able then we are done because he
										// will relay the messages. Break out to the main loop.
										else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
											break;
									}

									break;

				// When EarthSat has issued a command via MarsSat that we should stop we will get here.
				// Simply flag it so that we deal with it in the main loop.
				case Stop:			StopFlag = 1;
									break;

				// When MarsSat wants to tell a robot to stop but can't communicate with it then
				// it may send a message to us to relay that Stop command. If that happens we get here.
				case RelayStop:		RelayStopFlag = 1;					// Set a flag so we handle this in main loop.		
									WaitForMutex = 1;					// We need the mutex to relay command.
									RelayStopRobot = RecData.Data[2];	// Robot we need to relay to.
									break;	

				// When EarthSat has issued a command via MarsSat that we should resume we will get here.
				// Simply flag it so that we deal with it in the main loop.
				case Resume:		ResumeFlag = 1;
									break;

				// When MarsSat wants to tell a robot to resume but can't communicate with it then
				// it may send a message to us to relay that Resume command. If that happens we get here.
				case RelayResume:	RelayResumeFlag = 1;				// Set a flag so we handle this in main loop.		
									WaitForMutex = 1;					// We need the mutex to relay command.
									RelayResumeRobot = RecData.Data[2];	// Robot we need to relay to.
									break;

				// This message is to dump our EEPROM data to MarsSat. The first packet contains our command,
				// our source ID, two bytes for the size, and then four bytes for data. Every other packet
				// will contain one byte for the command followed by seven bytes of data. We will read from the
				// EEPROM until we have reached a 0x00 in the first byte (as we don't have a command of 0x00
				// this indicates that there are no more stored commands) of the message or we have reached 1024
				// sets.
				case DumpEEPROM:	TransData.DestAddLSB = MarsSat;		// Sending to MarsSat.
									TransData.Data[0] = REEPROM;		// Sending out EEPROM.
									TransData.Data[1] = MYARG;			// Next arg is the source of the EEPROM.

									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);

									// First Determine our size. We do this by going through the EEPROM data
									// 8 bytes at a time. No command begins with 0x00 so if we find the first
									// byte in a set of 8 bytes to be 0x00 then we have reached the end of the list.
									// The maximum is 1024 sets of bytes. If we go that far we are done.
									size = 0;
									while(size < 1024)						// Won't go more than 1024 sets.
									{
										while(ReadEEPROM(size << 3, buffer, 8));	// Read the next eight bytes.
										if(buffer[0] == 0)					// If we read 0x00 in the first byte
											break;							// Then we are done with list.
										else
											size++;							// Otherwise increment and check the next set.
									}
								
									// Multiply size by 8.
									size = size << 3;

									// Assign the next two arguements to the size.
									TransData.Data[2] = (unsigned char)size;
									TransData.Data[3] = (unsigned char)(size >> 8);

									// This is a fun challenge. We have one eight byte array to read from
									// and one seven byte array to read to. When we reach the end of the seven
									// byte array then we write that to the XBeeAPI. When we reach the end of the
									// eight byte array (which is the array read from the EEPROM) then get
									// eight more bytes from the EEPROM. If we read a 0x00 from the first byte
									// in the next eight byte set then we are done. But, we need to check that
									// there maybe data we haven't sent to the XBee API because we have not
									// reached the limit of the seven byte array just yet. If there is such
									// data then send it.

									// Set counter1=4 because that is where we begin the seven byte array due
									// to the first packets contains the command, source, and two bytes for size.
									// Counter2=8 to force a read from the EEPROM as soon as we get into the loop.
									counter1=4;counter2=8;size=0;
									while(1)
									{
										// If we have reached the end of the eight byte array then read in new
										// eight bytes and determine if we are done by looking at the first byte.
										if(counter2 == 8)
										{
											while(ReadEEPROM(size << 3, buffer, 8));	// Reads 8 bytes from the EEPROM.
											counter2 = 0;						// Reset the index to the 8 byte array.
											size++;								// Increment the size. No more than 1024 sets.
											if(buffer[0] == 0 || size == 1024)	// If the first byte is 0x00 or we have 1024 sets.
											{
												if(counter1 != 1)				// Check to see if we need to do another XBee write.
													while(XBWriteAPI() == -1);	// Write the API.

												break;							// We are done so break out of the loop.
											}
										}

										// If we have reached the end of the seven byte array then we need to write
										// the data of that array to the XBee module. This is slightly deceiving as
										// we wait for counter1 to be 8 but remember that the first byte is always the
										// command, which leaves us with seven bytes for the data (unless it is the first
										// data packet). That is why we reset the counter1 index to 1.
										if(counter1 == 8)
										{
											while(XBWriteAPI() == -1);			// Write the API.
											counter1 = 1;
										}

										// Assign the buffer data to the XBee API data and increment respective indexes.
										TransData.Data[counter1] = buffer[counter2];
										counter1++;counter2++;												
									}

									break;	

				// This message will get called when a robot we can talk to can't talk to MarsSat directly. As
				// a result MarsSat asks us to get the EERPOM data for him.
				case RelayDumpEEPROM:	// Build an array of zeros for clearing the EERPOM. We should clear
										// the EEPROM because there maybe other data in there from another
										// operation.
										for(index = 0;index < 64;index++)	// Clear the array.
											array[index] = 0;
						
										// Clear the EEPROM.
										for(index = 0x80;index < 0x100;index++)
											while(WriteEEPROM(index << 6, array, sizeof(array)));

										// Fill in the Transmit struct.
										TransData.DestAddLSB = RecData.Data[2]+3;	// Sending to MarsSat.
										TransData.Data[0] = RRDumpEEPROM;			// Telling robot to dump EEPROM.
										TransData.Data[1] = RecData.Data[2];		// Next arg is the robot destination.

										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);
									
										while(XBWriteAPI() == -1);				// Write the API.
										break;


				// During the task setup MarsSat will tell us what our damage is
				// using the command.
				case YourDamage:	Robot[MYARG].Status = RecData.Data[2];
									break;

				// Here MarsSat has relayed to us the Location of another robot. It has done this because either
				// we had LocalCOM damage and could not get the message and/or the other robot had LocalCOM damage
				// and couldn't tell us. Once we store the message we need to determine if one of the remaining
				// robots needs our help to send it the location info.
				// An interesting condition could occur here. We could get up to two (but not more than two) messages
				// to relay to a single robot at the same time. Very unlikely but it could happen. So every flag
				// for relaying a location message to another robot is acutally a two byte array. We will store the
				// first relay in the first index and if there is a second message we will store the value in the
				// second index. In the main loop we will be careful to clear the value in the array when we are
				// done as that is the method for determining here and in the main loop if there is data there.
				// If there is nothing do send then the flag will be 0xff else the flag will be the arguement of 
				// the robot location data we need to send to. 
				case MRLocation:	Robot[RecData.Data[2]].CurrentLocation = RecData.Data[3];	// Store the data.

									// If we are a Reader then determine if it is our time to go. If so
									// then set the go flag which will turn on our motors in the main loop.
									if(Robot[MYARG].TypeOfRobot == Reader && RecData.Data[2] == Seeker1Arg)
									{
										// If Seeker2 is operational.
										if( Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoLocalCOM || 
											Robot[Seeker2Arg].Status == NoSatCOM)
										{
											// When Seeker1 reaches 7 and Seeker2 reaches 8 then we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7 && Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
										// Else if Seeker2 is not operational.
										else
										{
											// Wait until Seeker1 reaches 7 to before we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
									}
									if(Robot[MYARG].TypeOfRobot == Reader && RecData.Data[2] == Seeker2Arg)
									{
										// If Seeker1 is operational.
										if( Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoLocalCOM || 
											Robot[Seeker1Arg].Status == NoSatCOM)
										{
											// When Seeker1 reaches 7 and Seeker2 reaches 8 then we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7 && Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
										// Else if Seeker1 is not operational.
										else
										{
											// Wait until Seeker1 reaches 7 to before we can go.
											if(Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
									}
									
									// Now determine if anyone else needs data relayed to them 
									// and if so determine if we are the ones that should do that.
									// First determine if the original robot was sick with NoLocalCOM
									if(Robot[RecData.Data[2]].Status != NoLocalCOM)
										break;		// If he wasn't sick then break we are done.

									// Store which robot we got the data from in a temp variable.
									temp = RecData.Data[2];

									// Determine if we can/should relay data to other robots.
									while(1)
									{
										temp++;			// Incrementing temp checks the next robot.
										if(temp == 4)	// If we reach 4 then we need to roll over.
											temp = 0;
	
										// Check to see if this us. If it is then be sure that we are able
										// to send to other robots.
										if(temp == MYARG)
										{
											// Check to be sure that we can send to the robot. If we can, flag
											// which robot should have the message relayed to him so that
											// we can do this in the main loop.
											if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
											{
												// If Seeker1 is not us and not the robot that has an updates location and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Seeker1Arg != MYARG && Seeker1Arg != RecData.Data[2] && Robot[Seeker1Arg].Status == NoSatCOM)
												{
													// Determine if we have already flagged a relay message to Seeker1. If
													// we have then we need to flag it in the second index of the array.
													if(RelayLocNoSatCOMS1[0] == 0xff)
														RelayLocNoSatCOMS1[0] = RecData.Data[2];
													else
														RelayLocNoSatCOMS1[1] = RecData.Data[2];
												}
												// If Seeker2 is not us and not the robot that has an updates location and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Seeker2Arg != MYARG && Seeker2Arg != RecData.Data[2] && Robot[Seeker2Arg].Status == NoSatCOM)
												{
													// Determine if we have already flagged a relay message to Seeker2. If
													// we have then we need to flag it in the second index of the array.
													if(RelayLocNoSatCOMS2[0] == 0xff)
														RelayLocNoSatCOMS2[0] = RecData.Data[2];
													else
														RelayLocNoSatCOMS2[1] = RecData.Data[2];
												}
												// If Reader1 is not us and not the robot that has an updates location and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Reader1Arg != MYARG && Reader1Arg != RecData.Data[2] && Robot[Reader1Arg].Status == NoSatCOM)
												{
													// Determine if we have already flagged a relay message to Reader1. If
													// we have then we need to flag it in the second index of the array.
													if(RelayLocNoSatCOMR1[0] == 0xff)
														RelayLocNoSatCOMR1[0] = RecData.Data[2];
													else
														RelayLocNoSatCOMR1[1] = RecData.Data[2];
												}
												// If Reader2 is not us and not the robot that has an updates location and
												// was unable to get the relayed message from MarsSat then we need to relay
												// him the message.
												if(Reader2Arg != MYARG && Reader2Arg != RecData.Data[2] && Robot[Reader2Arg].Status == NoSatCOM)
												{
													// Determine if we have already flagged a relay message to Reader2. If
													// we have then we need to flag it in the second index of the array.
													if(RelayLocNoSatCOMR2[0] == 0xff)
														RelayLocNoSatCOMR2[0] = RecData.Data[2];
													else
														RelayLocNoSatCOMR2[1] = RecData.Data[2];
												}
	
												WaitForMutex = 1;			// Flag that we want the mutex.
												break;
											}
											// If we are unable to send messages to other robots then we are done. 
											// Break out to the main loop.
											else
												break;
										}
	
										// Check to see if the next robot that is responsible for relaying the
										// message is able to do so. If he is able then we are done because he
										// will relay the messages. Break out to the main loop.
										else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
											break;
									}
	
									break;

				// ----------------------------------------------------------------------------
				// Here begins the robot to robot messages.
				// Note that the first arguement is the receiving robot.
				// ----------------------------------------------------------------------------
				// When MarsSat hasn't heard from us he will ask another robot to try and talk
				// to us. If we are not dead then we should answer.
				case RRAreYouThere:	if(Robot[MYARG].Status != Dead)
									{
										TransData.DestAddLSB = RecData.SourceAddLSB;// The desination is the message source.
										TransData.Data[0] = RRIAmHere;				// Tell robot we are here.
										TransData.Data[1] = RecData.SourceAddLSB-3;	// First Arg is always the receiving robot.
										TransData.Data[2] = Robot[MYARG].Status;	// Tell robot our status.
										
										// Write the Transmitted data to the EEPROM.
										while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
										EEPROMAddress += sizeof(TransData.Data);
										
										while(XBWriteAPI() == -1);					// Write the API.
									}
									break;

				// If we were unable to speak to MarsSat then we would never get the Go message.
				// Thus, one of our robot bretheren may decide to relay that message to us which
				// would lead us here. We will simply set a flag to handle this in the main loop.
				case RRGo:			if(Robot[MYARG].TypeOfRobot == Seeker)
										GoFlag = 1;			// Flag to say that we want to relay Go message to another robot.
									break;
				
				// We get this message when another robot is updating us with his location. Update that location
				// in our Robot struct array. Then check to see the status of that robot. If he can't
				// talk to MarsSat then determine if we should on his behalf.
				case RRLocation:	// Update Location.
									Robot[RecData.SourceAddLSB-3].CurrentLocation = RecData.Data[2];

									// If we are a Reader then determine if it is our time to go. If so
									// then set the go flag which will turn on our motors in the main loop.
									if((Robot[MYARG].TypeOfRobot == Reader) && ((RecData.SourceAddLSB-3) == Seeker1Arg))
									{
										// If Seeker2 is operational.
										if( Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoLocalCOM || 
											Robot[Seeker2Arg].Status == NoSatCOM)
										{
											// When Seeker1 reaches 7 and Seeker2 reaches 8 then we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7 && Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
										// Else if Seeker2 is not operational.
										else
										{
											// Wait until Seeker1 reaches 7 to before we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
									}
									if((Robot[MYARG].TypeOfRobot == Reader) && ((RecData.SourceAddLSB-3) == Seeker2Arg))
									{
										// If Seeker1 is operational.
										if( Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoLocalCOM || 
											Robot[Seeker1Arg].Status == NoSatCOM)
										{
											// When Seeker1 reaches 7 and Seeker2 reaches 8 then we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7 && Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
										// Else if Seeker1 is not operational.
										else
										{
											// Wait until Seeker1 reaches 7 to before we can go.
											if(Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
									}

									// Determine if we should relay the message. First check to see if the sending
									// robot is unable to send to MarsSat. If he is unable determine if we should
									// relay the data on his behalf.
									if(Robot[RecData.SourceAddLSB-3].Status == NoSatCOM)
									{
										temp = RecData.SourceAddLSB-3;	// Temp variable used as both index and to
																		// reference with robot we are considering.
										while(1)
										{
											temp++;			// Incrementing temp checks the next robot.
											if(temp == 4)	// If we reach 4 then we need to roll over.
												temp = 0;
		
											// Check to see if this us. If it is then be sure that we are able
											// to send to MarsSat.
											if(temp == MYARG)
											{
												// Check to be sure that we can send to MarsSat. If we can flag
												// which robot should have his location relayed and this will be
												// done in the main loop.
												if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
												{
													// Determine which robot and flag that one for relay.
													if(RecData.SourceAddLSB == Seeker1)
														RelayUpdateLocSeeker1 = 1;
													else if(RecData.SourceAddLSB == Seeker2)
														RelayUpdateLocSeeker2 = 1;
													else if(RecData.SourceAddLSB == Reader1)
														RelayUpdateLocReader1 = 1;
													else if(RecData.SourceAddLSB == Reader2)
														RelayUpdateLocReader2 = 1;

													WaitForMutex = 1;			// Flag that we want the mutex.
													break;
												}
												// If we can't then we are done. Break out to the main loop.
												else
													break;
											}

											// Check to see if the next robot that is responsible for relaying the
											// message is able to do so. If he is able then we are done. Break out
											// to the main loop.
											else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
												break;
										}
									}
									break;

				// We get this message when another robot is updating us that the ball is there. Update that
				// in our Ball struct array. Then check to see the status of that robot. If he can't
				// talk to MarsSat then determine if we should on his behalf.
				case RRBallFound:	// Update that a ball is found.
									// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
										Ball[RecData.Data[2] - 0x41].Status = There;
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										break;
									}

									// Determine if we should relay the message. First check to see if the sending
									// robot is unable to send to MarsSat. If he is unable determine if we should
									// relay the data on his behalf.
									if(Robot[RecData.SourceAddLSB-3].Status == NoSatCOM)
									{
										temp = RecData.SourceAddLSB-3;	// Temp variable used as both index and to
																		// reference with robot we are considering.
										while(1)
										{
											temp++;			// Incrementing temp checks the next robot.
											if(temp == 4)	// If we reach 4 then we need to roll over.
												temp = 0;
		
											// Check to see if this us. If it is then be sure that we are able
											// to send to MarsSat.
											if(temp == MYARG)
											{
												// Check to be sure that we can send to MarsSat. If we can, flag
												// the ball's relay status flag to relay in the main loop.
												if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
												{
													// Set a flag in the ball struct to indicate that we should
													// relay the ball's status to MarsSat. 
													Ball[RecData.Data[2] - 0x41].RelayStatus = 1;

													// Indicate what robot told us this information.
													Ball[RecData.Data[2] - 0x41].Robot = RecData.SourceAddLSB-3;

													// Flag so that the main loop will relay ball status to EarthSat.
													UpdateBTRelay = 1;

													WaitForMutex = 1;			// Flag that we want the mutex.
													break;
												}
												// If we can't then we are done. Break out to the main loop.
												else
													break;
											}

											// Check to see if the next robot that is responsible for relaying the
											// message is able to do so. If he is able then we are done. Break out
											// to the main loop.
											else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
												break;
										}
									}
									break;

				// We get this message when another robot is updating us that the ball is not there. Update that
				// in our Ball struct array. Then check to see the status of that robot. If he can't
				// talk to MarsSat then determine if we should on his behalf.
				case RRBallNotFound:// Update that a ball is not found.
									// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
										Ball[RecData.Data[2] - 0x41].Status = NotThere;
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										break;
									}

									// Determine if we should relay the message. First check to see if the sending
									// robot is unable to send to MarsSat. If he is unable determine if we should
									// relay the data on his behalf.
									if(Robot[RecData.SourceAddLSB-3].Status == NoSatCOM)
									{
										temp = RecData.SourceAddLSB-3;	// Temp variable used as both index and to
																		// reference with robot we are considering.
										while(1)
										{
											temp++;			// Incrementing temp checks the next robot.
											if(temp == 4)	// If we reach 4 then we need to roll over.
												temp = 0;
		
											// Check to see if this us. If it is then be sure that we are able
											// to send to MarsSat.
											if(temp == MYARG)
											{
												// Check to be sure that we can send to MarsSat. If we can, flag
												// the ball's relay status flag to relay in the main loop.
												if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
												{
													// Set a flag in the ball struct to indicate that we should
													// relay the ball's status to MarsSat. 
													Ball[RecData.Data[2] - 0x41].RelayStatus = 1;

													// Flag so that the main loop will relay ball status to EarthSat.
													UpdateBTRelay = 1;

													WaitForMutex = 1;			// Flag that we want the mutex.
													break;
												}
												// If we can't then we are done. Break out to the main loop.
												else
													break;
											}

											// Check to see if the next robot that is responsible for relaying the
											// message is able to do so. If he is able then we are done. Break out
											// to the main loop.
											else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
												break;
										}
									}
									break;

				// We get this message when another robot is updating us with ball data. Update that
				// in our Ball struct array. Then check to see the status of that robot. If he can't
				// talk to MarsSat then determine if we should on his behalf.
				case RRIRData:		// Update the ball IR data.
									// Check the index is in bounds.
									if( ((RecData.Data[2] - 0x41) >= 0) && ((RecData.Data[2] - 0x41) <= 7) )
										Ball[RecData.Data[2] - 0x41].IRData = RecData.Data[3];
									// If the index is out of bounds then turn on an LED and break out.
									else
									{
										LEDOn(LED6);
										break;
									}

									// Determine if we should relay the message. First check to see if the sending
									// robot is unable to send to MarsSat. If he is unable determine if we should
									// relay the data on his behalf.
									if(Robot[RecData.SourceAddLSB-3].Status == NoSatCOM)
									{
										temp = RecData.SourceAddLSB-3;	// Temp variable used as both index and to
																		// reference with robot we are considering.
										while(1)
										{
											temp++;			// Incrementing temp checks the next robot.
											if(temp == 4)	// If we reach 4 then we need to roll over.
												temp = 0;
		
											// Check to see if this us. If it is then be sure that we are able
											// to send to MarsSat.
											if(temp == MYARG)
											{
												// Check to be sure that we can send to MarsSat. If we can, flag
												// the ball's relay status flag to relay in the main loop.
												if(Robot[MYARG].Status == OK || Robot[MYARG].Status == NoMove)
												{
													// Set a flag in the ball struct to indicate that we should
													// relay the ball's IR Data to MarsSat. 
													Ball[RecData.Data[2] - 0x41].RelayData = 1;

													// Indicate what robot told us this information.
													Ball[RecData.Data[2] - 0x41].Robot = RecData.SourceAddLSB-3;

													// Flag so that the main loop will relay ball IR Data to EarthSat.
													UpdateBDRelay = 1;

													WaitForMutex = 1;			// Flag that we want the mutex.
													break;
												}
												// If we can't then we are done. Break out to the main loop.
												else
													break;
											}

											// Check to see if the next robot that is responsible for relaying the
											// message is able to do so. If he is able then we are done. Break out
											// to the main loop.
											else if(Robot[temp].Status == OK || Robot[temp].Status == NoMove)
												break;
										}
									}
									break;

				// When MarsSat wants us to stop but can't talk to us then he will ask another robot
				// to relay that message. If that happens we will get here. Simply set a flag to
				// stop in the main loop.
				case RRStop:		StopFlag = 1;	// Set the stop flag to stop in the main loop.
									break;

				// When MarsSat wants us to resume but can't talk to us then he will ask another robot
				// to relay that message. If that happens we will get here. Simply set a flag to
				// stop in the main loop.
				case RRResume:		ResumeFlag = 1;	// Set the stop flag to resume in the main loop.
									break;		

				// When MarsSat is unable to talk to us because we have no SatCom then it will ask
				// another robot get the EEPROM from us and then the other robot will relay that
				// EEPROM data. Here the other robot has asked for our EEPROM data.
				case RRDumpEEPROM:	TransData.DestAddLSB = RecData.SourceAddLSB;	// Sending to Requesting robot
									TransData.Data[0] = RRMyEEPROM;					// Sending out EEPROM.
									TransData.Data[1] = RecData.SourceAddLSB-3;		// First Arg is always the receiving robot.

									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);

									// First Determine our size. We do with by going through the EEPROM data
									// 8 bytes at a time. No command begins with 0x00 so if we find the first
									// byte in a set of 8 bytes to be 0x00 then we have reached the end of the list.
									// The maximum is 1024 sets of bytes. If we go that far we are done.
									size = 0;
									while(size < 1024)						// Won't go more than 1024 sets.
									{
										while(ReadEEPROM(size << 3, buffer, 8));	// Read the next eight bytes.
										if(buffer[0] == 0)					// If we read 0x00 in the first byte
											break;							// Then we are done with list.
										else
											size++;							// Otherwise increment and check the next set.
									}
								
									// Multiply size by 8.
									size = size << 3;

									// Assign the next two arguements to the size.
									TransData.Data[2] = (unsigned char)size;
									TransData.Data[3] = (unsigned char)(size >> 8);

									// This is a fun challenge. We have one eight byte array to read from
									// and one seven byte array to read to. When we reach the end of the seven
									// byte array then we write that to the XBeeAPI. When we reach the end of the
									// eight byte array (which is the array read from the EEPROM) then get
									// eight more bytes from the EEPROM. If we read a 0x00 from the first byte
									// in the next eight byte set then we are done. But, we need to check that
									// there maybe data we haven't sent to the XBee API because we have not
									// reached the limit of the seven byte array just yet. If there is such
									// data then send it.

									// Set counter1=4 because that is where we begin the seven byte array due
									// to the first packets contains the command, source, and two bytes for size.
									// Counter2=8 to force a read from the EEPROM as soon as we get into the loop.
									counter1=4;counter2=8;size=0;
									while(1)
									{
										// If we have reached the end of the eight byte array then read in new
										// eight bytes and determine if we are done by looking at the first byte.
										if(counter2 == 8)
										{
											while(ReadEEPROM(size << 3, buffer, 8));	// Reads 8 bytes from the EEPROM.
											counter2 = 0;						// Reset the index to the 8 byte array.
											size++;								// Increment the size. No more than 1024 sets.
											if(buffer[0] == 0 || size == 1024)	// If the first byte is 0x00 or we have 1024 sets.
											{
												if(counter1 != 1)				// Check to see if we need to do another XBee write.
													while(XBWriteAPI() == -1);	// Write the API.

												break;							// We are done so break out of the loop.
											}
										}

										// If we have reached the end of the seven byte array then we need to write
										// the data of that array to the XBee module. This is slightly deceiving as
										// we wait for counter1 to be 8 but remember that the first byte is always the
										// command, which leaves us with seven bytes for the data (unless it is the first
										// data packet). That is why we reset the counter1 index to 1.
										if(counter1 == 8)
										{
											while(XBWriteAPI() == -1);			// Write the API.
											counter1 = 1;
										}

										// Assign the buffer data to the XBee API data and increment respective indexes.
										TransData.Data[counter1] = buffer[counter2];
										counter1++;counter2++;												
									}

									break;	

				// This is the case where the robot that MarsSat wants the EEPROM from has sent us the EEPROM. We
				// will collect the EEPROM he has to offer and store in the upper half of our EEPROM. We then 
				// send that very same EEPROM data to MarsSat.
				case RRMyEEPROM:	// Begin by getting the robot data and storing it in the upper
									// part of our EEPROM.
									XArg = RecData.SourceAddLSB-3;		// This arguement is the robot the data is from.
									size = RecData.Data[2];				// Determine the size.
									size += (unsigned int)RecData.Data[3] << 8;

									// While we have not reached the end of the EEPROM read robot data
									// from the XBee receive buffer and store that data in our EEPROM.
									counter1 = 4;counter2 = 0;EEPROMRobotAddress = 0x2000;counter3=0;
									while(EEPROMRobotAddress < 0x4000)
									{
										// Store Robot data in a local buffer for now and increment the counters.
										buffer[counter2] = RecData.Data[counter1];
										counter2++;counter1++;

										// If we have filled the local buffer then write the data to the EEPROM.
										if(counter2 == 8)
										{
											// Store robot data.
											while(WriteEEPROM(EEPROMRobotAddress, buffer, sizeof(buffer)));
											EEPROMRobotAddress += sizeof(buffer);

											counter2 = 0;				// Clear the index counter.

											counter3 += sizeof(buffer);	// Increment the counter.

											// Determine if we are done getting data from the robot. If so, tell
											// Rec ISR to now record data and break out of the loop.
											if(counter3 == size)
											{
												// If we are done then we do want the Rec ISR to store the data.
												RecEEPROM = 0;

												// If we have read all the data from the robot and written all the
												// data to the our EEPROM then we are done and we should break out.
												break;
											}
										}

										// If we reached the end of the Rec buffer from the XBee module then
										// wait for more data from the robot.
										if(counter1 == 8)
										{
											
											// If we are not done then we need to wait for more data from the robot.
											RecEEPROM = 1;				// Tell the Rec ISR not to store the data.
											while(XBMessRec == 0);		// Spin around here waiting for data.
											XBMessRec = 0;				// Clear the flag.
											counter1 = 1;				// First byte is command so we start the data at 1.
										}
									}


									// Now that we have grabbed all the data from the robot we need to read that
									// data back out of the EEPROM and send it to MarsSat.
									TransData.DestAddLSB = MarsSat;		// Sending to MarsSat.
									TransData.Data[0] = REEPROMRelay;	// Sending out EEPROM.
									TransData.Data[1] = XArg;			// Next arg is the source of the EEPROM.

									// Write the Transmitted data to the EEPROM.
									while(WriteEEPROM(EEPROMAddress, TransData.Data, sizeof(TransData.Data)));
									EEPROMAddress += sizeof(TransData.Data);

									// Assign the next two arguements to the size.
									TransData.Data[2] = (unsigned char)size;
									TransData.Data[3] = (unsigned char)(size >> 8);

									// Set counter1=4 because that is where we begin the seven byte array due
									// to the first packets contains the command, source, and two bytes for size.
									// Counter2=8 to force a read from the EEPROM as soon as we get into the loop.
									counter1=4;counter2=8;EEPROMRobotAddress = 0x2000;
									while(1)
									{
										// If we have reached the end of the eight byte array then read in new
										// eight bytes and determine if we are done by looking at the first byte.
										if(counter2 == 8)
										{
											while(ReadEEPROM(EEPROMRobotAddress, buffer, sizeof(buffer)));	// Reads 8 bytes from the EEPROM.
											counter2 = 0;							// Reset the index to the 8 byte array.
											EEPROMRobotAddress += sizeof(buffer);	// Increment the size.
											if(buffer[0] == 0 || EEPROMRobotAddress == 0x4000)	// If the first byte is 0x00 or we have too much data.
											{
												if(counter1 != 1)				// Check to see if we need to do another XBee write.
													while(XBWriteAPI() == -1);	// Write the API.

												break;							// We are done so break out of the loop.
											}
										}

										// If we have reached the end of the seven byte array then we need to write
										// the data of that array to the XBee module. This is slightly deceiving as
										// we wait for counter1 to be 8 but remember that the first byte is always the
										// command, which leaves us with seven bytes for the data (unless it is the first
										// data packet). That is why we reset the counter1 index to 1.
										if(counter1 == 8)
										{
											while(XBWriteAPI() == -1);			// Write the API.
											counter1 = 1;
										}

										// Assign the buffer data to the XBee API data and increment respective indexes.
										TransData.Data[counter1] = buffer[counter2];
										counter1++;counter2++;												
									}

									break;
	
	
				// In the event that our SatCOM becomes damaged MarsSat will instruct another
				// robot to relay the damage info of all the robots to us using this command.
				case RRDamReport:	Robot[0].Status = RecData.Data[2];	// Record Seeker1 Damage.
									Robot[1].Status = RecData.Data[3];	// Record Seeker2 Damage.
									Robot[2].Status = RecData.Data[4];	// Record Reader1 Damage.
									Robot[3].Status = RecData.Data[5];	// Record Reader2 Damage.
									break;

				// Here a robot has relayed to us the location of another robot. For this to happen
				// the robot who has a new location was unable to talk to us. As a result, MarsSat
				// tried to tell all the robots the new location but we were unable to talk to MarsSat.
				// Thus a different robot needed to tell us the new location.
				case RRLocRelay:	Robot[RecData.Data[2]].CurrentLocation = RecData.Data[3];

									// If we are a Reader then determine if it is our time to go. If so
									// then set the go flag which will turn on our motors in the main loop.
									if(Robot[MYARG].TypeOfRobot == Reader && RecData.Data[2] == Seeker1Arg)
									{
										// If Seeker2 is operational.
										if( Robot[Seeker2Arg].Status == OK || Robot[Seeker2Arg].Status == NoLocalCOM || 
											Robot[Seeker2Arg].Status == NoSatCOM)
										{
											// When Seeker1 reaches 7 and Seeker2 reaches 8 then we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7 && Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
										// Else if Seeker2 is not operational.
										else
										{
											// Wait until Seeker1 reaches 7 to before we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
									}
									if(Robot[MYARG].TypeOfRobot == Reader && RecData.Data[2] == Seeker2Arg)
									{
										// If Seeker1 is operational.
										if( Robot[Seeker1Arg].Status == OK || Robot[Seeker1Arg].Status == NoLocalCOM || 
											Robot[Seeker1Arg].Status == NoSatCOM)
										{
											// When Seeker1 reaches 7 and Seeker2 reaches 8 then we can go.
											if(Robot[Seeker1Arg].CurrentLocation == 7 && Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
										// Else if Seeker1 is not operational.
										else
										{
											// Wait until Seeker1 reaches 7 to before we can go.
											if(Robot[Seeker2Arg].CurrentLocation == 8)
												GoFlag = 1;		// Set the go flag so that the main loop will
																// start our motors.
										}
									}

									break;
	
				// Otherwise we don't know what command just came in so light up an LED.
				default:			LEDOn(LED6);
									break;
			}
		}

		// This message is for when we have pressed
		// button1.
		if(Button1Pressed)
		{
			Button1Pressed = 0;	// Clear the flag.
			HumanInt();
		}

		// This message is for when we have pressed
		// button2.
		if(Button2Pressed)
		{
			unsigned char test = 0xff;
			Button2Pressed = 0;	// Clear the flag.
			//while(CommandMotor(Left, Forward, LeftSpeed) == -1);
			//while(CommandMotor(Right, Forward, RightSpeed) == -1);
			//MotorOn = 1;
		/*
			WaitForMutex = 1;
			
			
			if(MYARG == Seeker1Arg)
			{
				BallToUpdate = 'C';
				BallTUFound = 0;
				UpdateBallThere = 1;
			}
			if(MYARG == Seeker2Arg)
			{
				BallToUpdate = 'B';
				BallTUFound = 1;
				UpdateBallThere = 1;
			}
			if(MYARG == Reader1Arg)
			{
				BallToUpdate = 'E';
				BallIRData = 0x12;
				BallTUFound = 0;
				UpdateBallData = 1;
			}
			if(MYARG == Reader2Arg)
			{
				BallToUpdate = 'F';
				BallIRData = 0x34;
				BallTUFound = 1;
				UpdateBallData = 1;
			}*/

			GoFlag = 1;
			Ball[1].Status = There;
			Ball[3].Status = There;
			Ball[4].Status = There;
			Ball[6].Status = There;
			
		}

	}
}
