/***************************************************************
* This is the Lotus Library that contains all of the functions
* from PWM, SoundGin, ezVGA, Controller, and EEPROM
****************************************************************/
#include "hcs12.h"
#include "LotusLib.h"

unsigned char ContP1;
unsigned char ContP2;

unsigned int RandNum;

// The pulse width modulator function controls 8 LEDs on the Lotus Board.
// The available 8 LEDs are referrenced from 0-7 with 0 being the left most
// LED and 7 being the right most.  The intensity of the LED can be set in
// a range between 0 and 255 with 255 being the brightest.
void SetPWMDuty(char PWMChannel, unsigned char Value)
{
    switch(PWMChannel)
    {
        case 0:    PWMDTY0 = Value;
                   break;

        case 1:    PWMDTY1 = Value;
                   break;

        case 2:    PWMDTY2 = Value;
                   break;

        case 3:    PWMDTY3 = Value;
                   break;

        case 4:    PWMDTY4 = Value;
                   break;

        case 5:    PWMDTY5 = Value;
                   break;

        case 6:    PWMDTY6 = Value;
                   break;

        case 7:    PWMDTY7 = Value;
                   break;
    }
}

// This Controller function reads out the shift registers and store in two
// public values called ContP1 and ContP2.  ContP1 is the left controller
// and ContP2 is the right one.
void GetContData()
{
    char index;

    ContP1 = 0;
    ContP2 = 0;

    // Be sure that we start out with clk and latch set to zero.
    PORTB &= 0xfc;

    // Shift in data from controllers.
    // First we latch it.
    PORTB |= 2;

    // Wait 400ns
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");

    // Release the latch
    PORTB &= 0xfd;

    // Wait 400ns
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");
    __asm__ __volatile__  (" nop ");__asm__ __volatile__  (" nop ");

    // Get the data.
    if(PORTB & 0x04)
        ContP1 |= 0x01;

    if(PORTB & 0x08)
        ContP2 |= 0x01;

    for(index = 0;index < 7;index++)
    {
        // Bring the clock high.
        PORTB |= 0x01;

        ContP1 = ContP1 << 1;
        ContP2 = ContP2 << 1;

        // Bring the clock low.
        PORTB &= 0xfe;

        if(PORTB & 0x04)
            ContP1 |= 0x01;

        if(PORTB & 0x08)
            ContP2 |= 0x01;
    }

}

// This is a library function that will seed the random number generator
// with the timer/counter value.
void Seed()
{
    RandNum = TCNT;
}

// This is a library function which will return a pseudo random number.
// Note that the user should first call the Seed() function one time before
// calling Rand(). After that the user can call Rand() over and over again to
// get a random number. Note that the user can also access the random number
// at any time as it is a global variable.
unsigned int Rand()
{
    unsigned int temp;
    temp = ( RandNum << 1 ) ^ ((( RandNum & 0x8000 ) >> 15) |	/* 16 tap */
                               (( RandNum & 0x8000 ))       | 	/* 15 tap */
                               (( RandNum & 0x8000 ) >> 2)  | 	/* 13 tap */
                               (( RandNum & 0x8000 ) >> 11));	/* 4 tap */

    RandNum = temp;
    return temp;
}

// Note that when writing EEPROM we will block until the operation is complete.
// Thus, it is not advisable to do EEPROM write during gameplay as it will
// violate the RTI.
void WriteEEPROM(unsigned int add, unsigned char data)
{
    unsigned char MSB = add >> 8;
    unsigned char LSB = add;
    unsigned char dummy;

    // First we need to enable the write.
    PORTB &= 0xef;   // Clear CS.
    SPI0DR = 0x06;   // Send Command to enable writes.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    PORTB |= 0x10;   // Set CS.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    // Now write the data.
    PORTB &= 0xef;   // Clear CS.
    SPI0DR = 0x02;   // Send Command to write.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    SPI0DR = MSB;    // Write the MSB of the address.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    SPI0DR = LSB;    // Write the LSB of the address.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    SPI0DR = data;    // Write the data.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.
    PORTB |= 0x10;   // Set CS.

    // Now wait for the data to be written.
    dummy = 0xff;
    while(dummy & 0x01)
    {
        PORTB &= 0xef;     // Clear CS.
        SPI0DR = 0x05;     // Send the command to read the status register.
        while((SPI0SR & SPIF) == 0); // Wait for us to write command.
        dummy = SPI0DR;    // Dummy read to clear the interrupt.

        SPI0DR = 0xff;     // Dummy write value to clock data in.
        while((SPI0SR & SPIF) == 0); // Wait for dummy write to complete.
        dummy = SPI0DR;    // Dummy value we care about.
        PORTB |= 0x10;     // Set CS
    }

}

// Note that when reading EEPROM we will block until the operation is complete.
// Thus, it is not advisable to do EEPROM read during gameplay as it will
// violate the RTI.
unsigned char ReadEEPROM(unsigned int add)
{
    unsigned char dummy;
    unsigned char MSB = add >> 8;
    unsigned char LSB = add;
    unsigned char retval;

    // Now write the data.
    PORTB &= 0xef;   // Clear CS.
    SPI0DR = 0x03;   // Send Command to read.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    SPI0DR = MSB;    // Write the MSB of the address.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    SPI0DR = LSB;    // Write the LSB of the address.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    dummy = SPI0DR;  // Need to access SPI0DR to clear the interrupt.

    SPI0DR = 0xff;   // Dummy write to get the data.
    while((SPI0SR & SPIF) == 0);  // Wait for us to write data.
    retval = SPI0DR; // Need to access read the data we wrote.
    PORTB |= 0x10;   // Set CS.

    return retval;

}
