/***************************************************************
* This is the Lotus Library that contains all of the functions
* for SoundGin
****************************************************************/
#include "hcs12.h"
#include "soundgin.h"

// Parameter
unsigned int SoundBufferSize = 0;
char SoundBuffer[6];

//
// Assemble the command and send it out to the serial port
char IssueCommand(unsigned char Para, unsigned char Cmd, unsigned char Arg1,
                 unsigned char Arg2, unsigned char Arg3, unsigned char Arg4) {
                 
    char index = 0;

    // Set the global setting for the serial send interrupt
    SoundBuffer[0] = S_CMD;
    SoundBuffer[1] = Cmd;
    SoundBuffer[2] = Arg1;
    SoundBuffer[3] = Arg2;
    SoundBuffer[4] = Arg3;
    SoundBuffer[5] = Arg4;
    
    SoundBufferSize = Para + 2;     // accound for cmd and the issue cmd
    
    for(index=0;index<SoundBufferSize;index++)
    {
        while((SCI0SR1 & TDRE) == 0){};
        SCI0DRL = SoundBuffer[index];
    }

    return 0;
}

//
// Reads the memory at the location specified by Reg.
// i.e.: 27,0,24 - Causes Oscillator A1's Amplitude to be sent out TX.
char ReadOneByte(unsigned char Reg) {
    return(IssueCommand(1, S_R1B, Reg, 0, 0, 0));
}

//
// Writes Byte at memory at the location specified by Reg.
// i.e.: 27,1,24,127 - Sets Oscillator A1's Amplitude to 127
char WriteOneByte(unsigned char Reg, unsigned char Byte) {
    return(IssueCommand(2, S_W1B, Reg, Byte, 0, 0));
}

//
// Writes two Bytes at memory at the location specified by Reg.
// i.e.: 27,1,20,163,1 - Sets Oscillator A1's Frequency Target to 100Hz
char WriteTwoByte(unsigned char Reg, unsigned char Byte1, unsigned Byte2) {
    return(IssueCommand(3, S_W2B, Reg, Byte1, Byte2, 0));
}

//
// Writes three Bytes at memory at the location specified by Reg.
// i.e.: 27,1,17,110,163,1 - Sets Oscillator A1's Frequency to 100.00Hz
char WriteThreeByte(unsigned char Reg, unsigned char Byte1,
                   unsigned char Byte2, unsigned char Byte3) {
    return(IssueCommand(4, S_W3B, Reg, Byte1, Byte2, Byte3));
}

//
// Writes Byte masked with Mask at memory at the location specified by Reg.
// i.e.:27,4,0,2,255 - Causes Oscillator A2 to be included in Mixer A's output.
char WriteOneByteWMask(unsigned char Reg, unsigned Byte, unsigned char Mask) {
    return(IssueCommand(3, S_W1BM, Reg, Byte, Mask, 0));
}

//
// Clears memory locations 0-127.
char ClearMixerAB() {
    return(IssueCommand(0, S_CMAB, 0, 0, 0, 0));
}

//
// Causes the oscillators to calculate a ramp value and then turn on
// the ramp and target options.
char RampToTargets(char A, char B) {
    if(A & B) {                // ramp both mixers
        return(IssueCommand(0, S_RTAB, 0, 0, 0, 0));
    } else if(A) {             // ramp only mixer A
        return(IssueCommand(0, S_RTA, 0, 0, 0, 0));
    } else {                   // ramp only mixer B
        return(IssueCommand(0, S_RTB, 0, 0, 0, 0));
    }
}

//
// Sets the Voice frequency to the specified musical note frequency.
char SetVoiceNote(unsigned char Note) {
    return(IssueCommand(1, S_SVN, Note, 0, 0, 0));
}

//
// Sets the Voice frequency to the specified musical note frequency.
char SetVoiceFreq(unsigned int Freq) {
    // break apart the freq to be stored in 2 chars
    unsigned char lower = (unsigned char) Freq;
    unsigned char upper = (unsigned char) (Freq >> 8);

    return(IssueCommand(2, S_SVF, upper, lower, 0, 0));
}

//
// Sets the Voice delay to the specified 8-Bit amount.
char SetVoiceDelay(unsigned char Delay) {
    return(IssueCommand(1, S_SVD, Delay, 0, 0, 0));
}

//
// Sets the Voice options to the default values.
char SetVoiceDefault() {
    return(IssueCommand(0, S_SVDEF, 0, 0, 0, 0));
}

//
// Turns the Q line on.
char SetQOn() {
    return(IssueCommand(0, S_QON, 0, 0, 0, 0));
}

//
// Turns the Q line off.
char SetQOff() {
    return(IssueCommand(0, S_QOFF, 0, 0, 0, 0));
}

//
// Clears memory locations for mixer and its associated oscillators.
char ClearMixerOscillators(char A) {
    if(A) {                    // clear mixer A
        return(IssueCommand(0, S_CMOA, 0, 0, 0, 0));
    } else {                   // clear mixer B
        return(IssueCommand(0, S_CMOB, 0, 0, 0, 0));
    }
}

//
// Sets the oscillator's frequency to the specified musical note frequency.
char LoadlNote(char Osc, unsigned char Note) {
    switch(Osc) {
        case EO_A1: return(IssueCommand(1, S_LNOA1, Note, 0, 0, 0));
        case EO_A2: return(IssueCommand(1, S_LNOA2, Note, 0, 0, 0));
        case EO_A3: return(IssueCommand(1, S_LNOA3, Note, 0, 0, 0));
        case EO_B1: return(IssueCommand(1, S_LNOB1, Note, 0, 0, 0));
        case EO_B2: return(IssueCommand(1, S_LNOB2, Note, 0, 0, 0));
        case EO_B3: return(IssueCommand(1, S_LNOB3, Note, 0, 0, 0));
    }
}

//
// Sets the mixer and its associated oscillators to a predefined sound.
// 27,75,3 - Set the Oscillators in Mixer A to sound 3.
char LoadSound(char A, unsigned char Sound) {
    if(A) {                    // load sound into mixer A
        return(IssueCommand(1, S_LSMA, Sound, 0, 0, 0));
    } else {                   // load sound into mixer B
        return(IssueCommand(1, S_LSMB, Sound, 0, 0, 0));
    }
}

//
// Sets the oscillator's frequency to the specified 16-bit frequency.
char LoadFreq(char Osc, unsigned int Freq) {
    // break apart the freq to be stored in 2 chars
    unsigned char lower = (unsigned char) Freq;
    unsigned char upper = (unsigned char) (Freq >> 8);

    switch(Osc) {
        case EO_A1: return(IssueCommand(2, S_LFOA1, upper, lower, 0, 0));
        case EO_A2: return(IssueCommand(2, S_LFOA2, upper, lower, 0, 0));
        case EO_A3: return(IssueCommand(2, S_LFOA3, upper, lower, 0, 0));
        case EO_B1: return(IssueCommand(2, S_LFOB1, upper, lower, 0, 0));
        case EO_B2: return(IssueCommand(2, S_LFOB2, upper, lower, 0, 0));
        case EO_B3: return(IssueCommand(2, S_LFOB3, upper, lower, 0, 0));
    }
}

//
// Play the oscillator in its current settings.
char PlayCurrentOscillator(char Osc) {
    switch(Osc) {
        case EO_A1: return(IssueCommand(0, S_PCOA1, 0, 0, 0, 0));
        case EO_A2: return(IssueCommand(0, S_PCOA2, 0, 0, 0, 0));
        case EO_A3: return(IssueCommand(0, S_PCOA3, 0, 0, 0, 0));
        case EO_B1: return(IssueCommand(0, S_PCOB1, 0, 0, 0, 0));
        case EO_B2: return(IssueCommand(0, S_PCOB2, 0, 0, 0, 0));
        case EO_B3: return(IssueCommand(0, S_PCOB3, 0, 0, 0, 0));
    }
}

//
// Play the mixer in its current settings.
char PlayCurrentMixer(char A) {
    if(A) {                    // play mixer A
        return(IssueCommand(0, S_PCMA, 0, 0, 0, 0));
    } else {                   // play mixer B
        return(IssueCommand(0, S_PCMB, 0, 0, 0, 0));
    }
}

//
// Release the oscillator of its current settings.
char ReleaseOscillator(char Osc) {
    switch(Osc) {
        case EO_A1: return(IssueCommand(0, S_ROA1, 0, 0, 0, 0));
        case EO_A2: return(IssueCommand(0, S_ROA2, 0, 0, 0, 0));
        case EO_A3: return(IssueCommand(0, S_ROA3, 0, 0, 0, 0));
        case EO_B1: return(IssueCommand(0, S_ROB1, 0, 0, 0, 0));
        case EO_B2: return(IssueCommand(0, S_ROB2, 0, 0, 0, 0));
        case EO_B3: return(IssueCommand(0, S_ROB3, 0, 0, 0, 0));
    }
}

//
// Release the mixer of its current settings.
char ReleaseMixer(char A) {
    if(A) {                    // release mixer A
        return(IssueCommand(0, S_RMA, 0, 0, 0, 0));
    }
    else {                     // release mixer B
        return(IssueCommand(0, S_RMB, 0, 0, 0, 0));
    }
}

//
// Sets the oscillator's frequency to the specified musical note
// frequency and start the envelope.
// 27,88,64 - Sets Oscillator A1's frequency to C4 (Middle-C) and
// starts the envelope
char LoadPlayNote(char Osc, unsigned char Note) {
    switch(Osc) {
        case EO_A1: return(IssueCommand(1, S_LPNOA1, Note, 0, 0, 0));
        case EO_A2: return(IssueCommand(1, S_LPNOA2, Note, 0, 0, 0));
        case EO_A3: return(IssueCommand(1, S_LPNOA3, Note, 0, 0, 0));
        case EO_B1: return(IssueCommand(1, S_LPNOB1, Note, 0, 0, 0));
        case EO_B2: return(IssueCommand(1, S_LPNOB2, Note, 0, 0, 0));
        case EO_B3: return(IssueCommand(1, S_LPNOB3, Note, 0, 0, 0));
    }
}

//
// Sets the oscillator's frequency to the specified 16-bit frequency
// and starts the envelope.
char LoadPlayFreq(char Osc, unsigned int Freq) {
    // break apart the freq to be stored in 2 chars
    unsigned char lower = (unsigned char) Freq;
    unsigned char upper = (unsigned char) (Freq >> 8); // Got it pete.

    switch(Osc) {
        case EO_A1: return(IssueCommand(2, S_LPFOA1, upper, lower, 0, 0));
        case EO_A2: return(IssueCommand(2, S_LPFOA2, upper, lower, 0, 0));
        case EO_A3: return(IssueCommand(2, S_LPFOA3, upper, lower, 0, 0));
        case EO_B1: return(IssueCommand(2, S_LPFOB1, upper, lower, 0, 0));
        case EO_B2: return(IssueCommand(2, S_LPFOB2, upper, lower, 0, 0));
        case EO_B3: return(IssueCommand(2, S_LPFOB3, upper, lower, 0, 0));
    }
}

//
// Reset to voice parameters.
char ResetVoiceParameter(char A) {
    if(A) {                    // reset voice A
        return(IssueCommand(0, S_RVPA, 0, 0, 0, 0));
    } else {                   // reset voice B
        return(IssueCommand(0, S_RVPB, 0, 0, 0, 0));
    }
}
