/***************************************************************
* This would be the main user Skeleton code for any user game.
* Notice that we give the user access to the Init function
* in case they wish to initialize anything and the RTI ISR
* in case they need something done in the ISR. Thus, don't
* put these functions into other "library" files.
****************************************************************/
#include "hcs12.h"
#include "LotusLib.h"

// Define the number of random number we will get and go through.
#define NUMOFRAND 50

// Define how far we go back if we are wrong.
#define BACKAMOUNT 10

// Define the number of notes in the various tunes.
#define IntroNotes 18
#define WinNotes 24
#define GameNotes 2

// Define the names of the Music States to make it easier for coding.
#define Halt 0
#define MSHalt 1
#define MIState 3
#define MGState 20
#define MWState 50
#define MAState 40

// Define the names of the Music States to make it easier for coding.
#define SState 0
#define SHalt 30
#define VeryEnd 40

// Define the names of the Select States to make it easier for coding.
#define SelStart 1

// Define the names of the Player States to make it easier for coding.
#define P1State 1
#define P2State 2

// Define our fun pixel starting location.
#define FUNX 225
#define FUNY 180

// Usual stuff here for Terminal output.
void TermPutChar(char data){
  while((SCI0SR1 & TDRE) == 0){};
  SCI0DRL = data;
}

void TermPuts(char *pt){
  while(*pt){
    TermPutChar(*pt);
    pt++;
  }
}

// Define any global variables not belonging to the library here.
volatile unsigned char RTIFlag = 0;

// Initalizing all of the registers on the dragon board for the lotus system
void Init()
{
    char index;
    
    // Initialize the counter used for seeding the Random Number generator.
    TSCR1 = 0x80;

    // Initialze the Controller and SPI CS signal.
    DDRB = 0x13;      // Make bits 0,1, and 4 output.
    PORTB = 0x10;     // Make clk low and latch low.
                      // Make SPI CS signal high.

    // Initialize the SPI for the EEPROM.
    SPI0BR = 0x03;    // Set the SPI clk rate to 1.5MHz.
    SPI0CR1 = 0x50;   // Turn on SPI and make us a Master.

    // Initialize the PWM.
    PWMPOL = 0;       // Active low polarity. This means the greater the duty
                      // cycle then brighter the LED.

    PWMPRCLK = 0x55;  // This makes clock to the PWM divided by 32 which is
                      // good because the LED can't switch that fast.

    PWME = 0xFF;       // Enable all PWM channels.
    
    // Turn the LEDs off.
    for(index=0;index<8;index++)
        SetPWMDuty(index, 0x00);

    // Initialize the RTI.
    CRGINT |= RTIE;   // Enable the interrupt for the RTI.
    RTICTL = 0x78;    // Set the RTI frequency to 30.5Hz.

    // Initialize SCIs for SoundGin and VGA.
    // Initialize SCI0
    SCI0BDH = 0;      // br=MCLK/(16*baudRate)
    SCI0BDL = 0x9C;   // 9600 BAUD.
    SCI0CR1 = 0;
    /* bit value meaning
    7   0    LOOPS, no looping, normal
    6   0    WOMS, normal high/low outputs
    5   0    RSRC, not appliable with LOOPS=0
    4   0    M, 1 start, 8 data, 1 stop
    3   0    WAKE, wake by idle (not applicable)
    2   0    ILT, short idle time (not applicable)
    1   0    PE, no parity
    0   0    PT, parity type (not applicable with PE=0) */
    SCI0CR2 = 0x0C;
    /* bit value meaning
    7   0    TIE, no transmit interrupts on TDRE
    6   0    TCIE, no transmit interrupts on TC
    5   0    RIE, no receive interrupts on RDRF
    4   0    ILIE, no interrupts on idle
    3   1    TE, enable transmitter
    2   1    RE, enable receiver
    1   0    RWU, no receiver wakeup
    0   0    SBK, no send break */

    // Initialize SCI1
    SCI1BDH = 0;      // br=MCLK/(16*baudRate)
    //SCI1BDL = 13;     // 115200 BAUD.
    SCI1BDL = 14;     // 115200 BAUD.
    SCI1CR1 = 0;
    /* bit value meaning
    7   0    LOOPS, no looping, normal
    6   0    WOMS, normal high/low outputs
    5   0    RSRC, not appliable with LOOPS=0
    4   0    M, 1 start, 8 data, 1 stop
    3   0    WAKE, wake by idle (not applicable)
    2   0    ILT, short idle time (not applicable)
    1   0    PE, no parity
    0   0    PT, parity type (not applicable with PE=0) */
    SCI1CR2 = 0x0C;
    /* bit value meaning
    7   0    TIE, no transmit interrupts on TDRE
    6   0    TCIE, no transmit interrupts on TC
    5   0    RIE, no receive interrupts on RDRF
    4   0    ILIE, no interrupts on idle
    3   1    TE, enable transmitter
    2   1    RE, enable receiver
    1   0    RWU, no receiver wakeup
    0   0    SBK, no send break */

    // Wait for about 300ms for ezVGA to initialize.
    for(index=0;index<10;index++)
    {
        while(RTIFlag == 0);    // Wait for RTIFlag to be set.
        RTIFlag = 0;            // Clear it.
    }

    // Send the U character to ezVGA so that it know the BAUD rate.
    while((SCI1SR1 & TDRE) == 0);
    SCI1DRL = 'U';
    
    // Wait for ACK/NACK. This guarantees that we wont send
    // ezVGA another byte until he is ready.
    while((SCI1SR1 & RDRF) == 0);
    if(SCI1DRL == 0x15)       // Check to see if it is a NACK.
    {
        SetPWMDuty(1, 0xff);  // Turn on both red LEDs to indicate a problem.
        SetPWMDuty(5, 0xff);
        
        // Wait for about 1s for user to see there is a problem.
        for(index=0;index<30;index++)
        {
            while(RTIFlag == 0);    // Wait for RTIFlag to be set.
            RTIFlag = 0;            // Clear it.
        }
        
        SetPWMDuty(1, 0x00);  // Turn off both red LEDs.
        SetPWMDuty(5, 0x00);
    }
    
    /* Any user initializations here. */
}

// Interrupt Handler routine for RTI.
void __attribute__((interrupt)) RTIHandle()
{
    CRGFLG |= RTIF;   // Clear the interrupt flag.
    RTIFlag = 1;      // Set the global variable flag.

    /* Any user code needed withing the ISR here. */
}

int main()
{
    // -------------------------------------------------------------------
    // Here we have the data needed for the Splash Screen State machine.
    // -------------------------------------------------------------------
    // Splash Screen state machine variable.
    volatile unsigned int SplashState = 0;
    
    // Splash Screen strings that are displayed.
    char KKString[10] = "KILL KEVIN";
    char PSString[18] = "Push Start to Play";
    char PSelString[19] = "Push Select to Load";
    char P1ScoreString[11] = "P1 Score = ";
    char P2ScoreString[11] = "P2 Score = ";
    char ReadyString[5] = "READY";
    
    // Index for displaying strings.
    volatile int SplashIndex=0;
    
    // RTI counter for keeping track of time.
    volatile unsigned int SplashRTI = 0;
    volatile unsigned int VeryEndRTI = 0;

    // -------------------------------------------------------------------
    // Here we have the data needed for the Game Music State machine.
    // -------------------------------------------------------------------
    // Intro music is from 2001. Here are the notes and their duration.
    unsigned char IntroNote[18] = {A4s, F5, A5s, D6, C6s, G5, A5, A5s, C6, D6, D6s, F6, D6, D6s, F6, G6, A6, A6s};
    unsigned int IntroDur[18] =    {60, 60, 120, 15, 75,  15, 15, 60,  30, 15, 15,  90, 8,  8,   60, 40, 60, 80};
    unsigned char WinNote[24] = {D6, C6, D6, C6, A5s, A6s, A6, A6s, A6, G6, A6, F6, A6s, A6, F6, A6s, G6s, F6, A6s, G6s, F6, D6s, C6, F6};
    unsigned int WinDur[24] =   {23, 15, 8, 23, 23,  23,  15, 8,  23, 23, 23, 15, 8,  23, 15, 8,  23,  15, 8,  23,  30, 8,  8, 60};
    
    // Index to index the music notes and duration.
    volatile int MusicIndex=0;
    
    // Counter used to determine if we are done with the music.
    volatile unsigned char NoteCounter = 0;
    
    // RTI counter for keeping track of time.
    volatile unsigned int SoundRTI = 0;
    volatile unsigned int AnnoyingRTI = 0;
    
    // Flags for control.
    volatile char GMFlag = 0;
    volatile char WinFlag = 0;
    volatile char AnnoyingFlag = 0;
    volatile char GNFlag = 0;
    
    // Music state machine variable.
    volatile unsigned int MusicState = 0;

    // Variable for the dynamic duration of the Game Music.
    volatile unsigned char GDur;
    
    // Temp for Random numbers.
    volatile unsigned int SplashTemp;
    
    // -------------------------------------------------------------------
    // Here we have the data needed for the Select State machine.
    // -------------------------------------------------------------------
    // Select state machine variable.
    volatile unsigned int SelectState = 0;
    
    // Strings for the Select SM
    char ResumeString[6] =  "Resume";
    char RestoreString[7] = "Restore";
    char SaveString[4] =    "Save";
    char DoneString[4] =    "Done";
    char RTOString[26] = "Kevin is a Right Turn Only";
    
    // Index for the Select SM strings.
    volatile unsigned char SelIndex;
    
    // Variable that determine what we are selecting.
    volatile unsigned char SelVar;
    
    // Variable for the Select SM RTI counter.
    volatile unsigned char SelRTI;
    
    // -------------------------------------------------------------------
    // Here we have the data needed for the Player State machines.
    // -------------------------------------------------------------------
    // Strings for Player Wins.
    char P1WinString[9] = "P1 WINS!!";
    char P2WinString[9] = "P2 WINS!!";
    
    // Indexes for the strings.
    volatile unsigned char P1Index, P2Index;
    
    // Select state machine variable.
    volatile unsigned int Player1State = 0;
    volatile unsigned int Player2State = 0;
    
    // Temp variable.
    volatile unsigned int P1Temp,P2Temp;
    
    // RTI counter for keeping track of time.
    volatile unsigned int P1RTI = 0;
    volatile unsigned int P2RTI = 0;
    
    // Color Array.
    unsigned char ColorArray[6] = {0x1f, 0x0c, 0x07, 0x22, 0x3d, 0x1b};
    volatile unsigned char P1CIndex, P2CIndex;
    
    // -------------------------------------------------------------------
    // Here we have variables used by multiple state machines.
    // -------------------------------------------------------------------
    // Flag to indicate that we have in fact started.
    volatile unsigned char Started=0;
    
    // Flag to indicate that we don't need get random number as we have loaded.
    volatile unsigned char Loaded=0;
    
    // Random Player indexes that indicate where we are in the Random Array.
    volatile unsigned char P1RandIndex=0, P2RandIndex=0;
    
    // Random Array that stores the random numbers.
    unsigned int RandArray[NUMOFRAND];

    // First call the Init function to intialize all the IO.
    Init();
    
    // Then configure the sound envelope to our pleasing
    WriteOneByteWMask(30, 0xf0, 0xe0);
    WriteOneByteWMask(30, 15, 244);

    // Main loop.
    while(1)
    {
        // Wait for the RTI flag to be set.
        if(RTIFlag)
        {
            RTIFlag = 0;    // First thing we do is clear the flag.

            // We will always get the controller data so do that first.
            GetContData();
            
            // This is the Splash screen state machine. It will the be the
            // first state machine that will run.
            switch(SplashState)
            {
                case SHalt:     SplashState = SHalt;
                                break;

                // At the very end we wait for the user to push a button. When
                // they do we start the whole game over.
                // First wait for the user to stop pushing a button.
                case VeryEnd:   Loaded=0;Started=0;
                                if(ContP1 == 0xff && ContP2 == 0xff)
                                    SplashState = VeryEnd+1;
                                    
                                VeryEndRTI = 0;
                                break;
                                
                // Now wait two seconds.
                case VeryEnd+1: if(VeryEndRTI == 60)
                                    SplashState = VeryEnd+2;
                                    
                                VeryEndRTI++;
                                
                                break;

                // Then wait until we push something.
                case VeryEnd+2: if(ContP1 != 0xff || ContP2 != 0xff)
                                {
                                    MusicState = MSHalt;      // Stop the Music SM.
                                    SplashState = SState;     // Start all over.
                                    SplashState = VeryEnd+3;
                                }
                                break;

                // Then wait until we are done pushing everything.
                case VeryEnd+3: if(ContP1 == 0xff && ContP2 == 0xff)
                                    SplashState = SState;
                                break;
            
                // First we display the strings on the screen.
                case SState:    ClearScreen(0x00);    // Clear the screen.
                
                                // Draw the Splash Screen strings.
                                for(SplashIndex=0;SplashIndex<10;SplashIndex++)
                                    PlaceCharacter(KKString[SplashIndex], 0, 3, 0x03, 50+(20*SplashIndex), 80);
                                for(SplashIndex=0;SplashIndex<18;SplashIndex++)
                                    PlaceCharacter(PSString[SplashIndex], 0, 1, 0x3f, 60+(10*SplashIndex), 120);
                                for(SplashIndex=0;SplashIndex<19;SplashIndex++)
                                    PlaceCharacter(PSelString[SplashIndex], 0, 1, 0x3f, 56+(10*SplashIndex), 140);

                                if(Loaded == 0)     // Check to see if we have loaded a game.
                                {                   // If we have not then
                                    P1RandIndex=0;  // clear the Rand number indexes for both players.
                                    P2RandIndex=0;
                                }

                                MusicState = MIState;         // Start the Music Intro.
                                SplashState = SState+1;       // Goto next state.
                                break;

                // Wait for user input.
                case SState+1:  if((ContP1 & START) == 0 || (ContP2 & START) == 0)
                                    SplashState = SState + 3;
                                if((ContP1 & SEL) == 0 || (ContP2 & SEL) == 0)
                                    SplashState = SState + 2;
                                break;
                                
                // If the user pushes Select then they want to load a game.
                case SState+2:  MusicState = MSHalt;     // Halt the music.
                                SplashState = SHalt;     // Halt us.
                                SelectState = SelStart;  // Start the Select State Machine.
                                break;
                                
                // If the user pushes start then work towards playing the game.
                case SState+3:  MusicState = MSHalt;     // Halt the music.

                                Started = 1;             // Indicate that we have started.
                                SplashState = SState+4;  // Goto the next state.
                                
                                // If we have loaded up a previous game then
                                // we don't need to get any more random numbers.
                                if(Loaded)
                                    break;
                
                                Seed();                  // Seed the Random number generator.
                                
                                // Call the Rand() function 16 times to Roll
                                // all the way through the shift register one
                                // time.
                                for(SplashIndex=0;SplashIndex<16;SplashIndex++)
                                    Rand();
                                
                                // Fill the RandArray with random numbers. Be sure that
                                // the numbers are between 0 and 5.
                                for(SplashIndex=0;SplashIndex<NUMOFRAND;SplashIndex++)
                                {
                                    do
                                    {
                                        SplashTemp = Rand() & 0x07;
                                    }
                                    while(SplashTemp > 5);
                                    
                                    RandArray[SplashIndex] = SplashTemp;
                                }
                                
                                break;
                                
                case SState+4:  ClearScreen(0x00);    // Clear the screen.
                
                                // Draw the outline.
                                DrawLine(0x2A, 0, 0, 319, 0);
                                DrawLine(0x2A, 0, 0, 0, 239);
                                DrawLine(0x2A, 0, 239, 319, 239);
                                DrawLine(0x2A, 319, 0, 319, 239);
                                DrawLine(0x2A, 160, 0, 160, 239);
                                
                                // Draw Kill Kevin.
                                for(SplashIndex=0;SplashIndex<10;SplashIndex++)
                                    PlaceCharacter(KKString[SplashIndex], 0, 1, 0x03, 112+(10*SplashIndex), 5);
                                
                                // Draw P1 and P2 Score.
                                for(SplashIndex=0;SplashIndex<11;SplashIndex++)
                                    PlaceCharacter(P1ScoreString[SplashIndex], 0, 1, 0x3f, 10+(10*SplashIndex), 25);
                                for(SplashIndex=0;SplashIndex<11;SplashIndex++)
                                    PlaceCharacter(P2ScoreString[SplashIndex], 0, 1, 0x3f, 170+(10*SplashIndex), 25);
                                
                                // Draw the Crosses.
                                ClearArea(0x3f, 78, 70, 82, 170);
                                ClearArea(0x3f, 30, 118, 130, 122);
                                ClearArea(0x3f, 238, 70, 242, 170);
                                ClearArea(0x3f, 190, 118, 290, 122);
                                
                                // Draw Ready.
                                for(SplashIndex=0;SplashIndex<5;SplashIndex++)
                                    PlaceCharacter(ReadyString[SplashIndex], 0, 3, 0x2E, 109+(20*SplashIndex), 140);
                                    
                                // Place the countdown. Start with 3.
                                PlaceCharacter('3', 0, 3, 0x27, 148, 190);
                                
                                // Clear the RTI counter.
                                SplashRTI = 0;

                                SplashState = SState+5;   // Goto the next state.
                                break;

                // Wait 3 seconds before actually beginning. Countdown the time
                // in seconds. When ready clear Ready and number and halt.
                case SState+5:  if(SplashRTI == 30)
                                    PlaceCharacter('2', 0, 3, 0x1B, 148, 190);
                                if(SplashRTI == 60)
                                    PlaceCharacter('1', 0, 3, 0x1F, 148, 190);
                                if(SplashRTI == 90)
                                {
                                    ClearArea(0x00, 109, 140, 220, 220); // Clear Ready and number.
                                    DrawLine(0x2A, 160, 0, 160, 239);    // Redraw our middle line.
                                    Player1State = P1State;   // Start P1 SM.
                                    Player2State = P2State;   // Start P2 SM.
                                    MusicState = MGState;     // Start the Game Music.
                                    SplashState = SHalt;      // Halt ourselves.
                                    
                                    P1CIndex = RandArray[0] & 0x3; // Determine starting Color index.
                                    P2CIndex = P1CIndex;
                                }
                                
                                // Increment the RTI counter no matter what.
                                SplashRTI++;
                                
                                break;
                
            }
            
            // This is the Player1 state machine.
            switch(Player1State)
            {
                // Begin in the halt state. Stay halted until the Splash SM
                // takes us out.
                case Halt:        Player1State = Halt;
                                  break;

                // First thing we'll do is update the PWM. The farther we get
                // the more the LEDs will light up.
                case P1State:     // Right most Green LED.
                                  if(P1RandIndex < 3 && P1RandIndex > 0)
                                      SetPWMDuty(3, 0x40);
                                  else if(P1RandIndex < 6 && P1RandIndex > 0)
                                      SetPWMDuty(3, 0x80);
                                  else if(P1RandIndex < 9 && P1RandIndex > 0)
                                      SetPWMDuty(3, 0xc0);
                                  else if(P1RandIndex < 12 && P1RandIndex > 0)
                                      SetPWMDuty(3, 0xff);
                                  else if(P1RandIndex == 0)
                                      SetPWMDuty(3, 0x00);

                                  // Next Green LED.
                                  if(P1RandIndex < 15 && P1RandIndex >= 12)
                                      SetPWMDuty(2, 0x40);
                                  else if(P1RandIndex < 18 && P1RandIndex >= 12)
                                      SetPWMDuty(2, 0x80);
                                  else if(P1RandIndex < 21 && P1RandIndex >= 12)
                                      SetPWMDuty(2, 0xc0);
                                  else if(P1RandIndex < 24 && P1RandIndex >= 12)
                                      SetPWMDuty(2, 0xff);
                                  else if(P1RandIndex < 12)
                                      SetPWMDuty(2, 0x00);
                                    
                                  // Red LED.
                                  if(P1RandIndex < 27 && P1RandIndex >= 24)
                                      SetPWMDuty(1, 0x40);
                                  else if(P1RandIndex < 30 && P1RandIndex >= 24)
                                      SetPWMDuty(1, 0x80);
                                  else if(P1RandIndex < 33 && P1RandIndex >= 24)
                                      SetPWMDuty(1, 0xc0);
                                  else if(P1RandIndex < 36 && P1RandIndex >= 24)
                                      SetPWMDuty(1, 0xff);
                                  else if(P1RandIndex < 24)
                                      SetPWMDuty(1, 0x00);
                                    
                                  // Blue LED.
                                  if(P1RandIndex < 39 && P1RandIndex >= 36)
                                      SetPWMDuty(0, 0x40);
                                  else if(P1RandIndex < 42 && P1RandIndex >= 36)
                                      SetPWMDuty(0, 0x80);
                                  else if(P1RandIndex < 45 && P1RandIndex >= 36)
                                      SetPWMDuty(0, 0xc0);
                                  else if(P1RandIndex < 36)
                                      SetPWMDuty(0, 0x00);
                                  else
                                      SetPWMDuty(0, 0xff);
                                    
                                  Player1State = P1State+1;
                                  break;

                // Now display the tens digit of the score.
                case P1State+1:   if(P1RandIndex < 10)
                                      PlaceCharacter('0', 0, 1, 0x3F, 120, 25);
                                  else if(P1RandIndex < 20)
                                      PlaceCharacter('1', 0, 1, 0x3F, 120, 25);
                                  else if(P1RandIndex < 30)
                                      PlaceCharacter('2', 0, 1, 0x3F, 120, 25);
                                  else if(P1RandIndex < 40)
                                      PlaceCharacter('3', 0, 1, 0x3F, 120, 25);
                                  else if(P1RandIndex < 50)
                                      PlaceCharacter('4', 0, 1, 0x3F, 120, 25);
                                  else
                                      PlaceCharacter('5', 0, 1, 0x3F, 120, 25);

                                  Player1State = P1State+2;
                                  break;

                // Now display the ones digit of the score.
                case P1State+2:   if(P1RandIndex < 10)
                                      P1Temp = P1RandIndex+0x30;
                                  else if(P1RandIndex < 20)
                                      P1Temp = (P1RandIndex-10) + 0x30;
                                  else if(P1RandIndex < 30)
                                      P1Temp = (P1RandIndex-20) + 0x30;
                                  else if(P1RandIndex < 40)
                                      P1Temp = (P1RandIndex-30) + 0x30;
                                  else if(P1RandIndex < 50)
                                      P1Temp = (P1RandIndex-40) + 0x30;
                                  else
                                      P1Temp = 0x30;

                                  PlaceCharacter(P1Temp, 0, 1, 0x3F, 130, 25);
                                  Player1State = P1State+3;
                                  break;

                // Clear upper and lower boxes.
                case P1State+3:   ClearArea(0x00, 78, 64, 82, 68);
                                  ClearArea(0x00, 78, 172, 82, 176);
                                  
                                  Player1State = P1State+4;
                                  break;
                                  
                // Clear left and right boxes.
                case P1State+4:   ClearArea(0x00, 24, 118, 28, 122);
                                  ClearArea(0x00, 132, 118, 136, 122);
                                  
                                  P1RTI = 0;
                                  Player1State = P1State+5;
                                  break;

                // Wait for player to not push any buttons.
                case P1State+5:   if(ContP1 == 0xff)
                                      Player1State = P1State+6;
                                      
                                  break;
                                  
                // Determine color and location of box(es).
                case P1State+6:   P1CIndex++;        // Increment color index to next color.
                                  if(P1CIndex == 6)  // If we raech six then roll over to zero.
                                      P1CIndex = 0;

                                  // Based on the random number determine what
                                  // boxes we should draw in.
                                  if(RandArray[P1RandIndex] == 0)
                                      ClearArea(ColorArray[P1CIndex], 78, 64, 82, 68);
                                  else if(RandArray[P1RandIndex] == 1)
                                      ClearArea(ColorArray[P1CIndex], 78, 172, 82, 176);
                                  else if(RandArray[P1RandIndex] == 2)
                                  {
                                      ClearArea(ColorArray[P1CIndex], 78, 64, 82, 68);
                                      ClearArea(ColorArray[P1CIndex], 78, 172, 82, 176);
                                  }
                                  else if(RandArray[P1RandIndex] == 3)
                                      ClearArea(ColorArray[P1CIndex], 24, 118, 28, 122);
                                  else if(RandArray[P1RandIndex] == 4)
                                      ClearArea(ColorArray[P1CIndex], 132, 118, 136, 122);
                                  else if(RandArray[P1RandIndex] == 5)
                                  {
                                      ClearArea(ColorArray[P1CIndex], 24, 118, 28, 122);
                                      ClearArea(ColorArray[P1CIndex], 132, 118, 136, 122);
                                  }

                                  
                                  Player1State = P1State+7;
                                  break;

                // Wait for user input.
                case P1State+7:   if(ContP1 == 0xff)  // If user have input nothing.
                                  {
                                      Player1State = P1State+7;
                                      break;
                                  }
                                  else if((ContP1 & START) == 0 || (ContP1 & SEL) == 0) // Check for start or select.
                                  {
                                      Player2State = Halt;     // If we get a start or sel then
                                      Player1State = Halt;     // Halt both player SMs, the Music SM
                                      MusicState = MSHalt;     // and start the Select SM.
                                      SelectState = SelStart;
                                      break;
                                  }
                                  
                                  // Else check to see if what the user input was correct.
                                  else if((ContP1 & UP) == 0 && RandArray[P1RandIndex] == 0)
                                      P1RandIndex++;
                                  else if((ContP1 & DOWN) == 0 && RandArray[P1RandIndex] == 1)
                                      P1RandIndex++;
                                  else if((ContP1 & B) == 0 && RandArray[P1RandIndex] == 2)
                                      P1RandIndex++;
                                  else if((ContP1 & LEFT) == 0 && RandArray[P1RandIndex] == 3)
                                      P1RandIndex++;
                                  else if((ContP1 & RIGHT) == 0 && RandArray[P1RandIndex] == 4)
                                      P1RandIndex++;
                                  else if((ContP1 & A) == 0 && RandArray[P1RandIndex] == 5)
                                      P1RandIndex++;
                                      
                                  // Else the input was incorrect so pay the penalty.
                                  else
                                  {
                                      // Must go back a certain amount.
                                      if(P1RandIndex < BACKAMOUNT)
                                          P1RandIndex=0;
                                      else
                                          P1RandIndex -= BACKAMOUNT;

                                      // Play an annoying sound to indicate mistake.
                                      AnnoyingFlag = 1;
                                  }
                                  
                                  // See if we are done.
                                  if(P1RandIndex == NUMOFRAND)
                                  {
                                      // Display the 50
                                      PlaceCharacter('5', 0, 1, 0x3F, 120, 25);
                                      PlaceCharacter('0', 0, 1, 0x3F, 130, 25);
                                      
                                      // Display P1 WINS!!
                                      for(P1Index=0;P1Index<9;P1Index++)
                                          PlaceCharacter(P1WinString[P1Index], 0, 1, 0x2E, 121+(10*P1Index), 140);
                                      
                                      // If we are then set the win flag and Halt.
                                      WinFlag = 1;
                                      Player2State = Halt;
                                      Player1State = Halt;
                                      SplashState = VeryEnd;
                                  }
                                  else
                                      Player1State = P1State;

                                  break;
            }
            
            // This is the Player2 state machine.
            switch(Player2State)
            {
                // Begin in the halt state. Stay halted until the Splash SM
                // takes us out.
                case Halt:        Player2State = Halt;
                                  break;

                // First thing we'll do is update the PWM. The farther we get
                // the more the LEDs will light up.
                case P2State:     // Right most Green LED.
                                  if(P2RandIndex < 3 && P2RandIndex > 0)
                                      SetPWMDuty(7, 0x40);
                                  else if(P2RandIndex < 6 && P2RandIndex > 0)
                                      SetPWMDuty(7, 0x80);
                                  else if(P2RandIndex < 9 && P2RandIndex > 0)
                                      SetPWMDuty(7, 0xc0);
                                  else if(P2RandIndex < 12 && P2RandIndex > 0)
                                      SetPWMDuty(7, 0xff);
                                  else if(P2RandIndex == 0)
                                      SetPWMDuty(7, 0x00);

                                  // Next Green LED.
                                  if(P2RandIndex < 15 && P2RandIndex >= 12)
                                      SetPWMDuty(6, 0x40);
                                  else if(P2RandIndex < 18 && P2RandIndex >= 12)
                                      SetPWMDuty(6, 0x80);
                                  else if(P2RandIndex < 21 && P2RandIndex >= 12)
                                      SetPWMDuty(6, 0xc0);
                                  else if(P2RandIndex < 24 && P2RandIndex >= 12)
                                      SetPWMDuty(6, 0xff);
                                  else if(P2RandIndex < 12)
                                      SetPWMDuty(6, 0x00);

                                  // Red LED.
                                  if(P2RandIndex < 27 && P2RandIndex >= 24)
                                      SetPWMDuty(5, 0x40);
                                  else if(P2RandIndex < 30 && P2RandIndex >= 24)
                                      SetPWMDuty(5, 0x80);
                                  else if(P2RandIndex < 33 && P2RandIndex >= 24)
                                      SetPWMDuty(5, 0xc0);
                                  else if(P2RandIndex < 36 && P2RandIndex >= 24)
                                      SetPWMDuty(5, 0xff);
                                  else if(P2RandIndex < 24)
                                      SetPWMDuty(5, 0x00);

                                  // Blue LED.
                                  if(P2RandIndex < 39 && P2RandIndex >= 36)
                                      SetPWMDuty(4, 0x40);
                                  else if(P2RandIndex < 42 && P2RandIndex >= 36)
                                      SetPWMDuty(4, 0x80);
                                  else if(P2RandIndex < 45 && P2RandIndex >= 36)
                                      SetPWMDuty(4, 0xc0);
                                  else if(P2RandIndex < 36)
                                      SetPWMDuty(4, 0x00);
                                  else
                                      SetPWMDuty(4, 0xff);

                                  Player2State = P2State+1;
                                  break;

                // Now display the tens digit of the score.
                case P2State+1:   if(P2RandIndex < 10)
                                      PlaceCharacter('0', 0, 1, 0x3F, 280, 25);
                                  else if(P2RandIndex < 20)
                                      PlaceCharacter('1', 0, 1, 0x3F, 280, 25);
                                  else if(P2RandIndex < 30)
                                      PlaceCharacter('2', 0, 1, 0x3F, 280, 25);
                                  else if(P2RandIndex < 40)
                                      PlaceCharacter('3', 0, 1, 0x3F, 280, 25);
                                  else if(P2RandIndex < 50)
                                      PlaceCharacter('4', 0, 1, 0x3F, 280, 25);
                                  else
                                      PlaceCharacter('5', 0, 1, 0x3F, 280, 25);

                                  Player2State = P2State+2;
                                  break;

                // Now display the ones digit of the score.
                case P2State+2:   if(P2RandIndex < 10)
                                      P2Temp = P2RandIndex+0x30;
                                  else if(P2RandIndex < 20)
                                      P2Temp = (P2RandIndex-10) + 0x30;
                                  else if(P2RandIndex < 30)
                                      P2Temp = (P2RandIndex-20) + 0x30;
                                  else if(P2RandIndex < 40)
                                      P2Temp = (P2RandIndex-30) + 0x30;
                                  else if(P2RandIndex < 50)
                                      P2Temp = (P2RandIndex-40) + 0x30;
                                  else
                                      P2Temp = 0x30;

                                  PlaceCharacter(P2Temp, 0, 1, 0x3F, 290, 25);
                                  Player2State = P2State+3;
                                  break;

                // Clear upper and lower boxes.
                case P2State+3:   ClearArea(0x00, 238, 64, 242, 68);
                                  ClearArea(0x00, 238, 172, 242, 176);

                                  Player2State = P2State+4;
                                  break;

                // Clear left and right boxes.
                case P2State+4:   ClearArea(0x00, 184, 118, 188, 122);
                                  ClearArea(0x00, 292, 118, 296, 122);

                                  P2RTI = 0;
                                  Player2State = P2State+5;
                                  break;

                // Wait for player to not push any buttons.
                case P2State+5:   if(ContP2 == 0xff)
                                      Player2State = P2State+6;

                                  break;

                // Determine color and location of box(es).
                case P2State+6:   P2CIndex++;        // Increment color index to next color.
                                  if(P2CIndex == 6)  // If we raech six then roll over to zero.
                                      P2CIndex = 0;

                                  // Based on the random number determine what
                                  // boxes we should draw in.
                                  if(RandArray[P2RandIndex] == 0)
                                      ClearArea(ColorArray[P2CIndex], 238, 64, 242, 68);
                                  else if(RandArray[P2RandIndex] == 1)
                                      ClearArea(ColorArray[P2CIndex], 238, 172, 242, 176);
                                  else if(RandArray[P2RandIndex] == 2)
                                  {
                                      ClearArea(ColorArray[P2CIndex], 238, 64, 242, 68);
                                      ClearArea(ColorArray[P2CIndex], 238, 172, 242, 176);
                                  }
                                  else if(RandArray[P2RandIndex] == 3)
                                      ClearArea(ColorArray[P2CIndex], 184, 118, 188, 122);
                                  else if(RandArray[P2RandIndex] == 4)
                                      ClearArea(ColorArray[P2CIndex], 292, 118, 296, 122);
                                  else if(RandArray[P2RandIndex] == 5)
                                  {
                                      ClearArea(ColorArray[P2CIndex], 184, 118, 188, 122);
                                      ClearArea(ColorArray[P2CIndex], 292, 118, 296, 122);
                                  }


                                  Player2State = P2State+7;
                                  break;

                // Wait for user input.
                case P2State+7:   if(ContP2 == 0xff)  // If user have input nothing.
                                  {
                                      Player2State = P2State+7;
                                      break;
                                  }
                                  else if((ContP2 & START) == 0 || (ContP2 & SEL) == 0) // Check for start or select.
                                  {
                                      Player2State = Halt;     // If we get a start or sel then
                                      Player1State = Halt;     // Halt both player SMs, the Music SM
                                      MusicState = MSHalt;     // and start the Select SM.
                                      SelectState = SelStart;
                                      break;
                                  }

                                  // Else check to see if what the user input was correct.
                                  else if((ContP2 & UP) == 0 && RandArray[P2RandIndex] == 0)
                                      P2RandIndex++;
                                  else if((ContP2 & DOWN) == 0 && RandArray[P2RandIndex] == 1)
                                      P2RandIndex++;
                                  else if((ContP2 & B) == 0 && RandArray[P2RandIndex] == 2)
                                      P2RandIndex++;
                                  else if((ContP2 & LEFT) == 0 && RandArray[P2RandIndex] == 3)
                                      P2RandIndex++;
                                  else if((ContP2 & RIGHT) == 0 && RandArray[P2RandIndex] == 4)
                                      P2RandIndex++;
                                  else if((ContP2 & A) == 0 && RandArray[P2RandIndex] == 5)
                                      P2RandIndex++;

                                  // Else the input was incorrect so pay the penalty.
                                  else
                                  {
                                      // Must go back a certain amount.
                                      if(P2RandIndex < BACKAMOUNT)
                                          P2RandIndex=0;
                                      else
                                          P2RandIndex -= BACKAMOUNT;

                                      // Play an annoying sound to indicate mistake.
                                      AnnoyingFlag = 1;
                                  }

                                  // See if we are done.
                                  if(P2RandIndex == NUMOFRAND)
                                  {
                                      // Display the 50
                                      PlaceCharacter('5', 0, 1, 0x3F, 280, 25);
                                      PlaceCharacter('0', 0, 1, 0x3F, 290, 25);

                                      // Display P2 WINS!!
                                      for(P2Index=0;P2Index<9;P2Index++)
                                          PlaceCharacter(P2WinString[P2Index], 0, 1, 0x2E, 121+(10*P2Index), 140);
                                  
                                      // If we are then set the win flag and Halt.
                                      WinFlag = 1;
                                      Player2State = Halt;
                                      Player1State = Halt;
                                      SplashState = VeryEnd;
                                  }
                                  else
                                      Player2State = P2State;

                                  break;
            }
            

            // This is the Music State Machine. It will play the intro tune,
            // the winning tune, the gameplay tune, and the annoying tune.
            switch(MusicState)
            {
                // This is the state the SM starts up. Does nothing.
                case Halt:        MusicState = Halt;
                                  break;

                // This is the Halt state that any other state machine places
                // this SM in. It will release the osc and then wait for the
                // MGFlag (Music Go Flag) to be set.
                case MSHalt:      ReleaseOscillator(EO_A1); // Turns off the sound.
                                  MusicIndex = 0;           // Clear the index.
                                  NoteCounter = 0;          // Clear the note counter.
                                  MusicState = MSHalt+1;    // Goto next state.
                                  break;
                                  
                case MSHalt+1:    if(GMFlag)                // If the Flag gets set
                                      MusicState = MGState; // then play the Go Music.
                                  break;

                // This is the part where we play the intro music. It is the
                // 2001 Intro tune.
                case MIState:     if(GMFlag) // Check to see if we should play
                                  {          // Go Music.
                                      ReleaseOscillator(EO_A1);  // Turn off sound.
                                      MusicState = MGState;      // Play the Go Music.
                                      break;
                                  }
                                  LoadPlayNote(EO_A1,IntroNote[MusicIndex]);  // Play the next note.
                                  SoundRTI = 0;                               // Clear the RTI counter.
                                  MusicState = MIState+1;                     // Goto next state.
                                  break;
                                
                case MIState+1:   if(SoundRTI == IntroDur[MusicIndex]) // Determine how long to play note.
                                  {
                                      // If we are done playing the note then
                                      // determine if we should play another.
                                      NoteCounter++;
                                      if(NoteCounter == IntroNotes)
                                      {
                                          // If we are done playing notes then Halt.
                                          MusicState = MSHalt;  // Goto Halt state.
                                          MusicIndex = 0;       // Clear the music index.
                                      }
                                      else
                                      {
                                          // If we are not done then index then next note.
                                          MusicIndex++;         // Increment index to play next note.
                                          MusicState = MIState; // Goto the state that will play next note.
                                      }
                                  }
                                   
                                  // Else we are not done playing the note so
                                  // increment the RTI counter.
                                  else
                                      SoundRTI++;
                                       
                                  break;
                                   

                // This is the portion of the state machine that will play
                // the winning tune. It will play "We are the Champions."
                case MWState:     WinFlag = 0;                             // Clear the Flag that got us here.
                                  LoadPlayNote(EO_A1,WinNote[MusicIndex]); // Play the next note.
                                  SoundRTI = 0;                            // Clear RTI counter.
                                  MusicState = MWState+1;                  // Goto next state.
                                  break;

                case MWState+1:   if(SoundRTI == WinDur[MusicIndex])  // Determine how long we play note.
                                  {
                                      // If we are done playing note determine if we are done with tune.
                                      NoteCounter++;
                                      if(NoteCounter == WinNotes)  // Determines if we are done with tune.
                                      {
                                          MusicState = MSHalt;     // If we are then Halt.
                                          MusicIndex = 0;          // Reset the index that says what note to play.
                                      }
                                      else   // Else we are not done with tune.
                                      {
                                          MusicIndex++;            // Increment index to play the next tune.
                                          MusicState = MWState;    // Goto the state that plays next tune.
                                      }
                                  }

                                  // Else we are not done playing note.
                                  else
                                      SoundRTI++;  // Increment the RTI counter.

                                  break;
                                   
                // This is the state that will play the annoying sound when you
                // make a mistake.
                case MAState:     WriteOneByteWMask(16, 7, 4);  // Change to a square wave.
                                  MusicState = MAState+1;       // Goto next state.
                                  break;
                                  
                case MAState+1:   LoadPlayNote(EO_A1,C2);       // Play low note.
                                  AnnoyingRTI = 0;              // Clear Annoying RTI counter.
                                  MusicState = MAState+2;       // Goto next state.
                                  break;
                                  
                case MAState+2:   if(AnnoyingRTI == 8)          // Determine if we are done playing note.
                                  {
                                      WriteOneByteWMask(16, 7, 0); // If we are then change to sine wave.
                                      MusicState = MGState;        // Goto Game Music state.
                                      AnnoyingFlag = 0;            // Clear the flag that got us here.
                                  }

                                  // Else we are not done playing music so increment the RTI counter.
                                  else
                                      AnnoyingRTI++;

                                  break;

                // The is the Music Go state where we play the music that is
                // heard during actual game play.
                case MGState:     GMFlag = 0;                 // Clear the flag that got us here.
                                  if(GNFlag)                  // Check to see which note we play.
                                  {
                                      LoadPlayNote(EO_A1,C5); // Play C5 note.
                                      GNFlag = 0;             // Change flag to play other note next time.
                                  }
                                  else
                                  {
                                      LoadPlayNote(EO_A1,C5s);// Play C5 sharp note.
                                      GNFlag = 1;             // Change flag to play other note next time.
                                  }
                                  SoundRTI = 0;               // Clear the RTI counter.
                                  MusicState = MGState+1;     // Goto next state.
                                  break;

                case MGState+1:   if(P1RandIndex > P2RandIndex) // Determine our duration based on the
                                      GDur = P1RandIndex >> 1;  // PRandIndexes. As the player get closer
                                  else                          // to the end the duration gets shorter.
                                      GDur = P2RandIndex >> 1;

                                  GDur = 27 - GDur;
                                  MusicState = MGState+2;
                                  break;
                                  
                case MGState+2:   if(SoundRTI == GDur)          // Determine if we are done playing note.
                                      MusicState = MGState;     // If so then goto state that play the next note.
                                  else                          // Else we are not done play the note
                                      SoundRTI++;               // so just increment the RTI counter.
                                      
                                  if(AnnoyingFlag)              // Check to see if we should play the annoying note.
                                  {
                                      MusicState = MAState;
                                  }
                                  else if(WinFlag)
                                  {
                                      MusicState = MWState;     // Check if we should play the winning tune.
                                  }

                                  break;

            }
            
            // When we hit start or select then we get a menu to save and restore games.
            // This is the state machine which controls that.
            switch(SelectState)
            {
                // Halt state which just sits here and does nothing.
                case Halt:        SelectState = Halt;
                                  break;

                // Start by displaying the Resume, Restore, and Save strings.
                case SelStart:    ClearScreen(0x00);
                                  for(SelIndex=0;SelIndex<6;SelIndex++)
                                      PlaceCharacter(ResumeString[SelIndex], 0, 2, 0x3f, 90+(20*SelIndex), 80);
                                  for(SelIndex=0;SelIndex<7;SelIndex++)
                                      PlaceCharacter(RestoreString[SelIndex], 0, 2, 0x3f, 90+(20*SelIndex), 110);
                                  for(SelIndex=0;SelIndex<4;SelIndex++)
                                      PlaceCharacter(SaveString[SelIndex], 0, 2, 0x3f, 90+(20*SelIndex), 140);

                                  SelVar = 0;                // Clear the var that indicates out selection.
                                  
                                  SelectState = SelStart+1;  // Goto next state.
                                  
                                  /*
                                  // This is just for fun. Best to comment out later.
                                  for(SelIndex=0;SelIndex<26;SelIndex++)
                                      PlaceCharacter(RTOString[SelIndex], 0, 1, 0x37, 30+(10*SelIndex), 20);
                                      
                                  // Draw ECE625 Spike Lab Logo.
                                  WritePixel(0x06, FUNX, FUNY);
                                  DrawLine(0x1b, FUNX, FUNY-4, FUNX-2, FUNY-6);
                                  DrawLine(0x1b, FUNX-2, FUNY-6, FUNX-10, FUNY-6);
                                  DrawLine(0x1b, FUNX-10, FUNY-6, FUNX-14, FUNY-4);
                                  DrawLine(0x1b, FUNX-14, FUNY-4, FUNX-14, FUNY+2);
                                  DrawLine(0x1b, FUNX-14, FUNY+2, FUNX-10, FUNY+6);
                                  DrawLine(0x1b, FUNX-10, FUNY+6, FUNX-14, FUNY+26);
                                  DrawLine(0x1b, FUNX-8, FUNY+26, FUNX, FUNY+6);
                                  DrawLine(0x1b, FUNX, FUNY+6, FUNX+8, FUNY+26);
                                  DrawLine(0x1b, FUNX+14, FUNY+26, FUNX+10, FUNY+6);
                                  DrawLine(0x1b, FUNX+10, FUNY+6, FUNX+14, FUNY+2);
                                  DrawLine(0x1b, FUNX+14, FUNY+2, FUNX+14, FUNY-4);
                                  DrawLine(0x1b, FUNX+14, FUNY-4, FUNX+10, FUNY-6);
                                  DrawLine(0x1b, FUNX+10, FUNY-6, FUNX+2, FUNY-6);
                                  DrawLine(0x1b, FUNX+2, FUNY-6, FUNX, FUNY-4);
                                  
                                  WritePixel(0x31, FUNX-4, FUNY+16);
                                  WritePixel(0x31, FUNX-4, FUNY+17);
                                  DrawLine(0x31, FUNX-3, FUNY+18, FUNX+3, FUNY+18);
                                  DrawLine(0x31, FUNX+3, FUNY+18, FUNX+4, FUNY+16);
                                  DrawLine(0x31, FUNX, FUNY+18, FUNX, FUNY+17);
                                  */
                                  
                                  break;

                // Based on the value of SelVar place the cursor.
                case SelStart+1:  PlaceCharacter(' ', 0 , 1, 0x3f, 80, 83);  // Clear all possible
                                  PlaceCharacter(' ', 0 , 1, 0x3f, 80, 113); // cursor locations.
                                  PlaceCharacter(' ', 0 , 1, 0x3f, 80, 143);
                                  
                                  // Draw in the cursor in the correct place.
                                  if(SelVar == 0)
                                      PlaceCharacter('o', 0 , 1, 0x3f, 80, 83);
                                  else if(SelVar == 1)
                                      PlaceCharacter('o', 0 , 1, 0x3f, 80, 113);
                                  else if(SelVar == 2)
                                      PlaceCharacter('o', 0 , 1, 0x3f, 80, 143);
                                      
                                  SelectState = SelStart+2;
                                  break;

                // Wait for user input.
                case SelStart+2:  if((ContP1 & DOWN) == 0 || (ContP2 & DOWN) == 0)
                                  {
                                      // If the user pushed down then increment
                                      // the SelVar value if we can.
                                      if(SelVar < 2)
                                          SelVar++;
                                          
                                      SelectState = SelStart+3;
                                  }
                                  else if((ContP1 & UP) == 0 || (ContP2 & UP) == 0)
                                  {
                                      // If the user pushed up then decrement
                                      // the SelVar value if we can.
                                      if(SelVar > 0)
                                          SelVar--;

                                      SelectState = SelStart+3;
                                  }
                                  
                                  // If the user pushes start or A then execute
                                  // whatever the cursor is pointing to.
                                  else if((ContP1 & START) == 0 || (ContP2 & START) == 0)
                                  {
                                      SelectState = SelStart+4;
                                  }
                                  else if((ContP1 & A) == 0 || (ContP2 & A) == 0)
                                  {
                                      SelectState = SelStart+4;
                                  }
                                  
                                  break;

                // Next two state just wait for the user to stop pushing a
                // button before moving on.
                case SelStart+3:  if(ContP1 == 0xff && ContP2 == 0xff)
                                  {
                                      SelectState = SelStart+1;
                                  }
                                  break;
                                  
                case SelStart+4:  if(ContP1 == 0xff && ContP2 == 0xff)
                                  {
                                      SelectState = SelStart+5;
                                  }
                                  break;

                // Take action based on the value of SelVar.
                case SelStart+5:  if(SelVar == 0)
                                  {
                                      // If we sould resume then halt us. Depending
                                      // on whether we have already started or not
                                      // depends on where we start the Splash SM at.
                                      SelectState = Halt;
                                      if(Started)
                                          SplashState = SState+4;
                                      else
                                          SplashState = SState;
                                      break;
                                  }
                                  
                                  // Restore a previous game.
                                  else if(SelVar == 1)
                                  {
                                      // Restore the Player Indexes and the RandArray.
                                      P1RandIndex = ReadEEPROM(0x00);
                                      P2RandIndex = ReadEEPROM(0x01);
                                      for(SelIndex=0;SelIndex<NUMOFRAND;SelIndex++)
                                          RandArray[SelIndex] = ReadEEPROM(2+SelIndex);
                                          
                                      Loaded = 1;
                                  }
                                  
                                  // Save a game.
                                  else if(SelVar == 2)
                                  {
                                      // Save the Player Indexes and the RandArray.
                                      WriteEEPROM(0x00, P1RandIndex);
                                      WriteEEPROM(0x01, P2RandIndex);
                                      for(SelIndex=0;SelIndex<NUMOFRAND;SelIndex++)
                                          WriteEEPROM(2+SelIndex, RandArray[SelIndex]);
                                  }
                                  
                                  SelRTI = 0;  // Clear the RTI counter.
                                  SelectState = SelStart+6;
                                  break;
                                  
                // Display a done string to let the user know that the action was completed.
                case SelStart+6:  for(SelIndex=0;SelIndex<4;SelIndex++)
                                      PlaceCharacter(DoneString[SelIndex], 0, 2, 0x3f, 90+(20*SelIndex), 200);
                                      
                                  SelectState = SelStart+7;
                                  break;

                // Display the string for 1 second.
                case SelStart+7:  if(SelRTI == 30)
                                  {
                                      ClearArea(0x00, 90, 200, 200, 220); // Clear the done.
                                      SelectState = SelStart+1;
                                  }
                                  
                                  SelRTI++;
                                  break;

            }
        }
    }

    return 0;
}
