/***************************************************************
* This is the Lotus Library that contains all of the functions
* for ezVGA
****************************************************************/
#include "hcs12.h"
#include "ezVGA.h"

//Parameter
unsigned int ezVGABufferSize = 0;
volatile unsigned char ezVGAReturn;
char ezVGABuffer[10];

void TermPutChar(char data){
  while((SCI0SR1 & TDRE) == 0){};
  SCI0DRL = data;
}

void TermPuts(char *pt){
  while(*pt){
    TermPutChar(*pt);
    pt++;
  }
}

// Issue command
char ezVGAIssueCMD()
{
    int index = 0;

    // Write until all command has been written
    for(index=0;index<ezVGABufferSize;index++)
    {
        while((SCI1SR1 & TDRE) == 0) {};
        SCI1DRL = ezVGABuffer[index];
    }

    // Wait for ACK/NACK. This guarantees that we wont send
    // ezVGA another byte until he is ready.
    while((SCI1SR1 & RDRF) == 0);   // Wait for data.
    ezVGAReturn = SCI1DRL;            // Need to read register for flag to clear.
    if(ezVGAReturn == 21)
        TermPuts("Failed\r\n\0");

    return 0;
}

// This command can be used in a few different ways.  It can be used to clear a
// specific area on the screen or it can be used to draw filled squares and
// rectangles.  Each area is defined by two points.  The upper left-hand corner
// is referred to as point x1,y1 and the lower right-hand point as x2,y2.  When
// this command is executed the ezVGA Serial Module will draw an area from
// point x1,y1 to x2,y2.  If the color is the same as the background color then
// it will appear that the area was erased.  If any other color is used a
// square/rectangle will appear on the screen.
char ClearArea(char color, int x1, char y1, int x2, char y2)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    // Assemble the x values into high and low bits
    char x1HighByte = x1 >> 8;
    char x1LowByte = (char)x1;

    char x2HighByte = x2 >> 8;
    char x2LowByte = (char)x2;

    ezVGABuffer[0] = V_AREACLEAR;
    ezVGABuffer[1] = color;
    ezVGABuffer[2] = x1HighByte;
    ezVGABuffer[3] = x1LowByte;
    ezVGABuffer[4] = y1;
    ezVGABuffer[5] = x2HighByte;
    ezVGABuffer[6] = x2LowByte;
    ezVGABuffer[7] = y2;

    ezVGABufferSize = 8;

    // Issue command
    return(ezVGAIssueCMD());
}

// This command is used to change the background color of the screen.  Any
// objects on the screen and the floating character are not affected by this
// command, they are drawn around.
char SetBackground(char color)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    ezVGABuffer[0] = V_SETBCKGND;
    ezVGABuffer[1] = color;

    ezVGABufferSize = 2;

    // Issue command
    return(ezVGAIssueCMD());
}

// This command is used to clear the entire screen.  The screen is cleared to
// the color specified and this becomes the new background color.  The floating
// character is automatically turned off when this command is executed.  If it
// is needed it must be turned back on after the command is completed.
char ClearScreen(char color)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    ezVGABuffer[0] = V_SCRNCLEAR;
    ezVGABuffer[1] = color;

    ezVGABufferSize = 2;

    // Issue command
    return(ezVGAIssueCMD());
}

// Create a Character.  This command allows you to define your own custom
// characters or change any of the ASCII characters that came with the ezVGA
// Serial Module.  The character memory map is setup to allow 256 total
// characters.  Characters index 0 through 31 are undefined, 32 through 127
// are pre-defined from the factory to a standard ASCII character, and 128 to
// 255 are undefined.  To create a custom character you first draw the image
// that you want in an 8 by 8 matrix (as shown above).  Once you have what you
// want you have to determine a value for each of the eight rows.  At the
// bottom of each column there is a number starting at 128 under the left-most
// column.  These numbers represent the value of each pixel that is turned on
// in that row.  To get the byte value for a row you add up all the values for
// each pixel that is turned on.
char CreateCharacter(char idx, char row0, char row1, char row2, char row3,
                     char row4, char row5, char row6, char row7)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    ezVGABuffer[0] = V_CREATCHAR;
    ezVGABuffer[1] = idx;
    ezVGABuffer[2] = row0;
    ezVGABuffer[3] = row1;
    ezVGABuffer[4] = row2;
    ezVGABuffer[5] = row3;
    ezVGABuffer[6] = row4;
    ezVGABuffer[7] = row5;
    ezVGABuffer[8] = row6;
    ezVGABuffer[9] = row7;

    ezVGABufferSize = 10;

    // Issue command
    return(ezVGAIssueCMD());
}

// This command enables you to draw a character that will float over the
// other characters on the screen without erasing or disturbing them in any way.
char FloatingChar(char idx, char mode, char color, int x, char y)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    // Assemble the x values into high and low bits
    char xHighByte = x >> 8;
    char xLowByte = (char)x;

    //char cMode;
    //if(mode) {
    //    cMode = 1;
    //} else {
    //    cMode = 0;
    //}

    ezVGABuffer[0] = V_FLOATCHAR;
    ezVGABuffer[1] = idx;
    ezVGABuffer[2] = mode;
    ezVGABuffer[3] = color;
    ezVGABuffer[4] = xHighByte;
    ezVGABuffer[5] = xLowByte;
    ezVGABuffer[6] = y;

    ezVGABufferSize = 7;

    // Issue command
    return(ezVGAIssueCMD());
}

// This command enables you to draw a line on the screen.  Any type of straight
// and angled line can be drawn.  Each line is defined by two points.  The
// first is the starting point which is referred to as x1,y1.  The second is
// the ending point which is x2,y2.  A line can be erased from the screen by
// re-drawing it in the current background color.
char DrawLine(char color, int x1, char y1, int x2, char y2)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    // Assemble the x values into high and low bits
    char x1HighByte = x1 >> 8;
    char x1LowByte = (char)x1;

    char x2HighByte = x2 >> 8;
    char x2LowByte = (char)x2;

    ezVGABuffer[0] = V_DRAWALINE;
    ezVGABuffer[1] = color;
    ezVGABuffer[2] = x1HighByte;
    ezVGABuffer[3] = x1LowByte;
    ezVGABuffer[4] = y1;
    ezVGABuffer[5] = x2HighByte;
    ezVGABuffer[6] = x2LowByte;
    ezVGABuffer[7] = y2;

    ezVGABufferSize = 8;

    // Issue command
    return(ezVGAIssueCMD());
}

// This command places a character on the screen.  The character can be placed
// in a variety of different ways depending on your needs.  The first option is
// characters can be placed as an opaque or as a transparent character.  When a
// character is placed as an opaque all the pixels are drawn.  The pixels that
// are off are drawn in the current background color and the pixels that are on
// are drawn in the character color.  When drawn this way a character will
// completely overwrite whatever it is placed over.  When a character is placed
// as a transparent then the pixels that are off are not drawn.  Only the
// pixels that are on are drawn in the character color.  This allows you to
// place a character over a background and the background will be visible
// through the off pixels of the character.  Secondly, characters can be placed
// in normal or reverse mode.
char PlaceCharacter(char idx, char mode, char size, char color, int x, char y)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    // Assemble the x values into high and low bits
    char xHighByte = x >> 8;
    char xLowByte = (char)x;

    ezVGABuffer[0] = V_PLACECHAR;
    ezVGABuffer[1] = idx;
    ezVGABuffer[2] = mode;
    ezVGABuffer[3] = size;
    ezVGABuffer[4] = color;
    ezVGABuffer[5] = xHighByte;
    ezVGABuffer[6] = xLowByte;
    ezVGABuffer[7] = y;

    ezVGABufferSize = 8;

    // Issue command
    return(ezVGAIssueCMD());
}

// This command is used to read a pixel from the screen.  When executed the
// color of the pixel at the x,y position is read from the video memory and
// returned as a single byte.
char ReadPixel(int x, char y)
{

    // Check if buffer is empty
    //if(ezVGABufferSize != 0)
    //{
        // Wait until buffer is empty
        //while(ezVGABufferSize > 0);
    //}

    // Assemble the x values into high and low bits
    char xHighByte = x >> 8;
    char xLowByte = (char)x;

    ezVGABuffer[0] = V_READPIXEL;
    ezVGABuffer[1] = xHighByte;
    ezVGABuffer[2] = xLowByte;
    ezVGABuffer[3] = y;

    ezVGABufferSize = 4;

    // Issue command
    return(ezVGAIssueCMD());
}

//This command is used to write a pixel to the screen.
char WritePixel(char color, int x, char y)
{

    // Check if buffer is empty
//    if(ezVGABufferSize != 0)
//    {
//        // Wait until buffer is empty
//        while(ezVGABufferSize > 0);
//    }

    // Assemble the x values into high and low bits
    char xHighByte = x >> 8;
    char xLowByte = (char)x;

    ezVGABuffer[0] = V_WRITPIXEL;
    ezVGABuffer[1] = color;
    ezVGABuffer[2] = xHighByte;
    ezVGABuffer[3] = xLowByte;
    ezVGABuffer[4] = y;

    ezVGABufferSize = 5;

    // Issue command
    return(ezVGAIssueCMD());
}

// Mix colors
char MixColor(char Blue, char Green, char Red)
{

    // Limit the color intensity to max
    if(Blue > V_MAXINTENSITY)
    {
        Blue = V_MAXINTENSITY;
    }

    if(Green > V_MAXINTENSITY)
    {
        Green = V_MAXINTENSITY;
    }

    if(Red > V_MAXINTENSITY)
    {
        Red = V_MAXINTENSITY;
    }

    return(V_BLUE*Blue + V_GREEN*Green + V_RED*Red);

}
