/***************************************************************
* This would be the main user Skeleton code for any user game.
* Notice that we give the user access to the Init function
* in case they wish to initialize anything and the RTI ISR
* in case they need something done in the ISR. Thus, don't
* put these functions into other "library" files.
****************************************************************/
#include "hcs12.h"
#include "LotusLib.h"
#include "main.h"

// Define any global variables not belonging to the library here.
unsigned char RTIFlag = 0;

// Blue variables
unsigned int m_Blue_PosX;
unsigned char m_Blue_PosY;
char m_Blue_Dir;
char m_Blue_Color;
char m_Blue_Life;

// Red variables
unsigned int m_Red_PosX;
unsigned char m_Red_PosY;
char m_Red_Dir;
char m_Red_Color;
char m_Red_Life;

//char m_Arena[128][128];

// Initalizing all of the registers on the dragon board for the lotus system
void Init()
{
    // Initialze the Controller and SPI CS signal.
    DDRB = 0x13;      // Make bits 0,1, and 4 output.
    PORTB = 0x10;     // Make clk low and latch low.
                      // Make SPI CS signal high.

    // Initialize the SPI for the EEPROM.
    SPI0BR = 0x03;    // Set the SPI clk rate to 1.5MHz.
    SPI0CR1 = 0x50;   // Turn on SPI and make us a Master.

    // Initialize the PWM.
    PWMPOL = 0;       // Active low polarity. This means the greater the duty
                      // cycle then brighter the LED.

    PWMPRCLK = 0x55;  // This makes clock to the PWM divided by 32 which is
                      // good because the LED can't switch that fast.

    PWME = 0xFF;       // Enable all PWM channels.

    // Initialize the RTI.
    CRGINT |= RTIE;   // Enable the interrupt for the RTI.
    RTICTL = 0x78;    // Set the RTI frequency to 30.5Hz.

    // Initialize SCIs for SoundGin and VGA.
    // Initialize SCI0
    SCI0BDH = 0;      // br=MCLK/(16*baudRate)
    SCI0BDL = 0x9C;   // 9600 BAUD.
    SCI0CR1 = 0;
    /* bit value meaning
    7   0    LOOPS, no looping, normal
    6   0    WOMS, normal high/low outputs
    5   0    RSRC, not appliable with LOOPS=0
    4   0    M, 1 start, 8 data, 1 stop
    3   0    WAKE, wake by idle (not applicable)
    2   0    ILT, short idle time (not applicable)
    1   0    PE, no parity
    0   0    PT, parity type (not applicable with PE=0) */
    SCI0CR2 = 0x0C;
    /* bit value meaning
    7   0    TIE, no transmit interrupts on TDRE
    6   0    TCIE, no transmit interrupts on TC
    5   0    RIE, no receive interrupts on RDRF
    4   0    ILIE, no interrupts on idle
    3   1    TE, enable transmitter
    2   1    RE, enable receiver
    1   0    RWU, no receiver wakeup
    0   0    SBK, no send break */

    // Initialize SCI1
    SCI1BDH = 0;      // br=MCLK/(16*baudRate)
//    SCI1BDL = 13;     // 115200 BAUD.
    SCI1BDL = 14;     // 115200 BAUD.
    SCI1CR1 = 0;
    /* bit value meaning
    7   0    LOOPS, no looping, normal
    6   0    WOMS, normal high/low outputs
    5   0    RSRC, not appliable with LOOPS=0
    4   0    M, 1 start, 8 data, 1 stop
    3   0    WAKE, wake by idle (not applicable)
    2   0    ILT, short idle time (not applicable)
    1   0    PE, no parity
    0   0    PT, parity type (not applicable with PE=0) */
    SCI1CR2 = 0x0C;
    /* bit value meaning
    7   0    TIE, no transmit interrupts on TDRE
    6   0    TCIE, no transmit interrupts on TC
    5   0    RIE, no receive interrupts on RDRF
    4   0    ILIE, no interrupts on idle
    3   1    TE, enable transmitter
    2   1    RE, enable receiver
    1   0    RWU, no receiver wakeup
    0   0    SBK, no send break */

    // Send the U character to ezVGA so that it know the BAUD rate.
    while((SCI1SR1 & TDRE) == 0);
    SCI1DRL = 'U';

    // Wait for ACK/NACK. This guarantees that we wont send
    // ezVGA another byte until he is ready.
    while((SCI1SR1 & RDRF) == 0);
    if(SCI1DRL == 0x15)       // Check to see if it is a NACK.
    {
        int index;
        SetPWMDuty(1, 0xff);  // Turn on both red LEDs to indicate a problem.
        SetPWMDuty(5, 0xff);

        // Wait for about 1s for user to see there is a problem.
        for(index=0;index<30;index++)
        {
            while(RTIFlag == 0);    // Wait for RTIFlag to be set.
            RTIFlag = 0;            // Clear it.
        }

        SetPWMDuty(1, 0x00);  // Turn off both red LEDs.
        SetPWMDuty(5, 0x00);
    }

    // Initialize the counter for the random number generator
    TSCR1 = 0x80;

    /* Any user initializations here. */
    Seed();
}

// Interrupt Handler routine for RTI.
void __attribute__((interrupt)) RTIHandle()
{
    CRGFLG |= RTIF;   // Clear the interrupt flag.
    RTIFlag = 1;      // Set the global variable flag.

    /* Any user code needed withing the ISR here. */
}

// Main Function
int main()
{
    // Declare splash screen text
    char BWString[9] = "BLUE WINS";
    char RWString[8] = "RED WINS";
    char TieString[9] = "GAME DRAW";
    char TitleString[20] = "Tron Bike: Knockoff!";
    char PSString[18] = "Push Start to Play";

    // Setup the colors
    char Status = WFSTART;

    // General purpose for loop index counter
    int index;

    // First call the Init function to intialize all the IO.
    Init();

    // Write to the EEPROM max Life value
    WriteEEPROM(10, MAXLIVES);        // Blue
    WriteEEPROM(20, MAXLIVES);        // Red
    
    // Main loop.
    while(1)
    {
        // Wait for the RTI flag to be set.
        if(RTIFlag)
        {
            RTIFlag = 0;    // First thing we do is clear the flag.

            /* User code goes here. */

            // Get the controller input
            GetContData();

            switch(Status)
            {

              // Waiting for start state, user must hit start to start the game
              case WFSTART:
                if((m_Blue_Life == 0) || (m_Red_Life == 0))
                {
                    WriteEEPROM(10, MAXLIVES);        // Blue
                    WriteEEPROM(20, MAXLIVES);        // Red

                    // Write the SPLASH Waiting for Start screen
                    ClearScreen(BLACK);
                    for(index=0;index<20;index++)
                        PlaceCharacter(TitleString[index], 0, 2, GREEN, 15+(15*index), 80);

                    for(index=0;index<18;index++)
                        PlaceCharacter(PSString[index], 0, 1, GREEN, 70+(10*index), 140);

                    // Get the lives remaining
                    m_Blue_Life = ReadEEPROM(10);
                    m_Red_Life = ReadEEPROM(20);
                    
                }


                Status = StartGame();
                break;
                
              // Initialize arena state, user has hit start and the game begins
              case STARTED:
                InitArena();

                // Get the lives remaining
                m_Blue_Life = ReadEEPROM(10);
                m_Red_Life = ReadEEPROM(20);
                
                for(index=0;index<10;index++) {
                    unsigned int BlockX = Rand() % 319;
                    unsigned char BlockY = Rand() % 239;
                
                    if(((BlockX > 85)  || (BlockX < 79))  &&
                       ((BlockX > 245) || (BlockX < 239)) &&
                       ((BlockY > 65)  || (BlockY < 59))  &&
                       ((BlockY > 185) || (BlockY < 179)))
                    {
                        ClearArea(GREEN, BlockX, BlockY, BlockX-1, BlockY-1);
                    }
                }
                
                // Light the LED according the the lives remaining (out of 4)
                for(index=0;index<8;index++)
                {
                    if(index <= m_Blue_Life-1)
                        SetPWMDuty(index, 0xFF);
                    else if(index >= (8 - m_Red_Life))
                        SetPWMDuty(index, 0xFF);
                    else
                        SetPWMDuty(index, 0x00);
                }

                // Then configure the sound envelope to our pleasing
                WriteOneByteWMask(30, 0xf0, 0xe0);
                WriteOneByteWMask(30, 15, 244);

                Status = RUNNING;
                break;

              // Running State, the game is in progress
              case RUNNING:

                // Write to screen the Blue Player
                ClearArea(BLUE, m_Blue_PosX, m_Blue_PosY,
                                m_Blue_PosX-1, m_Blue_PosY-1);

                // Write to screen the Red Player
                ClearArea(RED, m_Red_PosX, m_Red_PosY,
                               m_Red_PosX-1, m_Red_PosY-1);

                // Parse the controller command
                ParseController();

                // Move the players and get the new status
                Status = MovePlayers();

                // Check the case
                switch(Status)
                {
                  case BOTHDIE:
                    // Clear screen and write the words Draw
                    ClearScreen(BLACK);
                    for(index=0;index<9;index++)
                        PlaceCharacter(TieString[index], 0, 2, 0x3f, 20+(32*index), 120);

                    // Play lose tone
                    //LoadPlayNote(EO_A1, E3)

                    // Write to EEPROM both new life counts
                    WriteEEPROM(10, m_Blue_Life-1);        // Blue
                    WriteEEPROM(20, m_Red_Life-1);          // Red

                    break;
                  case REDDIED:
                    // Clear screen and write the words Wins in blue
                    ClearScreen(BLACK);
                    for(index=0;index<9;index++)
                        PlaceCharacter(BWString[index], 0, 2, BLUE, 20+(32*index), 120);

                    // Play winning tone
                    //LoadPlayNote(EO_A1, C3)

                    // Write to EEPROM both new life counts
                    WriteEEPROM(10, m_Blue_Life);        // Blue
                    WriteEEPROM(20, m_Red_Life-1);          // Red
                    break;
                  case BLUEDIE:
                    // Clear screen and write the words Wins in red
                    ClearScreen(BLACK);
                    for(index=0;index<8;index++)
                        PlaceCharacter(RWString[index], 0, 2, RED, 36+(32*index), 120);

                    // Play winning tone
                    //LoadPlayNote(EO_A1, C3)

                    // Write to EEPROM new blue life count
                    WriteEEPROM(10, m_Blue_Life-1);        // Blue
                    WriteEEPROM(20, m_Red_Life);          // Red
                    break;
                }  // end switch

                // Current game ends, switch back to waiting for start state
                if(Status != RUNNING) {
                    Status = WFSTART;

                    // Stop the sound
                    ReleaseOscillator(EO_A1);
                }
                break;
            }  // end switch status
        }  // end if RTIFlag
    }  // end while

    return 0;
}

char StartGame()
{
    char Start = WFSTART;

    // Start command
    if((ContP1 & 0x10) == 0)
        Start = STARTED;

    // Start command
    if((ContP2 & 0x10) == 0)
        Start = STARTED;

    return Start;
}

void ParseController()
{
    // Right command
    if((ContP1 & 0x01) == 0)
    {
        if(m_Blue_Dir != ETBD_Left)
            m_Blue_Dir = ETBD_Right;
    }

    // Left command
    if((ContP1 & 0x02) == 0)
    {
        if(m_Blue_Dir != ETBD_Right)
            m_Blue_Dir = ETBD_Left;
    }

    // Down command
    if((ContP1 & 0x04) == 0)
    {
        if(m_Blue_Dir != ETBD_Up)
            m_Blue_Dir = ETBD_Down;
    }

    // Up command
    if((ContP1 & 0x08) == 0)
    {
        if(m_Blue_Dir != ETBD_Down)
            m_Blue_Dir = ETBD_Up;
     }

    // Right command
    if((ContP2 & 0x01) == 0)
    {
        if(m_Red_Dir != ETBD_Left)
            m_Red_Dir = ETBD_Right;
    }

    // Left command
    if((ContP2 & 0x02) == 0)
    {
        if(m_Red_Dir != ETBD_Right)
            m_Red_Dir = ETBD_Left;
    }

    // Down command
    if((ContP2 & 0x04) == 0)
    {
        if(m_Red_Dir != ETBD_Up)
            m_Red_Dir = ETBD_Down;
    }

    // Up command
    if((ContP2 & 0x08) == 0)
    {
        if(m_Red_Dir != ETBD_Down)
            m_Red_Dir = ETBD_Up;
    }
}

// Initialize the arena and the players
void InitArena()
{

    // Clear all of the playing field on screen
    ClearScreen(BLACK);

    // Draw Boarder
    DrawLine(GREEN, 0, 0, 319, 0);
    DrawLine(GREEN, 0, 0, 0, 239);
    DrawLine(GREEN, 0, 239, 319, 239);
    DrawLine(GREEN, 319, 0, 319, 239);

    // Set up the blue player
    m_Blue_PosX = 80;
    m_Blue_PosY = 60;
    m_Blue_Color = ETBC_Blue;
    m_Blue_Dir = ETBD_Right;

    // Set up the red player
    m_Red_PosX = 240;
    m_Red_PosY = 180;
    m_Red_Color = ETBC_Red;
    m_Red_Dir = ETBD_Left;

    // Play start tone
    LoadPlayNote(EO_A1, C3);
}

// Move the players and determine if they have not run into the wall or
// the other player's trail.
char MovePlayers() {

    char Result = RUNNING;

    LoadPlayNote(EO_A1, F3);

    // Move the blue player
    switch(m_Blue_Dir)
    {
      case ETBD_Up:
        m_Blue_PosY = m_Blue_PosY - 2;
        break;
      case ETBD_Right:
        m_Blue_PosX = m_Blue_PosX + 2;
        break;
      case ETBD_Down:
        m_Blue_PosY = m_Blue_PosY + 2;
        break;
      case ETBD_Left:
        m_Blue_PosX = m_Blue_PosX - 2;
        break;
    }

    // Check if blue player is still alive
    if((m_Blue_PosX > 1) && (m_Blue_PosX < ARENAWIDTH-1) &&
       (m_Blue_PosY > 1) && (m_Blue_PosY < ARENAHEIGHT-1))
    {
        // Check the lower right of the current move
        ReadPixel(m_Blue_PosX, m_Blue_PosY);
        if(ezVGAReturn != BLACK)
        {
            Result = BLUEDIE;

            // Play wall tone
            LoadPlayNote(EO_A1, D3);

        }
        // Check the upper left of the current move
        ReadPixel(m_Blue_PosX, m_Blue_PosY);
        if(ezVGAReturn != BLACK)
        {
            Result = BLUEDIE;

            // Play wall tone
            LoadPlayNote(EO_A1, D3);

        }
    }
    else
    {
        Result = BLUEDIE;

        // Play crash tone
        LoadPlayNote(EO_A1, E3);
    }

    // Move the red player
    switch(m_Red_Dir)
    {
      case ETBD_Up:
        m_Red_PosY = m_Red_PosY - 2;
        break;
      case ETBD_Right:
        m_Red_PosX = m_Red_PosX + 2;
        break;
      case ETBD_Down:
        m_Red_PosY = m_Red_PosY + 2;
        break;
      case ETBD_Left:
        m_Red_PosX = m_Red_PosX - 2;
        break;
    }

    // Check if red player is still alive
    if((m_Red_PosX > 1) && (m_Red_PosX < ARENAWIDTH-1) &&
       (m_Red_PosY > 1) && (m_Red_PosY < ARENAHEIGHT-1))
    {
        // Check lower right of the current move
        ReadPixel(m_Red_PosX, m_Red_PosY);
        if(ezVGAReturn != BLACK)
        {
            if(Result == BLUEDIE)
                Result = BOTHDIE;
            else
                Result = REDDIED;

            // Play wall tone
            LoadPlayNote(EO_A1, D3);

        }
        // Check the upper left of the current move
        ReadPixel(m_Red_PosX-1, m_Red_PosY-1);
        if(ezVGAReturn != BLACK)
        {
            if(Result == BLUEDIE)
                Result = BOTHDIE;
            else
                Result = REDDIED;

            // Play wall tone
            LoadPlayNote(EO_A1, D3);

        }
    }
    else
    {
        if(Result == BLUEDIE)
            Result = BOTHDIE;
        else
            Result = REDDIED;

        // Play crash tone
        LoadPlayNote(EO_A1, E3);
    }

    // Check case where the two occupies the same space
    if((m_Red_PosX == m_Blue_PosX) && (m_Red_PosY == m_Blue_PosY)) {
        Result = BOTHDIE;
    }

    return Result;
}
